﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_PANE_H_
#define NW_LYT_PANE_H_

#include <cstddef>
#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_Rect.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{
namespace internal
{

//---------------------------------------------------------------------------
//! @brief ペインの基底クラスに継承されるクラスです。
//!
//---------------------------------------------------------------------------
class PaneBase
{
    NW_DISALLOW_COPY_AND_ASSIGN(PaneBase);

public:
    PaneBase();
    virtual ~PaneBase();

public:

    //! @brief 内部用機能のため使用禁止です。
    ut::LinkListNode m_Link;
};

} // namespace nw::lyt::internal

namespace res
{

struct Pane;
class ExtUserData;
struct ExtUserDataList;

}   // namespace res

class AnimTransform;
class AnimResource;
class Material;
class DrawInfo;
class Pane;

//! @brief ペインの一覧を保持するリストの定義です。
//!
typedef ut::LinkList<Pane, offsetof(internal::PaneBase, m_Link)>   PaneList;

//---------------------------------------------------------------------------
//! @brief ペインの基底クラスです。
//!
//---------------------------------------------------------------------------
class Pane : public internal::PaneBase
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! デフォルト設定の状態で初期化します。
    //!
    Pane();

    //! @brief コンストラクタです。
    //!
    //! @details
    //! データブロックの設定を反映した状態で初期化します。
    //!
    //! @param[in] pBlock       ペインデータブロックへのポインタです。
    //! @param[in] buildArgSet  構築時の引数への参照です。
    //!
    Pane(const res::Pane* pBlock, const BuildArgSet& buildArgSet);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したペインをコピーしたペインを作成します。
    //!
    //! ペインの元々のレイアウトデータの状態ではなく、そのときの状態でコピーします
    //! のでご注意ください。つまり、アニメーションを再生した後はそのままの状態で
    //! コピーします。
    //!
    //! ただし、このメソッドではペインの親子関係はコピーしませんので、ご注意ください。
    //! つまり、ペインは親を持たず、子のリストが空の状態で作られます。
    //! ペインツリーに登録するためには、明示的にペインにAppendChildしてください。
    //!
    //! アニメーションについては、何もバインドされていない状態になります。
    //! SetMtxで設定した行列は設定解除されますので、必要であれば再度設定してください。
    //!
    //! @param[in] pane コピー元のペインです。
    //!
    explicit Pane(const Pane& pane);

    //! @brief デストラクタです。
    //!
    virtual ~Pane();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief ペインの名前を取得します。
    //!
    //! @return ペインの名前を返します。
    //!
    //! @sa SetName
    //!
    const char* GetName() const
    {
        return m_Name;
    }

    //! @brief ペインの名前を設定します。
    //!
    //! @details
    //! セットできるペイン名文字列の最大長は lyt::ResourceNameStrMax です。
    //!
    //! @param[in] name 名前です。
    //!
    //! @sa GetName
    //!
    void SetName(const char* name);

    //! @brief ユーザー情報を取得します。
    //!
    //! @return ユーザー情報を返します。
    //!
    //! @sa SetUserData
    //!
    const char* GetUserData() const
    {
        return m_UserData;
    }

    //! @brief ユーザー情報を設定します。
    //!
    //! @details
    //! セットできるユーザー情報文字列の最大長は lyt::UserDataStrMax です。
    //!
    //! @param[in] userData ユーザー情報です。
    //!
    //! @sa GetUserData
    //!
    void SetUserData(const char* userData);

    //! @brief 親のアルファ値の変化が子に影響するかどうかを判定します。
    //!
    //! @return 親のアルファ値の変化が子に影響する場合は true を返します。
    //!
    //! @sa SetInfluencedAlpha
    //!
    bool IsInfluencedAlpha() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_INFLUENCEDALPHA);
    }

    //! @brief 親のアルファ値の変化が子に影響するかどうかを設定します。
    //!
    //! @param[in] bAlpha   親のアルファ値の変化が子に影響する場合には true を指定します。
    //!
    //! @sa IsInfluencedAlpha
    //!
    void SetInfluencedAlpha(bool bAlpha)
    {
        internal::SetBit(&m_Flag, PANEFLAG_INFLUENCEDALPHA, bAlpha);
    }

    //! @brief 位置調整処理の対象となるかどうかを判定します。
    //!
    //! @return 位置調整処理の対象となる場合は true を返します。
    //!
    //! @sa SetLocationAdjust
    //!
    bool IsLocationAdjust() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_LOCATIONADJUST);
    }

    //! @brief 位置調整処理の対象となるかどうかを設定します。
    //!
    //! @param[in] bAdjust  位置調整処理の対象となる場合には true を指定します。
    //!
    //! @sa IsLocationAdjust
    //! @sa DrawInfo::SetLocationAdjust
    //!
    void SetLocationAdjust(bool bAdjust)
    {
        internal::SetBit(&m_Flag, PANEFLAG_LOCATIONADJUST, bAdjust);
        this->SetGlbMtxDirty();
    }

    //! @brief ペインの表示位置を取得します。
    //!
    //! @return ペインの表示位置を返します。
    //!
    //! @sa SetTranslate
    //!
    const math::VEC3& GetTranslate() const
    {
        return m_Translate;
    }

    //! @brief ペインの表示位置を設定します。
    //!
    //! @param[in] value    ペインの表示位置です。
    //!
    //! @sa GetTranslate
    //!
    void SetTranslate(const math::VEC3& value)
    {
        m_Translate = value;
        this->SetGlbMtxDirty();
    }

    //! @brief ペインの表示位置を設定します。
    //!
    //! @details
    //! Z 成分は 0 に設定されます。
    //!
    //! @param[in] value    ペインの表示位置です。
    //!
    //! @sa GetTranslate
    //!
    void SetTranslate(const math::VEC2& value)
    {
        SetTranslate(math::VEC3(value.x, value.y, 0.f));
    }

    //! @brief ペインの回転を取得します。
    //!
    //! @return X, Y, Z軸まわりの回転角度(Degree)を格納したベクトルを返します。
    //!
    //! @sa SetRotate
    //!
    const math::VEC3& GetRotate() const
    {
        return m_Rotate;
    }

    //! @brief ペインの回転を設定します。
    //!
    //! @param[in] value    X, Y, Z軸まわりの回転角度(Degree)を格納したベクトルです。
    //!
    //! @sa GetRotate
    //!
    void SetRotate(const math::VEC3& value)
    {
        m_Rotate = value;
        this->SetGlbMtxDirty();
    }

    //! @brief ペインの拡大率を取得します。
    //!
    //! @return ペインの拡大率を返します。
    //!
    //! @sa SetScale
    //!
    const math::VEC2& GetScale() const
    {
        return m_Scale;
    }

    //! @brief ペインの拡大率を設定します。
    //!
    //! @param[in] value    ペインの拡大率です。
    //!
    //! @sa GetScale
    //!
    void SetScale(const math::VEC2& value)
    {
        m_Scale = value;
        this->SetGlbMtxDirty();
    }

    //! @brief ペインのサイズを取得します。
    //!
    //! @return ペインのサイズを返します。
    //!
    //! @sa SetSize
    //!
    const Size& GetSize() const
    {
        return m_Size;
    }

    //! @brief ペインのサイズを設定します。
    //!
    //! @param[in] value    ペインのサイズです。
    //!
    //! @sa GetSize
    //!
    void SetSize(const Size& value)
    {
        m_Size = value;
        this->SetGlbMtxDirty();
    }

    //! @brief ペインの基準位置設定の水平位置指定を取得します。
    //!
    //! @details
    //! 水平位置指定の値については lyt::HorizontalPosition を参照してください。
    //!
    //! @return 水平位置指定の値を返します。
    //!
    //! @sa SetBasePositionH
    //! @sa GetBasePositionV
    //! @sa lyt::HorizontalPosition
    //!
    u8 GetBasePositionH() const
    {
        return static_cast<u8>(internal::GetHorizontalPosition(m_BasePosition));
    }

    //! @brief ペインの基準位置設定の水平位置指定を設定します。
    //!
    //! @param[in] val  ペインの基準位置指定です。 lyt::HorizontalPosition を指定します。
    //!
    //! @sa GetBasePositionH
    //! @sa SetBasePositionV
    //! @sa lyt::HorizontalPosition
    //!
    void SetBasePositionH(u8 val)
    {
        internal::SetHorizontalPosition(&m_BasePosition, val);
    }

    //! @brief ペインの基準位置設定の垂直位置指定を取得します。
    //!
    //! @details
    //! 垂直位置指定の値については lyt::VerticalPosition を参照してください。
    //!
    //! @return 垂直位置指定の値を返します。
    //!
    //! @sa SetBasePositionV
    //! @sa GetBasePositionH
    //! @sa lyt::VerticalPosition
    //!
    u8 GetBasePositionV() const
    {
        return static_cast<u8>(internal::GetVerticalPosition(m_BasePosition));
    }

    //! @brief ペインの基準位置設定の垂直位置指定を設定します。
    //!
    //! @param[in] val  ペインの基準位置指定です。 lyt::VerticalPosition を指定します。
    //!
    //! @sa GetBasePositionV
    //! @sa SetBasePositionH
    //! @sa lyt::VerticalPosition
    //!
    void SetBasePositionV(u8 val)
    {
        internal::SetVerticalPosition(&m_BasePosition, val);
    }

    //! @brief 親ペイン相対での原点位置の水平位置指定を取得します。
    //!
    //! @return 水平位置指定の値を返します。
    //!
    //! @sa SetParentRelativePositionH
    //! @sa GetParentRelativePositionV
    //! @sa lyt::HorizontalPosition
    //!
    u8 GetParentRelativePositionH() const
    {
        return static_cast<u8>((m_BasePosition >> 4) & 0x3);
    }

    //! @brief 親ペイン相対での原点位置の水平位置指定を設定します。
    //!
    //! @param[in] val  ペインの基準位置指定です。 lyt::HorizontalPosition を指定します。
    //!
    //! @sa GetParentRelativePositionH
    //! @sa SetParentRelativePositionV
    //! @sa lyt::HorizontalPosition
    //!
    void SetParentRelativePositionH(u8 val)
    {
        NW_ASSERTMSG(val < HORIZONTALPOSITION_MAX, "out of bounds: val[%u] < HORIZONTALPOSITION_MAX[%u] for Pane[%s]", val, HORIZONTALPOSITION_MAX, GetName());
        m_BasePosition = (m_BasePosition & (~0x30)) | (val << 4);
    }

    //! @brief 親ペイン相対での原点位置の垂直位置指定を取得します。
    //!
    //! @return 垂直位置指定の値を返します。
    //!
    //! @sa SetParentRelativePositionV
    //! @sa GetParentRelativePositionH
    //! @sa lyt::VerticalPosition
    //!
    u8 GetParentRelativePositionV() const
    {
        return static_cast<u8>((m_BasePosition >> 6) & 0x3);
    }

    //! @brief 親ペイン相対での原点位置の垂直位置指定を設定します。
    //!
    //! @param[in] val  ペインの基準位置指定です。 lyt::VerticalPosition を指定します。
    //!
    //! @sa GetParentRelativePositionV
    //! @sa SetParentRelativePositionH
    //! @sa lyt::VerticalPosition
    //!
    void SetParentRelativePositionV(u8 val)
    {
        NW_ASSERTMSG(val < VERTICALPOSITION_MAX, "out of bounds: val[%u] < VERTICALPOSITION_MAX[%u] for Pane[%s]", val, VERTICALPOSITION_MAX, GetName());
        m_BasePosition = (m_BasePosition & (~0xc0)) | (val << 6);
    }

    //! @brief ペインの SRT 行列を取得します。
    //!
    //! @details
    //! SRT 行列は通常、 CalculateMtx() で計算されます。
    //!
    //! @return SRT行列です。
    //!
    //! @sa SetMtx
    //! @sa CalculateMtx
    //!
    const math::MTX34* GetMtx() const
    {
        return m_UserMtx;
    }

    //! @brief ペインの SRT 行列を設定します。
    //!
    //! @details
    //! SRT 行列は通常は CalculateMtx() で計算されます。
    //!
    //! この関数で行列を指定すると、設定した行列がそのまま使用されます。
    //!
    //! CalculateMtx() による計算に戻すには ResetMtx() を使用します。
    //!
    //! SetGlobalMtx() を使用した場合には、そちらの設定が優先されます。
    //!
    //! Paneは与えた行列のポインタを保持しますので、ResetMtxで解除するまで
    //! 与えた行列は静的に保持するようにしてください。
    //!
    //! @param[in] mtx  SRT 行列のポインタです。
    //!
    //! @sa GetMtx
    //! @sa CalculateMtx
    //! @sa ResetMtx
    //! @sa SetGlobalMtx
    //!
    void SetMtx(const math::MTX34* mtx)
    {
        m_UserMtx = mtx;
        m_Flag = internal::SetBit(m_Flag, PANEFLAG_USERMTX, true);
    }

    //! @brief ペインの SRT 行列がユーザによって設定されているか調べます。
    //!
    //! @return
    //! SRT 行列がユーザによって設定されている場合は true を返します。
    //!
    //! @sa SetMtx
    //!
    bool IsUserMtx() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_USERMTX);
    }

    //! @brief SRT 行列の設定を解除します。
    //!
    //! @details
    //! SRT 行列は CalculateMtx() で計算されます。
    //!
    //! @sa SetMtx
    //!
    void ResetMtx()
    {
        m_Flag = internal::SetBit(m_Flag, PANEFLAG_USERMTX, false);
    }

    //! @brief 親子での計算後の SRT 行列を取得します。
    //!
    //! @details
    //! SRT 行列は通常、 CalculateMtx() で計算されます。
    //!
    //! なお、CalculateMtxで計算した場合、この行列はペインのモデル行列にDrawInfoに
    //! SetViewMtxで設定したビュー行列がかかった状態（モデルビュー行列）になります
    //! ので、ご注意ください。
    //!
    //! @return 親子での計算後の SRT 行列を返します。
    //!
    //! @sa SetGlobalMtx
    //! @sa CalculateMtx
    //!
    const math::MTX34& GetGlobalMtx() const
    {
        return m_GlbMtx;
    }

    //! @brief グローバル行列を設定します。
    //!
    //! @details
    //! グローバル行列はペインのローカル座標系からビュー座標系への
    //! 変換行列です。
    //!
    //! グローバル行列は通常は CalculateMtx() で計算されます。
    //!
    //! この関数で行列を指定すると、設定した行列がそのまま使用されます。
    //!
    //! CalculateMtx() による計算に戻すには ResetGlobalMtx() を
    //! 使用します。
    //!
    //! @param[in] mtx  SRT 行列です。
    //!
    //! @sa GetGlobalMtx
    //! @sa ResetGlobalMtx
    //! @sa CalculateMtx
    //!
    //! @date 2010/04/23 設定の状態を記憶するようにしました。
    void SetGlobalMtx(const math::MTX34& mtx)
    {
        m_GlbMtx = mtx;
        m_Flag = internal::SetBit(m_Flag, PANEFLAG_USERGLOBALMTX, true);
    }

    //! @brief グローバル行列がユーザによって設定されているか調べます。
    //!
    //! @return
    //! グローバル行列がユーザによって設定されている場合は true を返します。
    //!
    //! @sa SetGlobalMtx
    //!
    bool IsUserGlobalMtx() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_USERGLOBALMTX);
    }

    //! @brief グローバル行列の設定を解除します。
    //!
    //! @details
    //! グローバル行列は CalculateMtx() で計算されます。
    //!
    //! @sa SetGlobalMtx
    //!
    void ResetGlobalMtx()
    {
        m_Flag = internal::SetBit(m_Flag, PANEFLAG_USERGLOBALMTX, false);
    }

    //! @brief ペインのローカル座標系での矩形を取得します。
    //!
    //! @return 矩形を返します。
    //!
    const ut::Rect GetPaneRect() const;

    //! @brief 頂点カラーを取得します。
    //!
    //! @details
    //! 派生クラスで実装します。
    //!
    //! このクラスでは idx の指定にかかわらず ut::Color8::WHITE を返します。
    //!
    //! @param[in] idx  インデックスです。 lyt::VertexColor を指定します。
    //!
    //! @return ペインの四隅の頂点カラーを返します。
    //!
    //! @sa SetVtxColor
    //! @sa GetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual const ut::Color8 GetVtxColor(u32 idx) const;

    //! @brief 頂点カラーを設定します。
    //!
    //! @details
    //! idx の指定には lyt::VERTEXCOLOR_* を使用してください。
    //!
    //! 派生クラスでオーバーライドされます。基底クラスではなにもしません。
    //!
    //! @param[in] idx      インデックスです。 lyt::VertexColor を指定します。
    //! @param[in] value    頂点カラーです。
    //!
    //! @sa GetVtxColor
    //! @sa SetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual void SetVtxColor(u32 idx, ut::Color8 value);

    //! @brief ペインのアルファ値を取得します。
    //!
    //! @return ペインのアルファ値を返します。
    //!
    //! @sa SetAlpha
    //! @sa GetGlobalAlpha
    //!
    u8 GetAlpha() const
    {
        return m_Alpha;
    }

    //! @brief ペインのアルファ値を設定します。
    //!
    //! @param[in] alpha    アルファ値です。
    //!
    //! @sa GetAlpha
    //! @sa SetGlobalAlpha
    //!
    void SetAlpha(u8 alpha)
    {
        m_Alpha = alpha;
    }

    //! @brief ペインのグローバルアルファ値を取得します。
    //!
    //! @details
    //! グローバルアルファ値は親ペインのアルファ値の影響を受けた
    //! 状態でのペインのアルファ値です。
    //!
    //! @return ペインのグローバルアルファ値を返します。
    //!
    //! @sa SetAlpha
    //! @sa GetAlpha
    //!
    u8 GetGlobalAlpha() const
    {
        return m_GlbAlpha;
    }

    //! @brief ペインのグローバルアルファ値を設定します。
    //!
    //! @details
    //! グローバルアルファ値は親ペインのアルファ値の影響を受けた状態でのペインのアルファ値です。
    //!
    //! グローバルアルファ値は通常、 CalculateMtx() で計算されます。
    //!
    //! @param[in] alpha    グローバルアルファ値です。
    //!
    //! @sa GetGlobalAlpha
    //! @sa SetAlpha
    //! @sa CalculateMtx
    //!
    void SetGlobalAlpha(u8 alpha)
    {
        m_GlbAlpha = alpha;
    }

    //! @brief インデックスを指定してペインの SRT 要素を取得します。
    //!
    //! @details
    //! idx で取得したいペインの表示位置、回転、拡大率、サイズの要素を指定します。
    //!
    //! @param[in] idx  インデックスです。 lyt::AnimTargetPane を指定します。
    //!
    //! @return 指定された要素の値を返します。
    //!
    //! @sa SetSRTElement
    //! @sa GetTranslate
    //! @sa GetRotate
    //! @sa GetScale
    //! @sa GetSize
    //! @sa lyt::AnimTargetPane
    //!
    f32 GetSRTElement(u32 idx) const
    {
        NW_ASSERTMSG(idx < ANIMTARGET_PANE_MAX, "out of bounds: idx[%u] < ANIMTARGET_PANE_MAX[%u] for Pane[%s]", idx, ANIMTARGET_PANE_MAX, GetName());

        const f32* srtAry = &m_Translate.x;
        return srtAry[idx];
    }

    //! @brief インデックスを指定してペインの SRT 要素を設定します。
    //!
    //! @details
    //! ペインの表示位置、回転、拡大率、サイズの要素を設定します。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPane() を指定します。
    //! @param[in] value    要素の値です。
    //!
    //! @sa GetSRTElement
    //! @sa SetTranslate
    //! @sa SetRotate
    //! @sa SetScale
    //! @sa SetSize
    //! @sa lyt::AnimTargetPane
    //!
    void SetSRTElement(u32 idx, f32 value)
    {
        NW_ASSERTMSG(idx < ANIMTARGET_PANE_MAX, "out of bounds: idx[%u] < ANIMTARGET_PANE_MAX[%u] for Pane[%s]", idx, ANIMTARGET_PANE_MAX, GetName());

        f32* srtAry = &m_Translate.x;
        srtAry[idx] = value;
        this->SetGlbMtxDirty();
    }

    //! @brief ペインのアルファ値または頂点カラーの各カラー成分を取得します。
    //!
    //! @details
    //! ペインのアルファ値または四隅の頂点カラーの各成分を返します。
    //!
    //! このクラスでは idx の指定に lyt::ANIMTARGET_PANE_ALPHA を指定した場合は
    //! ペインのアルファ値を返しますが、その他の指定では必ず 255 を返します。
    //!
    //! 他のカラー成分を持つ派生クラスでは本関数はオーバーライドされます。
    //!
    //! 一般的な利用用途では、GetVtxColor や GetAlpha の利用をお勧めします。
    //!
    //! @param[in] idx  インデックスです。 lyt::AnimTargetPaneColor の値を指定します。
    //!
    //! @return ペインのアルファ値または四隅の頂点カラーの各成分を返します。
    //!
    //! @sa SetColorElement
    //! @sa GetVtxColorElement
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual u8 GetColorElement(u32 idx) const;

    //! @brief ペインのアルファ値または頂点カラーの各カラー成分を設定します。
    //!
    //! @details
    //! 導出クラスでオーバライドされます。
    //!
    //! このクラスではペインのアルファ値のみ設定することができます。
    //!
    //! 一般的な利用用途では、SetVtxColor や SetAlpha の利用をお勧めします。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //! @param[in] value    アルファ値またはカラーの成分値です。
    //!
    //! @sa GetColorElement
    //! @sa SetVtxColorElement
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual void SetColorElement(u32 idx, u8 value);

    //! @brief 頂点カラーの各カラー成分を取得します。
    //!
    //! @details
    //! 派生クラスでオーバーライドされます。
    //! このクラスでは idx の指定にかかわらず 255 を返します。
    //!
    //! idx の指定には lyt::ANIMTARGET_VERTEXCOLOR_* を使用してください。
    //!
    //! @param[in] idx  インデックスです。 lyt::AnimTargetPaneColor の値を指定します。
    //!
    //! @return ペイン四隅の頂点カラーの各成分を返します。
    //!
    //! @sa GetColorElement
    //! @sa SetVtxColorElement
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual u8 GetVtxColorElement(u32 idx) const;

    //! @brief 頂点カラーの各カラー成分を設定します。
    //!
    //! @details
    //! 派生クラスでオーバーライドされます。
    //! このクラスでは何もしません。
    //!
    //! idx の指定には lyt::ANIMTARGET_VERTEXCOLOR_* を使用してください。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPaneColor の値を指定します。
    //! @param[in] value    カラーの成分値です。
    //!
    //! @sa GetVtxColorElement
    //! @sa SetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual void SetVtxColorElement(u32 idx, u8 value);

    //! @brief ペインのマテリアルを取得します。
    //!
    //! @details
    //! 複数のマテリアルを持つペインは最初のマテリアル (インデックスが0) を返します。
    //!
    //! マテリアルを持たない場合はNULLを返します。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    Material* GetMaterial() const;

    //! @brief ペインが持つマテリアルの数を取得します。
    //!
    //! @return マテリアルの数を返します。
    //!
    //! @sa GetMaterial
    //!
    virtual u8 GetMaterialNum() const;

    //! @brief ペインのマテリアルを取得します。
    //!
    //! @details
    //! インデックスに対応するマテリアルが返されます。
    //!
    //! インデックスの意味は導出されたクラスによって異なります。
    //!
    //! インデックスは GetMaterialNum() が返す値よりも小さくなければなりません。
    //!
    //! マテリアルを持たない場合は NULL を返します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa GetMaterialNum
    //!
    virtual Material* GetMaterial(u32 idx) const;

    //! @brief 拡張ユーザデータの個数を取得します。
    //!
    //! @return 拡張ユーザデータの個数を返します。
    //!
    //! @sa GetExtUserDataArray
    //!
    u16 GetExtUserDataNum() const;

    //! @brief 拡張ユーザデータ配列を取得します。
    //!
    //! @return 拡張ユーザデータ配列の先頭要素のポインタを返します。
    //!
    //! @sa GetExtUserDataNum
    //!
    const res::ExtUserData* GetExtUserDataArray() const;

    //! @brief 拡張ユーザデータを名前をもとに検索します。
    //!
    //! @param[in] name 検索する拡張ユーザデータ名です。
    //!
    //! @return 見つかった場合は拡張ユーザデータへのポインタ、
    //! 見つからなかった場合は NULL を返します。
    //!
    const res::ExtUserData* FindExtUserDataByName(const char* name) const;

    //@}

    //----------------------------------------
    //! @name 親子関係
    //@{

    //! @brief 親ペインを取得します。
    //!
    //! @return
    //! 親ペインへのポインタを返します。親ペインが存在しない場合は NULL を返します。
    //!
    Pane* GetParent() { return m_pParent; }

    //! @brief 親ペインを取得します。
    //!
    //! @return 親ペインです。
    //!
    const Pane* GetParent() const { return m_pParent; }

    //! @brief 子ペインの一覧を取得します。
    //!
    //! @return 子ペインの一覧を返します。
    //!
    const PaneList& GetChildList() const
    {
        return m_ChildList;
    }

    //! @brief 子ペインの一覧を取得します。
    //!
    //! @return 子ペインの一覧を返します。
    //!
    PaneList& GetChildList()
    {
        return m_ChildList;
    }

    //! @brief ペインを子供ペインリストの末尾に追加します。
    //!
    //! @details
    //! このペインやこのペインを子ペインに持つペインを追加しないように注意してください。
    //!
    //! @param[in] pChild   追加するペインです。
    //!
    //! @sa PrependChild
    //! @sa InsertChild
    //! @sa RemoveChild
    //!
    void AppendChild(Pane* pChild);

    //! @brief ペインを子供ペインリストの先頭に追加します。
    //!
    //! @details
    //! このペインやこのペインを子ペインに持つペインを追加しないように注意してください。
    //!
    //! @param[in] pChild   追加するペインです。
    //!
    //! @sa AppendChild
    //! @sa InsertChild
    //! @sa RemoveChild
    //!
    void PrependChild(Pane* pChild);

    //! @brief 子ペインを指定した位置に挿入します。
    //!
    //! @details
    //! pChild で指定したペインをこのペインの子として追加します。
    //! pNext で指定したペインの前の位置に挿入します。
    //!
    //! このペインやこのペインを子ペインに持つペインを追加しないように注意してください。
    //!
    //! @param[in] pNext    挿入する位置の1つ後となるペインです。
    //! @param[in] pChild   追加するペインです。
    //!
    //! @sa AppendChild
    //! @sa PrependChild
    //! @sa RemoveChild
    //!
    void InsertChild(
        Pane* pNext,
        Pane* pChild);

    //! @brief 子ペインを指定した位置に挿入します。
    //!
    //! @details
    //! pChild で指定したペインをこのペインの子として追加します。
    //! next で指定したイテレータの前の位置に挿入します。
    //!
    //! このペインやこのペインを子ペインに持つペインを追加しないように注意してください。
    //!
    //! @param[in] next     挿入する位置の1つ後となるイテレータです。
    //! @param[in] pChild   追加するペインです。
    //!
    //! @sa AppendChild
    //! @sa PrependChild
    //! @sa RemoveChild
    //!
    void InsertChild(
        PaneList::Iterator next,
        Pane* pChild);

    //! @brief ペインを子供ペインリストから削除します。
    //!
    //! @param[in] pChild   削除するペインです。
    //!
    //! @sa AppendChild
    //! @sa PrependChild
    //! @sa InsertChild
    //!
    void RemoveChild(Pane* pChild);

    //@}

    //----------------------------------------
    //! @name 検索
    //@{

    //! @brief 名前でペインを検索します。
    //!
    //! @details
    //! findName のペイン名を持つペインを検索します。
    //!
    //! bRecursive に true を渡した場合は子ペインと、
    //! 子ペインが子を持つ場合はさらにその子ペインからも検索します。
    //!
    //! @param[in] findName     検索するペイン名です。
    //! @param[in] bRecursive   子ペインも検索する場合は true を指定します。
    //!
    //! @return
    //! findName のペイン名を持つペインが見つかった場合はペインへのポインタを、
    //! 見つからなかった場合は NULL を返します。
    //!
    virtual Pane* FindPaneByName(
        const char* findName,
        bool bRecursive = true);

    //! @brief 名前でマテリアルを検索します。
    //!
    //! @details
    //! findName のマテリアル名を持つマテリアルを検索します。
    //!
    //! bRecursive に true を渡した場合は子ペインと、
    //! 子ペインが子を持つ場合はさらにその子ペインからも検索します。
    //!
    //! @param[in] findName     検索するマテリアル名です。
    //! @param[in] bRecursive   子ペインも検索する場合は true を指定します。
    //!
    //! @return
    //! findName のマテリアル名を持つマテリアルが見つかった場合は
    //! マテリアルへのポインタを、見つからなかった場合は NULL を返します。
    //!
    virtual Material* FindMaterialByName(
        const char* findName,
        bool bRecursive = true);

    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    //! @brief アニメーションを関連付けます。
    //!
    //! @details
    //! bRecursive に true を渡して呼び出した場合は、
    //! 関連付けるペインを子ペインからも検索します。
    //!
    //! bEnable に false を渡して呼び出した場合は、
    //! アニメーションを無効状態で関連付けます。
    //! 有効にする場合は AnimTransform::SetEnable() を使用してください。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //! @param[in] bRecursive       子ペインも検索する場合は true を指定します。
    //! @param[in] bEnable          アニメーションを無効状態で関連付ける場合は false を指定します。
    //!
    //! @sa UnbindAnimation
    //! @sa UnbindAnimationSelf
    //! @sa AnimTransform::SetEnable
    //!
    virtual void BindAnimation(
        AnimTransform* pAnimTrans,
        bool bRecursive = true,
        bool bEnable = true);

    //! @brief アニメーションの関連付けを解除します。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //! @param[in] bRecursive       子ペインも検索する場合は true を指定します。
    //!
    //! @sa BindAnimation
    //! @sa UnbindAnimationSelf
    //!
    virtual void UnbindAnimation(
        AnimTransform* pAnimTrans,
        bool bRecursive = true);

    //! @brief 自身のアニメーションの関連付けを解除します。
    //!
    //! @param[in,out] pAnimTrans   アニメーションへのポインタです。
    //!
    //! @sa BindAnimation
    //! @sa UnbindAnimation
    //!
    virtual void UnbindAnimationSelf(AnimTransform* pAnimTrans);

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ペインが表示されるかどうかを判定します。
    //!
    //! @return ペインが表示される場合は true を、表示されない場合は false を返します。
    //!
    //! @sa SetVisible
    //!
    bool IsVisible() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_VISIBLE);
    }

    //! @brief ペインの表示／非表示を設定します。
    //!
    //! @param[in] bVisible ペインを表示させる場合は true を指定します。
    //!
    //! @sa IsVisible
    //!
    void SetVisible(bool bVisible)
    {
        internal::SetBit(&m_Flag, PANEFLAG_VISIBLE, bVisible);
    }

#if ! defined(NW_RELEASE)
    //! @brief ペインがビューア上で非表示かどうかを判定します。
    //!
    //! @details
    //! この設定は、Debugビルド及びDevelopmentビルドで有効です。
    //!
    //! @return ペインがビューア上で非表示の場合は true を、表示される場合は false を返します。
    //!
    bool IsViewerInvisible() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_VIEWER_INVISIBLE);
    }
#endif

    //! @brief CalculateMtxを行う際のコンテキストです。
    //!
    struct CalculateMtxContext
    {
        font::RectDrawer* rectDrawer;   //!< フォント描画クラス
        const math::MTX34* viewMtx;     //!< ビュー行列
        math::VEC2 locationAdjustScale; //!< 位置調整のスケール
        float influenceAlpha;           //!< 子に反映させるアルファ値
        bool isLocationAdjust;          //!< 位置調整を行うか否か
        bool isInvisiblePaneCalculateMtx;   //!< 非表示のペインでCalculateMtxを行うか否か
        bool isAlphaZeroPaneCalculateMtx;   //!< アルファが0でかつ子階層にアルファの影響を与えるペインでCalculateMtxを行うか否か
        bool isInfluenceAlpha;              //!< 子にアルファ値を反映させるか否か
        const Layout* pLayout; //!< レイアウト

        //! @brief デフォルトコンストラクタです。
        //!
        CalculateMtxContext();

        //! @brief DrawInfoのパラメータを元に初期化するコンストラクタです。
        //!
        //! @param[in] drawInfo 描画の情報
        //! @param[in] layout   レイアウト
        //!
        explicit CalculateMtxContext(const DrawInfo& drawInfo, const Layout* layout = NULL);
    };

    //! @brief 描画のための行列を計算します。
    //!
    //! @param[in] context          描画情報と内部で使用する変数を含むコンテキストです。
    //! @param[in] isDirtyParentMtx 親の行列がDirtyか否かです。
    //!
    virtual void CalculateMtx(CalculateMtxContext& context, bool isDirtyParentMtx);

    //@}

    //! @brief ユーザーがペインを構築したかどうかを判定します。
    //!
    //! @return ペインをユーザーが構築した場合は true を返します。
    //!
    bool IsUserAllocated() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_USERALLOCATED);
    }

    //! @brief ユーザーが構築したペインであることを宣言します。
    //!
    //! @details
    //! ユーザが構築したと宣言されたペインはライブラリによって解放されません。
    //!
    void SetUserAllocated()
    {
        internal::SetBit(&m_Flag, PANEFLAG_USERALLOCATED, true);
    }

    //! @brief 拡張ユーザデータリストを設定します。
    //!
    //! @param[in] pBlock   拡張ユーザデータリソースへのポインタです。
    //!
    void SetExtUserDataList(const res::ExtUserDataList* pBlock)
    {
        m_pExtUserDataList = pBlock;
    }

    //! @brief ペイン自身と子ペインの描画処理を行います。
    //!
    //! @details
    //! Layout および親ペインから呼ばれます。
    //!
    //! DrawSelf() でペイン自身の描画を行い、次に子ペインの Draw() を
    //! 呼び出します。
    //!
    //! @param[in] drawInfo 描画情報です。
    //!
    virtual void Draw(DrawInfo& drawInfo);

    //! @brief 自身の描画処理を行います。
    //!
    //! @details
    //! Layout および親ペインから呼ばれます。
    //!
    //! @param[in] drawInfo 描画情報です。
    //!
    virtual void DrawSelf(DrawInfo& drawInfo);

    //! @brief グローバル行列がDirtyか否かを返します。
    //!
    //! @return グローバル行列がDirtyか否か
    //!
    NW_FORCE_INLINE bool IsGlbMtxDirty() const
    {
        return internal::TestBit(m_Flag, PANEFLAG_ISGLOBALMTXDIRTY);
    }

    //! @brief 内部フラグのポインタを取得します。
    //!
    //! @details
    //! ペインの表示/非表示等のフラグを保持している内部フラグのポインタを取得します。
    //! このフラグを直接操作するのは非常に危険であるため、通常はSetVisible等のアクセサ
    //! を使用してください。このメソッドを使用するのは、ポインタを直接操作しなければ
    //! いけないようなケースのみにしてください。
    //!
    //! @return 内部フラグのポインタ
    //!
    u8* GetFlagPtr()
    {
        return &m_Flag;
    }

    //! @brief アルファ値のポインタを取得します。
    //!
    //! @details
    //! ペインのアルファ値を保持しているメンバ変数のポインタを取得します。
    //! 通常はSetAlpha等のアクセサを使用してください。このメソッドを使用するのは、
    //! ポインタを直接操作しなければいけないようなケースのみにしてください。
    //!
    //! @return アルファ値のポインタ
    //!
    u8* GetAlphaPtr()
    {
        return &m_Alpha;
    }

protected:
    //! @brief モデルビュー行列を計算し DrawInfo に設定します。
    //!
    //! @param[in,out] drawInfo 描画情報です。
    //!
    virtual void LoadMtx(DrawInfo& drawInfo);

    //! @brief ペインの左上の座標を取得します。
    //!
    //! @return 座標を返します。
    //!
    const math::VEC2 GetVtxPos() const;

    //! @brief グローバル行列をDirtyにします。
    //!
    NW_FORCE_INLINE void SetGlbMtxDirty()
    {
        internal::SetBit(&m_Flag, PANEFLAG_ISGLOBALMTXDIRTY, true);
    }

    //! @brief グローバル行列のDirty状態を解除します。
    //!
    NW_FORCE_INLINE void CleanGlbMtx()
    {
        internal::SetBit(&m_Flag, PANEFLAG_ISGLOBALMTXDIRTY, false);
    }

    //! FindPaneByNameが再帰的に呼ばれる場合に使用します。
    //!
    //! @param[in] findName 名前です。
    //!
    //! @return ペインです。
    //!
    virtual Pane* FindPaneByNameRecursive(const char* findName);

    //! FindMaterialByNameが再帰的に呼ばれる場合に使用します。
    //!
    //! @param[in] findName 名前です。
    //!
    //! @return マテリアルです。
    //!
    virtual Material* FindMaterialByNameRecursive(const char* findName);

private:
    Pane* m_pParent;
    PaneList m_ChildList;

    math::VEC3 m_Translate;
    math::VEC3 m_Rotate;
    math::VEC2 m_Scale;
    Size m_Size;

    NW_STATIC_ASSERT((
        sizeof(math::VEC3) + // m_Translate
        sizeof(math::VEC3) + // m_Rotate
        sizeof(math::VEC2) + // m_Scale
        sizeof(Size)) / // m_Size
        sizeof(f32) == ANIMTARGET_PANE_MAX
    );

    u8 m_Flag;
    u8 m_Alpha;
    u8 m_GlbAlpha;
    u8 m_BasePosition;

    math::MTX34 m_GlbMtx; //!< ペインのSRT行列 (親子での計算後) です。

    const math::MTX34* m_UserMtx;
    const res::ExtUserDataList* m_pExtUserDataList;

    char m_Name[ResourceNameStrMax + 1];
    char m_UserData[UserDataStrMax + 1];

private:
    void Init();
    // コピー演算子を禁止します。
    const Pane& operator=( const Pane& );
    void CalcScaleFromPartsRoot(math::VEC2* scale, Pane* pane);
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_PANE_H_

