﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_CONTROL_CREATOR_H_
#define NW_LYT_CONTROL_CREATOR_H_

#include <nw/lyt/lyt_Types.h>

namespace nw
{
namespace lyt
{

class Layout;

namespace res {
class ExtUserData;
struct ExtUserDataList;
}   // namespace res

//---------------------------------------------------------------------------
//! @brief コントロールを作成する際に必要な情報を保持するクラスです。
//!
//! @details
//! ControlCreator::CreateControl() の呼び出しで引数として渡され、
//! コントロールの作成に必要な情報を提供します。
//!
//! ControlSrc は、コントロール名、機能アニメーション、機能ペインという
//! 三種類の情報を提供します。
//!
//! コントロール名は、レイアウトライブラリでレイアウトを作成する際に、
//! 部品設定サブウィンドウの「コントロールの種類」で選択したコントロール
//! の名前が入ります。この名前は、レイアウトエディタで直接入力するのでは
//! なく、コントロール定義ファイルで定義された名前の一覧から選ぶ形式に
//! なります。
//!
//! 機能アニメーションは、コントロールの機能（例えばボタンを押したときの
//! 動きなど）に対応付けられたアニメのアニメタグ名を指定するものです。
//!
//! 機能アニメーションは、例えば「一つ目はボタンが選択されたとき、二つ目は
//! ボタンが押されたとき」等とあらかじめ決めておき、コントロール定義ファイル
//! にはその順番に書いておきます。機能アニメーションのインデックスによって、
//! どの機能にアニメタグが指定されたか判別します。
//!
//! 機能ペインも機能アニメーションとほぼ同じ使い方ですが、アニメタグではなく
//! ペインを指定します。これは、例えばボタンの当たり判定の範囲を示すペインを
//! 指定する場合などに使用します。
//!
//! 機能アニメーションと機能ペインは、レイアウトエディタ側で「指定しない」こと
//! もできることにご注意ください。その場合、 GetFunctionalAnimName や
//! GetFunctionalPaneName でNULLが返されます。
//!
//! @sa ControlCreator
//---------------------------------------------------------------------------
class ControlSrc
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! メンバが未設定の状態で構築します。使用する前にInit関数でメンバの設定を
    //! 行うか、既に構築済みのインスタンスをコピーしてください。
    //!
    ControlSrc();

    //! @brief コンストラクタです。
    //!
    //! @param[in] dataPtr  コントロールデータブロックのアドレスを渡します。
    //! @param[in] pUserDataBlock  拡張ユーザーデータブロックのアドレスを渡します。
    //!
    explicit ControlSrc(const void* dataPtr, const res::ExtUserDataList* pUserDataBlock) { Init(dataPtr, pUserDataBlock); }

    //! @brief コントロールデータブロックのポインタを用いて初期化を行います。
    //!
    //! @param[in] dataPtr  コントロールデータブロックのアドレスを渡します。
    //! @param[in] pUserDataBlock  拡張ユーザーデータブロックのアドレスを渡します。
    //!
    void Init(const void* dataPtr, const res::ExtUserDataList* pUserDataBlock);

    //! @brief コントロール名を取得します。
    //!
    //! @return コントロール名
    //!
    const char* GetControlName() const { return m_ControlName; }

    //! @brief コントロールのユーザーが付けた名前を取得します。
    //!
    //! @return コントロールのユーザーが付けた名前
    //!
    const char* GetControlUserName() const { return m_ControlUserName; }

    //! @brief 機能ペインの数を取得します。
    //!
    //! @return 機能ペインの数
    //!
    u32 GetFunctionalPaneNum() const { return m_FunctionalPaneNum; }

    //! @brief 機能アニメーションの数を取得します。
    //!
    //! @return 機能アニメーションの数
    //!
    u32 GetFunctionalAnimNum() const { return m_FunctionalAnimNum; }

    //! @brief 機能ペイン名を定義ファイルに書かれた順番指定で取得します。
    //!
    //! @details
    //! 何個目の機能ペインがどのような意味を持つかは、あらかじめ決めておく
    //! 必要があります。
    //!
    //! @param[in] index    名前を取得したい機能ペインのインデックス
    //!
    //! @return 機能ペイン名。引数のインデックスで指定されていない場合はNULL。
    //!
    const char* GetFunctionalPaneName(u32 index) const;

    //! @brief 機能ペイン名をパラメータ名指定で取得します。
    //!
    //! @details
    //! 文字列比較を利用して探すぶん、GetFunctionalPaneNameより処理がかかりますが、
    //! コントロール定義ファイルを変更したときの対応が柔軟に行えるというメリットがあります。
    //!
    //! @param[in] parameterName    パラメータ名
    //!
    //! @return 機能ペイン名。引数のインデックスで指定されていない場合はNULL。
    //!
    const char* FindFunctionalPaneName(const char* parameterName) const;

    //! @brief 機能アニメーションのタグ名を定義ファイルに書かれた順番指定で取得します。
    //!
    //! @details
    //! 何個目の機能アニメーションがどのような意味を持つかは、あらかじめ決めておく
    //! 必要があります。
    //!
    //! @param[in] index    タグ名を取得したい機能アニメーションのインデックス
    //!
    //! @return 機能アニメーションタグ名。引数のインデックスで指定されていない場合はNULL。
    //!
    const char* GetFunctionalAnimName(u32 index) const;

    //! @brief 機能アニメーションのタグ名をパラメータ名指定で取得します。
    //!
    //! @details
    //! 文字列比較を利用して探すぶん、GetFunctionalAnimNameより処理がかかりますが、
    //! コントロール定義ファイルを変更したときの対応が柔軟に行えるというメリットがあります。
    //!
    //! @param[in] parameterName    タグ名を取得したい機能アニメーションのパラメータ名
    //!
    //! @return 機能アニメーションタグ名。引数のインデックスで指定されていない場合はNULL。
    //!
    const char* FindFunctionalAnimName(const char* parameterName) const;

    //! @brief 拡張ユーザデータの個数を取得します。
    //!
    //! @return 拡張ユーザデータの個数を返します。
    //!
    //! @sa GetExtUserDataArray
    //!
    u16 GetExtUserDataNum() const;

    //! @brief 拡張ユーザデータ配列を取得します。
    //!
    //! @return 拡張ユーザデータ配列の先頭要素のポインタを返します。
    //!
    //! @sa GetExtUserDataNum
    //!
    const res::ExtUserData* GetExtUserDataArray() const;

    //! @brief 拡張ユーザデータを名前をもとに検索します。
    //!
    //! @param[in] name 検索する拡張ユーザデータ名です。
    //!
    //! @return 見つかった場合は拡張ユーザデータへのポインタ、
    //! 見つからなかった場合は NULL を返します。
    //!
    const res::ExtUserData* FindExtUserDataByName(const char* name) const;

protected:
    const char* m_ControlName;
    const char* m_ControlUserName;
    u16 m_FunctionalPaneNum;
    u16 m_FunctionalAnimNum;
    const char* m_FunctionalPaneNamesPtr;
    const void* m_ControlFunctionalAnimNameOffsetsPtr;
    const void* m_ControlFunctionalPaneParameterNameOffsetsPtr;
    const void* m_ControlFunctionalAnimParameterNameOffsetsPtr;
    const res::ExtUserDataList* m_pExtUserDataList;
};

//---------------------------------------------------------------------------
//! @brief コントロールの作成を行う基底クラスです。
//!
//! @details
//! Layout::Build() メソッドの引数として渡すことで、そのレイアウト及び部品レイアウトが
//! コントロールを持っていた場合にコントロールを作成します。
//!
//! このクラスは純粋仮想クラスとなっていますので、実際に使う際は、継承して CreateControl
//! メソッドを実装する必要があります。
//!
//! 名前空間 ctrl に、レイアウトライブラリが用意しているコントロールの実装があり、これを
//! デフォルトコントロールと呼んでいます。デフォルトコントロールでは、 ControlCreator を
//! 継承した、 DefaultControlCreator クラスでコントロールの作成を行います。
//!
//! デフォルトコントロールを使用せずに、独自にコントロールを実装することも可能です。
//!
//---------------------------------------------------------------------------
class ControlCreator
{
public:
    //! @brief デストラクタです。
    //!
    virtual ~ControlCreator() {}

    //! @brief 指定されたコントロールソースに対応するコントロールを作成します。
    //!
    //! @param[in] controlSrc   作成すべきコントロールの情報。このオブジェクトのインスタンスは一時的な領域に作成しますので、
    //!                         このメソッドを抜けた後もポインタを保持しておくことはしないでください。情報を保持しておく
    //!                         必要がある場合は、コピーして保持するようにしてください。
    //! @param[in,out] layout   コントロールを作成するレイアウト
    //!
    //! @details
    //! Layout::Build() 内で、レイアウトリソース内にコントロールがあった場合に呼び出されます。
    //!
    //! 呼び出されるタイミングは、ペインツリーの構築が終わった後です。つまり、引数で渡されるレイアウト
    //! は、コントロールを除いて構築済みの状態になっています。
    //!
    //! 作成したコントロールは Layout が管理するわけではありませんので、 ControlCreator が内部に保持
    //! する等の対応が必要です。
    //!
    //! 使い方としては、幾つかのパターンがあります。
    //!
    //! @li DefaultControlCreator のように作成したコントロールをメンバのリストに入れ、使いたいときは部品ペイン名で検索する。
    //! @li ControlCreator を継承したクラスで、内部にレイアウトへのポインタを持ち、 CreateControl が呼ばれたらレイアウトの
    //!     適切なメソッドに委譲する。レイアウトはその情報を元にコントロールを作成し、内部のメンバ変数に保持する。
    //!
    virtual void CreateControl(const ControlSrc& controlSrc, Layout* layout) = 0;

};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_BOUNDING_H_

