﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_GFND_TEXTURE_GL_H_
#define NW_GFND_TEXTURE_GL_H_

#include <nw/gfnd/gfnd_Texture.h>
#include <GL/glew.h>

namespace nw
{
namespace gfnd
{

//---------------------------------------------------------------------------
//! @brief        GL 用テクスチャを表すクラスです。
//---------------------------------------------------------------------------
class TextureGL : public Texture
{
public:
    NW_UT_RUNTIME_TYPEINFO(Texture)

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    TextureGL()
        : Texture(),
        m_ID( 0 ),
        m_Target( GL_TEXTURE_2D ),
        m_Level( 1 ),
        m_InternalFormat( GL_RGBA ),
        m_Width( 0 ),
        m_Height( 0 ),
        m_BorderWidth( 0 ),
        m_PixelFormat( GL_RGBA ),
        m_DataType( GL_UNSIGNED_BYTE ) {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~TextureGL() {}

    //---------------------------------------------------------------------------
    //! @brief        2D テクスチャを生成します。
    //!
    //! @param[in]    level           ミップマップレベルです。
    //! @param[in]    internalFormat  内部フォーマットです。
    //! @param[in]    width           テクスチャの幅です。
    //! @param[in]    height          テクスチャの高さです。
    //! @param[in]    border          境界の幅です。
    //! @param[in]    format          テクスチャ画像データのフォーマットです。
    //! @param[in]    type            テクスチャ画像データのデータ型です。
    //! @param[in]    pixels          テクスチャ画像データへのポインタです。
    //!
    //! @return テクスチャ生成に成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool CreateTextureImage2D(
        GLint level,
        GLint internalFormat,
        GLsizei width,
        GLsizei height,
        GLint border,
        GLint format,
        GLenum type,
        const GLvoid* pixels
    );

    //---------------------------------------------------------------------------
    //! @brief 幅を取得します。
    //!
    //! @return 幅の値を返します。
    //---------------------------------------------------------------------------
    virtual u32 GetWidth() const { return m_Width; }

    //---------------------------------------------------------------------------
    //! @brief 高さを取得します。
    //!
    //! @return 高さの値を返します。
    //---------------------------------------------------------------------------
    virtual u32 GetHeight() const { return m_Height; }

    //---------------------------------------------------------------------------
    //! @brief テクスチャ ID を取得します。
    //!
    //! @return テクスチャ ID を返します。
    //---------------------------------------------------------------------------
    GLuint GetID() const { return m_ID; }

    //---------------------------------------------------------------------------
    //! @brief テクスチャターゲットを取得します。
    //!
    //! @return テクスチャターゲットを返します。
    //---------------------------------------------------------------------------
    GLenum GetTarget() const { return m_Target; }

    //---------------------------------------------------------------------------
    //! @brief ミップマップレベルを取得します。
    //!
    //! @return ミップマップレベルを返します。
    //---------------------------------------------------------------------------
    GLint GetMipMapLevel() const { return m_Level; }

    //---------------------------------------------------------------------------
    //! @brief 1 pixel ごとの内部フォーマット(色数)を取得します。
    //!
    //! @return 内部フォーマットを返します。
    //---------------------------------------------------------------------------
    GLint GetInternalFormat() const { return m_InternalFormat; }

    //---------------------------------------------------------------------------
    //! @brief 枠線の幅を取得します。
    //!
    //! @return 枠線の幅を返します。
    //---------------------------------------------------------------------------
    GLint GetBorderWidth() const { return m_BorderWidth; }

    //---------------------------------------------------------------------------
    //! @brief 1 pixelごとの色の配置を取得します。
    //!
    //! @return 1 pixelごとの色の配置を返します。
    //---------------------------------------------------------------------------
    GLint GetPixelFormat() const { return m_PixelFormat; }

    //---------------------------------------------------------------------------
    //! @brief 1 つのデータがどのような型で表されているかを取得します。
    //!
    //! @return データ型を返します。
    //---------------------------------------------------------------------------
    GLenum GetDataType() const { return m_DataType; }


    //---------------------------------------------------------------------------
    //! @brief テクスチャ ID を設定します。
    //!
    //! @param[in]    id              テクスチャ ID です。
    //---------------------------------------------------------------------------
    void SetID( GLuint id ) { m_ID = id; }

    //---------------------------------------------------------------------------
    //! @brief テクスチャターゲットを設定します。
    //!
    //! @param[in]    target          テクスチャターゲットです。
    //---------------------------------------------------------------------------
    void SetTarget( GLenum target ) { m_Target = target; }

    //---------------------------------------------------------------------------
    //! @brief ミップマップレベルを設定します。
    //!
    //! @param[in]    level           ミップマップレベルです。
    //---------------------------------------------------------------------------
    void SetMipMapLevel( GLint level ) { m_Level = level; }

    //---------------------------------------------------------------------------
    //! @brief 1 pixel ごとの内部フォーマット(色数)を設定します。
    //!
    //! @param[in]    fmt             内部フォーマットです。
    //---------------------------------------------------------------------------
    void SetInternalFormat( GLint fmt ) { m_InternalFormat = fmt; }

    //---------------------------------------------------------------------------
    //! @brief 幅を設定します。
    //!
    //! @param[in]    width           幅です。
    //---------------------------------------------------------------------------
    void SetWidth( GLsizei width ) { m_Width = width; }

    //---------------------------------------------------------------------------
    //! @brief 高さを設定します。
    //!
    //! @param[in]    height          高さです。
    //---------------------------------------------------------------------------
    void SetHeight( GLsizei height ) { m_Height = height; }

    //---------------------------------------------------------------------------
    //! @brief 枠の幅を設定します。
    //!
    //! @param[in]    width           枠の幅です(0 or 1)。
    //---------------------------------------------------------------------------
    void SetBorderWidth( GLint width ) { m_BorderWidth = width; }

    //---------------------------------------------------------------------------
    //! @brief ピクセルフォーマットを設定します。
    //!
    //! @param[in]    fmt             ピクセルフォーマットです。
    //---------------------------------------------------------------------------
    void SetPixelFormat( GLint fmt ) { m_PixelFormat = fmt; }

    //---------------------------------------------------------------------------
    //! @brief 1 つのデータがどのような型で表されているかを設定します。
    //!
    //! @param[in]    type            データ型です。
    //---------------------------------------------------------------------------
    void SetDataType( GLenum type ) { m_DataType = type; }

private:
    GLuint  m_ID;
    GLenum  m_Target;
    GLint   m_Level;
    GLint   m_InternalFormat;
    GLsizei m_Width;
    GLsizei m_Height;
    GLint   m_BorderWidth;
    GLint   m_PixelFormat;
    GLenum  m_DataType;
};

} // namespace gfnd
} // namespace nw

#endif // NW_GFND_TEXTURE_GL_H_
