﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_GFND_SHADERHELPER_H_
#define NW_GFND_SHADERHELPER_H_

#include <nw/ut/ut_Range.h>
#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_Foreach.h>
#include <nw/ut/ut_TypeTraits.h>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/gx2.h>
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    #include <GL/glew.h>
#endif

namespace nw
{
namespace gfnd
{

#if defined(NW_PLATFORM_CAFE)
typedef u32 ShaderUniformId; //!< シェーダのユニフォームレジスタ割り当て ID の型です。
typedef s32 ShaderInt; //!< シェーダのユニフォーム整数レジスタ用の型です。

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
typedef GLint ShaderUniformId; //!< シェーダのユニフォームレジスタ割り当て ID の型です。
typedef GLint ShaderInt; //!< シェーダのユニフォーム整数レジスタ用の型です。

#else
#error "Not support platform."
#endif

//! @brief シェーダの設定を支援するためのユーティリティです。
//!
//! "namespace shhelp = nw::gfnd::shader_helper;" のようにエイリアス名を付けることで、
//! 名前を短くすることができます。
//!
namespace shader_helper
{

//----------------------------------------
enum
{
    UNIFORM_COMPONENT_QUANTITY = 4 //!< シェーダユニフォームの要素数です。
};

//! @brief ユニフォーム値の種類を表します。
enum UniformValueType
{
    VERTEX_UNIFORM, //!< 頂点シェーダのユニフォーム値を表します。
    PIXEL_UNIFORM, //!< ピクセルシェーダのユニフォーム値を表します。
    PIXEL_SAMPLER, //!< ピクセルシェーダのサンプラ値を表します。
};

//----------------------------------------
//! @brief シェーダのユニフォームレジスタをキャッシュするためのパラメータです。
struct UniformParam
{
    UniformValueType valueType;
    u32 id;
    const char* uniformName;
};
typedef ut::Range<const UniformParam> UniformParamRange;

#if defined(NW_PLATFORM_CAFE)

//! @brief 頂点シェーダのユニフォームレジスタをキャッシュします。
//!
//! @param[out] uniformIds 取得したユニフォーム ID をキャッシュしておく配列です。
//! @param[in] uniformIdsQuantity uniformIds 配列のサイズです。
//! @param[in] uniformParams キャッシュするユニフォームのパラメータのリストです。
//! @param[in] vertexShader キャッシュ対象の頂点シェーダです。
//! @param[in] pixelShader キャッシュ対象のピクセルシェーダです。
//!
NW_INLINE bool
CacheUniforms(
    ShaderUniformId* uniformIds,
    u32 uniformIdsQuantity,
    const UniformParamRange& uniformParams,
    GX2VertexShader* vertexShader,
    GX2PixelShader* pixelShader)
{
    bool result = true;
    NW_FOREACH(const UniformParam& param, uniformParams)
    {
        if (param.id < uniformIdsQuantity)
        {
            //NW_LOG("Bind uniform value : %s\n", value.uniformName);
            switch (param.valueType)
            {
            case VERTEX_UNIFORM:
                uniformIds[param.id] = static_cast<u32>(
                    GX2GetVertexUniformVarOffset(vertexShader, param.uniformName));
                break;
            case PIXEL_UNIFORM:
                uniformIds[param.id] = static_cast<u32>(
                    GX2GetPixelUniformVarOffset(pixelShader, param.uniformName));
                break;
            case PIXEL_SAMPLER:
                uniformIds[param.id] = static_cast<u32>(
                    GX2GetPixelSamplerVarLocation(pixelShader, param.uniformName));
                break;
            }

            if (uniformIds[param.id] == GX2_UNIFORM_VAR_INVALID_OFFSET)
            {
                result = false;
                NW_LOG("Uniform is invalid.[%s]\n", param.uniformName);
            }
        }
        else
        {
            result = false;
            NW_LOG("Uniform ID is overflow.[%s]\n", param.uniformName);
        }
    }
    return result;
}
#endif

//! @brief バーテックスシェーダの固定レジスタを１つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, TValue value0)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, 0, 0, 0 };
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform1f(id, value0);
#endif
}

//! @brief バーテックスシェーダの固定レジスタを２つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, TValue value0, TValue value1)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, 0, 0 };
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform2f(id, value0, value1);
#endif
}

//! @brief バーテックスシェーダの固定レジスタを３つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, TValue value0, TValue value1, TValue value2)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, value2, 0 };
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform3f(id, value0, value1, value2);
#endif
}

//! @brief バーテックスシェーダの固定レジスタを４つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, TValue value0, TValue value1, TValue value2, TValue value3)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, value2, value3 };
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform4f(id, value0, value1, value2, value3);
#endif
}

//! @brief ピクセルシェーダの固定レジスタを１つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, TValue value0)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, 0, 0, 0 };
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform1f(id, value0);
#endif
}

//! @brief ピクセルシェーダの固定レジスタを２つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, TValue value0, TValue value1)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, 0, 0 };
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform2f(id, value0, value1);
#endif
}

//! @brief ピクセルシェーダの固定レジスタを３つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, TValue value0, TValue value1, TValue value2)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, value2, 0 };
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform3f(id, value0, value1, value2);
#endif
}

//! @brief ピクセルシェーダの固定レジスタを4つ設定します。
//!
//! @tparam レジスタに設定する値の型です。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] value 設定する値です。
//!
template<typename TValue>
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, TValue value0, TValue value1, TValue value2, TValue value3)
{
#if defined(NW_PLATFORM_CAFE)
    NW_STATIC_ASSERT(sizeof(TValue) == 4);
    const TValue values[UNIFORM_COMPONENT_QUANTITY] = { value0, value1, value2, value3 };
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY, values);

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_STATIC_ASSERT((ut::IsSame<TValue, float>::value));
    glUniform4f(id, value0, value1, value2, value3);
#endif
}

//! @brief バーテックスシェーダの固定レジスタにカラーを設定します。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] color 設定するカラーです。
//!
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, const ut::FloatColor& color)
{
#if defined(NW_PLATFORM_CAFE)
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY, color.ToArray());
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    glUniform4fv(id, 1, color.ToArray());
#endif
}

//! @brief ピクセルシェーダの固定レジスタにカラーを設定します。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] color 設定するカラーです。
//!
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, const ut::FloatColor& color)
{
#if defined(NW_PLATFORM_CAFE)
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY, color.ToArray());
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    glUniform4fv(id, 1, color.ToArray());
#endif
}

//! @brief バーテックスシェーダの固定レジスタに4x4マトリクスを設定します。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] matrix 設定するマトリクスです。
//!
NW_INLINE void
SetVertexUniformReg(ShaderUniformId id, const math::MTX44& matrix)
{
#if defined(NW_PLATFORM_CAFE)
    GX2SetVertexUniformReg(id, UNIFORM_COMPONENT_QUANTITY * 4, matrix);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    glUniformMatrix4fv(id, 1, GL_FALSE, matrix);
#endif
}

//! @brief ピクセルシェーダの固定レジスタに4x4マトリクスを設定します。
//!
//! @param[in] id 設定する固定レジスタのオフセット値です。
//! @param[in] matrix 設定するマトリクスです。
//!
NW_INLINE void
SetPixelUniformReg(ShaderUniformId id, const math::MTX44& matrix)
{
#if defined(NW_PLATFORM_CAFE)
    GX2SetPixelUniformReg(id, UNIFORM_COMPONENT_QUANTITY * 4, matrix);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    glUniformMatrix4fv(id, 1, GL_FALSE, matrix);
#endif
}

//----------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
template<>
NW_INLINE void
SetVertexUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0)
{
    glUniform1i(id, value0);
}

template<>
NW_INLINE void
SetVertexUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1)
{
    glUniform2i(id, value0, value1);
}

template<>
NW_INLINE void
SetVertexUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1, ShaderInt value2)
{
    glUniform3i(id, value0, value1, value2);
}

template<>
NW_INLINE void
SetVertexUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1, ShaderInt value2, ShaderInt value3)
{
    glUniform4i(id, value0, value1, value2, value3);
}

template<>
NW_INLINE void
SetPixelUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0)
{
    glUniform1i(id, value0);
}

template<>
NW_INLINE void
SetPixelUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1)
{
    glUniform2i(id, value0, value1);
}

template<>
NW_INLINE void
SetPixelUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1, ShaderInt value2)
{
    glUniform3i(id, value0, value1, value2);
}

template<>
NW_INLINE void
SetPixelUniformReg<ShaderInt>(ShaderUniformId id, ShaderInt value0, ShaderInt value1, ShaderInt value2, ShaderInt value3)
{
    glUniform4i(id, value0, value1, value2, value3);
}

#endif

} // namespace shader_helper

} // namespace gfnd
} // namespace nw

#endif // NW_GFND_SHADERHELPER_H_
