﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_USERDATA_H_
#define NW_G3D_RES_USERDATA_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>

namespace nw { namespace g3d { namespace res {

#if defined( _MSC_VER )
#pragma warning(push)
#pragma warning(disable:4201)
#endif

//! @brief ユーザーデータの構造体です。
struct ResUserDataData
{
    BinString ofsName;
    u16 count;
    u8 type;
    u8 reserved;
    union Data
    {
        s32 iValue[1];
        f32 fValue[1];
        Offset ofsString[1];
        struct
        {
            u32 size;
            u8 value[4];
        };
    } data;
};

#if defined( _MSC_VER )
#pragma warning(pop)
#endif

//! @brief ユーザーデータのリソースです。
class ResUserData : private ResUserDataData
{
    NW_G3D_RES_COMMON(ResUserData);

public:
    //! @brief データタイプを表します。
    enum Type
    {
        //! @brief Int 値として表されるデータです。
        //! @deprecated 非推奨です。Type_Int を使用してください。
        INT,
        //! @brief Float 値として表されるデータです。
        //! @deprecated 非推奨です。Type_Float を使用してください。
        FLOAT,
        //! @brief ascii の値として表されるデータです。
        //! @deprecated 非推奨です。Type_String を使用してください。
        STRING, // 改行は LF
        //! @brief utf-16 の値として表されるデータです。
        //! @deprecated 非推奨です。Type_WString を使用してください。
        WSTRING, // 改行は LF、エンディアンはターゲットと同一
        //! @brief バイト列 として表されるデータです。
        //! @deprecated 非推奨です。Type_Stream を使用してください。
        STREAM,

        //! @brief 32-bit 符号付き整数です。
        Type_Int = INT,
        //! @brief 浮動小数点数です。
        Type_Float,
        //! @brief utf-8 文字列です。
        Type_String,
        //! @brief utf-16 の値として表されるデータです。
        Type_WString,
        //! @brief バイト列です。
        Type_Stream,

    };

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief データの数を取得します。
    int GetCount() const { return ref().count; }

    //! @brief データのタイプを取得します。
    Type GetType() const { return static_cast<Type>(ref().type); }

    //! @brief Int 値としてデータを取得します。
    int* GetInt()
    {
        NW_G3D_ASSERT(ref().type == Type_Int);
        return reinterpret_cast<int*>(ref().data.iValue);
    }

    //! @brief Int 値としてデータを取得します。
    const int* GetInt() const
    {
        NW_G3D_ASSERT(ref().type == Type_Int);
        return reinterpret_cast<const int*>(ref().data.iValue);
    }

    //! @brief Float 値としてデータを取得します。
    float* GetFloat()
    {
        NW_G3D_ASSERT(ref().type == Type_Float);
        return reinterpret_cast<float*>(ref().data.fValue);
    }

    //! @brief Float 値としてデータを取得します。
    const float* GetFloat() const
    {
        NW_G3D_ASSERT(ref().type == Type_Float);
        return reinterpret_cast<const float*>(ref().data.fValue);
    }

    //! @brief 文字列としてデータを取得します。
    const char* GetString(int strIndex) const
    {
        NW_G3D_ASSERT(ref().type == Type_String);
        NW_G3D_ASSERT_INDEX_BOUNDS(strIndex, ref().count);
        return ref().data.ofsString[strIndex].to_ptr<char>();
    }

    //! @brief 文字列としてデータを取得します。
    const char16* GetWString(int strIndex) const
    {
        NW_G3D_ASSERT(ref().type == Type_WString);
        NW_G3D_ASSERT_INDEX_BOUNDS(strIndex, ref().count);
        return ref().data.ofsString[strIndex].to_ptr<char16>();
    }

    //! @brief バイト列としてデータを取得します。
    void* GetStream()
    {
        NW_G3D_ASSERT(ref().type == Type_Stream);
        return ref().data.value;
    }

    //! @brief バイト列としてデータを取得します。
    const void* GetStream() const
    {
        NW_G3D_ASSERT(ref().type == Type_Stream);
        return ref().data.value;
    }

    //! @brief バイト列のサイズを取得します。
    size_t GetStreamSize() const
    {
        NW_G3D_ASSERT(ref().type == Type_Stream);
        return ref().data.size;
    }
};

}}} // namespace nw::g3d

#endif // NW_G3D_RES_USERDATA_H_
