﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESTEXTURE_H_
#define NW_G3D_RES_RESTEXTURE_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/res/g3d_ResUserData.h>

namespace nw { namespace g3d { namespace res {

class BindCallback;
class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャの構造体です。
struct ResTextureData
{
    BinaryBlockHeader blockHeader;
    GfxTexture_t gfxTexture;

    BinString ofsName;
    BinString ofsPath;
    Offset ofsData;
    Offset ofsMipData;
    Offset ofsUserDataDic;

    u16 numUserData;
    u16 reserved;
};

//! @brief テクスチャのリソースです。
class ResTexture : private ResTextureData
{
    NW_G3D_RES_COMMON(ResTexture);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'T', 'E', 'X') };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief テクスチャをセットアップします。
    void Setup();

    //! @brief テクスチャをクリーンアップします。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief ミップマップ０のイメージサイズを取得します。
    size_t GetBaseSize() const { return GetGfxTexture()->GetBaseSize(); }

    //! @brief ミップマップ１以上の合計サイズを取得します。
    size_t GetMipSize() const { return GetGfxTexture()->GetMipSize(); }

    //! @brief ミップマップ０のイメージデータへのポインタを取得します。
    void* GetData() { return ref().ofsData.to_ptr(); }

    //! @brief ミップマップ０のイメージデータへのポインタを取得します。
    const void* GetData() const { return ref().ofsData.to_ptr(); }

    //! @brief ミップマップデータへのポインタを取得します。
    void* GetMipData() { return ref().ofsMipData.to_ptr(); }

    //! @brief ミップマップデータへのポインタを取得します。
    const void* GetMipData() const { return ref().ofsMipData.to_ptr(); }

    //! @brief GfxTexture を取得します。
    GfxTexture* GetGfxTexture() { return GfxTexture::DownCast(&ref().gfxTexture); }

    //! @brief GfxTexture を取得します。
    const GfxTexture* GetGfxTexture() const { return GfxTexture::DownCast(&ref().gfxTexture); }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャリソースへの参照の構造体です。
struct ResTextureRefData
{
    BinString ofsName; //!< 参照先テクスチャの名前
    Offset ofsRefData;
};

//! @brief テクスチャリソースへの参照です。
class ResTextureRef : private ResTextureRefData
{
    NW_G3D_RES_COMMON(ResTextureRef);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 参照先が設定されているかどうかを取得します。
    bool IsBound() const { return ref().ofsRefData != 0; }

    //! @brief 名前が一致するファイル内のデータと参照解決を行います。
    BindResult Bind(const ResFile* pFile);

    //! @brief  名前が異なっていても強制的に参照設定を行います。
    void ForceBind(const ResTexture* pTarget);

    //! @brief 参照設定を解除します。
    void Release();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_DEREFERENCE_DECL(Texture)

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESTEXTURE_H_
