﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESTEXPATTERNANIM_H_
#define NW_G3D_RES_RESTEXPATTERNANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_Binding.h>

namespace nw { namespace g3d { namespace res {

class BindCallback;
class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief パターンアニメーションの構造体です。
struct ResTexPatternMatAnimData
{
    u16 numPatAnim;
    u16 numCurve;
    s32 beginCurve; // TexPatternAnim 内での先頭 Curve のインデックス
    s32 beginPatAnim; // TexPatternAnim 内での PatAnimInfo のインデックス

    BinString ofsName;

    Offset ofsPatAnimInfoArray;
    Offset ofsCurveArray;
    Offset ofsBaseValueArray;
};

//! @brief パターンアニメーションのリソースです。
class ResTexPatternMatAnim : private ResTexPatternMatAnimData
{
    NW_G3D_RES_COMMON(ResTexPatternMatAnim);

public:
    //! @brief パターンアニメーションの要素情報です。
    struct PatAnimInfo
    {
        s8 curveIndex; // MatAnim 内での Curve のインデックス
        s8 subbindIndex; // Material 内での Texture のインデックス
        u16 reserved;

        BinString ofsName;
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{
    //! @brief 事前に ResMaterial と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResMaterial* target);

    //! @brief 事前に ResModel と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResMaterial* target) const;

    //! @brief 初期化を行います。
    void Init(u16* pResult) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(u16* pResult) const
    {
        Init(pResult);
    }

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief パターンアニメーションを評価します。
    void Eval(u16* pResult, float frame, const s8* pSubBindIndex) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(u16* pResult, float frame, const s8* pSubBindIndex) const
    {
        Eval(pResult, frame, pSubBindIndex);
    }

    //! @brief パターンアニメーションを評価します。
    void Eval(u16* pResult, float frame, const s8* pSubBindIndex,
        AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(u16* pResult, float frame, const s8* pSubBindIndex,
        AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pSubBindIndex, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //! @brief アニメーション数を取得します。
    int GetPatAnimCount() const { return ref().numPatAnim; }

    //@}

};

//--------------------------------------------------------------------------------------------------

//! @brief テクスチャパターンアニメーションの構造体です。
struct ResTexPatternAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    bit16 flag;
    u16 numUserData;
    s32 numFrame;
    u16 numTextureRef;
    u16 numMatAnim; // BindTable のサイズに影響
    s32 numPatAnim; // SubBindTable のサイズに影響
    s32 numCurve; // Context のサイズに影響
    u32 bakedSize;

    Offset ofsBindModel;
    Offset ofsBindIndexArray; // Material のインデックス
    Offset ofsMatAnimArray;
    Offset ofsTextureRefDic;
    Offset ofsUserDataDic;
};

//! @brief テクスチャパターンアニメーションのリソースです。
class ResTexPatternAnim : private ResTexPatternAnimData
{
    NW_G3D_RES_COMMON(ResTexPatternAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'T', 'X', 'P') };

    //! @brief テクスチャパターンアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループすることを表します。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief テクスチャを含む ResFile を用いて参照解決します。
    BindResult Bind(const ResFile* pFile);

    //! @brief テクスチャを含む ResFile を用いて参照解決します。
    BindResult Bind(const ResFile* pFile, BindCallback* pCallback);

    //! @brief 指定した名前の参照に対してテクスチャを設定します。
    bool ForceBind(const ResTexture* pTexture, const char* name);

    //! @brief テクスチャの参照設定を解除します。
    void Release();

    //! @brief 事前に ResModel と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResModel* pModel);

    //! @brief 事前に ResModel と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResModel* target) const;

    //! @brief カーブをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief カーブに戻します。
    void* ResetCurve();

    //! @brief テクスチャパターンアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief パターンアニメーション数を取得します。
    int GetPatAnimCount() const { return ref().numPatAnim; }

    //! @brief カーブ数を取得します。
    int GetCurveCount() const { return ref().numCurve; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)
    NW_G3D_RES_FIELD_CLASS_DECL(ResModel, BindModel)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResTexPatternMatAnim, MatAnim, GetName())
    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResTextureRef, TextureRef, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESTEXPATTERNANIM_H_
