﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSHADER_H_
#define NW_G3D_RES_RESSHADER_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GX2Struct.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/math/g3d_MathCommon.h>

#include <limits.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

enum ShaderLocation
{
    SHADER_LOCATION_NONE        = 0xFFFFFFFF,
    SHADER_OFFSET_NONE          = -1,
    SHADER_PROGRAM_NONE         = -1
};

enum Stage
{
    STAGE_VERTEX,
    STAGE_GEOMETRY,
    STAGE_FRAGMENT,
    STAGE_COMPUTE,
    NUM_STAGE
};

//! @brief シェーダバリエーション範囲の構造体です。
//!
//! 内部の変数は直接使用しないでください。
//!
struct ShaderRange
{
    //! @briefprivate
    const bit32* pBegin;
    //! @briefprivate
    const bit32* pEnd;
};

//! @briefprivate シェーダプログラムの更新が必要な場合に呼び出されるコールバックで、内部で使用します。
//!
typedef void (*UpdateProgramCallback)(class ResShadingModel* pShader, int programIndex);

//--------------------------------------------------------------------------------------------------

//! @brief オフライン DL シェーダの構造体です。
struct ResOfflineDLShaderData
{
    u8 gx2PatchType;
    u8 gx2ShaderMode;
    u16 reserved;
    u32 byteOffset;

    u32 shaderSize;
    u32 offlineDLSize;
    Offset ofsShader;
    mutable Offset ofsOfflineDL; // ディスプレイリストコールの引数が非 const なので mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダプログラムの構造体です。
struct ResShaderProgramData
{
    u16 flag;
    u8 numSampler;
    u8 numUniformBlock;
    u32 attribActiveFlag; // 下位ビットから順に 1 属性 1 ビットで有効 / 無効を表すフラグ
    u16 numALUInst[NUM_STAGE];
    u16 numTfetchInst[NUM_STAGE];
    u16 numVfetchInst[NUM_STAGE];
    u16 numCflowInst[NUM_STAGE];
    u32 handle; // for PC

    Offset ofsSamplerTable;
    Offset ofsUniformBlockTable;
    Offset ofsVertexShader;
    Offset ofsGeometryShader;
    Offset ofsGeometryCopyShader;
    Offset ofsFragmentShader;
    Offset ofsComputeShader;
    Offset ofsShadingModel; // 親への参照
};

//! @brief シェーダプログラムのリソースです。
class ResShaderProgram : private ResShaderProgramData
{
    NW_G3D_RES_COMMON(ResShaderProgram);

public:
    enum Flag
    {
        UPDATE_REQURIED     = 0x1 << 0,
        INITIALIZED         = 0x1 << 1,
        OWN_VERTEX_SHADER   = 0x1 << 2,
        OWN_GEOMETRY_SHADER = 0x1 << 3,
        OWN_FRAGMENT_SHADER = 0x1 << 4,
        OWN_COMPUTE_SHADER  = 0x1 << 5,

        OFFLINE_DL          = 0x1 << 8
    };

    // 互換用
    enum Stage
    {
        STAGE_VERTEX = ::nw::g3d::res::STAGE_VERTEX,
        STAGE_GEOMETRY = ::nw::g3d::res::STAGE_GEOMETRY,
        STAGE_FRAGMENT = ::nw::g3d::res::STAGE_FRAGMENT,
        STAGE_COMPUTE = ::nw::g3d::res::STAGE_COMPUTE,
        NUM_STAGE = ::nw::g3d::res::NUM_STAGE
    };

    struct GLShaderProgram
    {
        u32 shaderCodeSize;
        u32 compressedShaderCodeSize;
        u32 shaderSourceSize;
        u32 compressedShaderSourceSize;
        u32 binaryFormat;
        Offset ofsShaderCode;
        Offset ofsShaderSource;
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief シェーダプログラムをセットアップします。
    //!
    //! この関数は ResShaderArchive::Setup() から間接的によばれます。
    //! シェーダブログラムは内部的に共有されている可能性があるため、直接よばないようにしてください。
    //!
    void Setup();

    //! @brief シェーダプログラムをクリーンアップします。
    //!
    //! この関数は ResShaderArchive::Cleanup() から間接的によばれます。
    //! シェーダプログラムは内部的に共有されている可能性があるため、直接よばないようにしてください。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief 必要な場合にシェーダプログラムの遅延コンパイルを行います。
    void Update();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief シェーダプログラムを設定します。
    //!
    //! オフライン DL シェーダの場合はディスプレイリストをコピーします。
    //!
    void Load() const;

    //! @brief シェーダプログラムのオフラインディスプレイリストを現在のディスプレイリストまたはコマンドバッファに追加します。
    //!
    //! シェーダがオフライン DL でコンバートされている必要があります。
    //!
    void LoadDL() const;

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief インデクス引きでサンプラーのロケーションを取得します。
    //!
    //! インデクスは ResShadingModel::GetSamplerIndex() で取得できます。
    //! 初期化時にインデクスを取得しておきキャッシュすることを推奨します。
    //!
    u32 GetSamplerLocation(int samplerIndex, Stage stage) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(samplerIndex, ref().numSampler);
        NW_G3D_ASSERT_INDEX_BOUNDS(stage, NUM_STAGE);
        const s8* pSamplerTable = ref().ofsSamplerTable.to_ptr<s8>();
        return static_cast<u32>(pSamplerTable[samplerIndex * NUM_STAGE + stage]);
    }

    //! @brief インデクス引きで UniformBlock のロケーションを取得します。
    //!
    //! インデックスは ResShadingModel::GetSystemBlockIndex() か
    //! ResShadingModel::GetUniformBlockIndex() で取得できます。
    //! 初期化時にインデクスを取得しておきキャッシュすることを推奨します。
    //!
    //!
    u32 GetUniformBlockLocation(int blockIndex, Stage stage) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(blockIndex, ref().numUniformBlock);
        NW_G3D_ASSERT_INDEX_BOUNDS(stage, NUM_STAGE);
        const s8* pUniformBlockTable = ref().ofsUniformBlockTable.to_ptr<s8>();
        return static_cast<u32>(pUniformBlockTable[blockIndex * NUM_STAGE + stage]);
    }

    //! @brief シェーダモードを取得します。
    GX2ShaderMode GetShaderMode() const;

    //! @brief シェーダプログラム内の ALU 命令の数を取得します。
    int GetALUInstCount(Stage stage) const { return ref().numALUInst[stage]; }

    //! @brief シェーダプログラム内の TFETCH 命令の数を取得します。
    int GetTfetchInstCount(Stage stage) const { return ref().numTfetchInst[stage]; }

    //! @brief シェーダプログラム内のコントロールフロー命令の数を取得します。
    int GetCflowInstCount(Stage stage) const { return ref().numCflowInst[stage]; }

    //! @brief インデクス引きで対象の頂点属性がアクティブかどうかを取得します。
    bool IsAttribActive(int attribIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(attribIndex, sizeof(u32) * CHAR_BIT);
        return ( ref().attribActiveFlag & ( 0x01 << attribIndex ) ) != 0;
    }

    //! @brief IsAttribActive() の別名関数です。
    bool IsAttrActive(int attrIndex) const
    {
        return IsAttribActive(attrIndex);
    }

    //! @brief シェーダのディスプレイリストが利用可能かどうかを取得します。
    bool IsDLAvailable() const
    {
        return (ref().flag & OFFLINE_DL) != 0;
    }

    //! @brief シェーディングモデルを取得します。
    //!
    //! @return ResShadingModel のポインタを返します。
    //!
    ResShadingModel* GetShadingModel()
    {
        return ref().ofsShadingModel.to_ptr<ResShadingModel>();
    }

    //! @brief シェーディングモデルを取得します。
    //!
    //! @return ResShadingModel のポインタを返します。
    //!
    const ResShadingModel* GetShadingModel() const
    {
        return ref().ofsShadingModel.to_ptr<const ResShadingModel>();
    }

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダオプションの構造体です。
struct ResShaderOptionData
{
    u8 numChoice;
    u8 defaultIndex;
    u16 branchOffset; // uniform block でのオフセット。
    u8 flag;

    u8 keyOffset; // static option の場合は 0、dynamic option の場合は staticKeyLength
    u8 bit32Index; // 何個目の bit32 か。
    u8 bit32Shift; // choiceIndex を何ビットシフトするか。
    bit32 bit32Mask; // 所属する bit32 における該当ビットのマスク。

    BinString ofsName;
    Offset ofsChoiceDic;
    Offset ofsChoiceValues; // choice を int に変換した可変長のバイナリ列。
};

//! @brief シェーダオプションのリソースです。
class ResShaderOption : private ResShaderOptionData
{
    NW_G3D_RES_COMMON(ResShaderOption);

public:
    enum
    {
        INVALID_OFFSET = -1
    };

    enum Flag
    {
        BRANCH      = 0x1 << 0,
        STATIC      = 0x1 << 1
    };

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief デフォルトの Choice のインデクス取得します。
    int GetDefaultIndex() const { return ref().defaultIndex; }

    //! @brief UniformBlock を使用して分岐する際の変数のオフセットを取得します。
    s32 GetBranchOffset() const { return ref().branchOffset - 1; }

    //! @brief branch かどうかを取得します。
    bool IsBranch() const { return !!(flag & BRANCH); }

    //! @brief static かどうかを取得します。
    bool IsStatic() const { return !!(flag & STATIC); }

    //! @brief dynamic かどうかを取得します。
    bool IsDynamic() const { return !(flag & STATIC); }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_EMPTY_DIC_DECL(Choice)

    //! @brief 指定した Choice でキーを書き換えます。
    void WriteStaticKey(bit32* pStaticKey, int choiceIndex) const;

    //! @brief キーから Choice を読み込みます。
    int ReadStaticKey(const bit32* pStaticKey) const;

    //! @brief 指定した Choice でキーを書き換えます。
    void WriteDynamicKey(bit32* pDynamicKey, int choiceIndex) const;

    //! @brief キーから Choice を読み込みます。
    int ReadDynamicKey(const bit32* pDynamicKey) const;

    //@}
};

//! @brief 頂点属性変数の構造体です。
struct ResAttribVarData
{
    u8 index; //!< シェーダ内でのインデクス
    u8 gx2VarType;
    u8 arrayLength;
    s8 location; // 頂点属性は explicit location 必須
};

//! @brief 頂点属性変数です。
class ResAttribVar : private ResAttribVarData
{
    NW_G3D_RES_COMMON(ResAttribVar);

public:
    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief シェーディングモデル内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    //! @brief GX2VarType を取得します。
    GX2VarType GetGX2VarType() const { return static_cast<GX2VarType>(ref().gx2VarType); }

    //! @brief 配列長を取得します。
    int GetArrayLength() const { return ref().arrayLength; }

    //! @brief ロケーションを取得します。
    u32 GetLocation() const { return static_cast<u32>(ref().location); }

    //@}
};

typedef ResAttribVar ResAttrVar;

//--------------------------------------------------------------------------------------------------

//! @brief サンプラー変数の構造体です。
struct ResSamplerVarData
{
    u8 index; //!< シェーダ内でのインデクス
    u8 gx2SamplerType;
    u8 arrayLength;
    u8 reserved;
    BinString ofsAlt;
};

//! @brief サンプラー変数です。
class ResSamplerVar : private ResSamplerVarData
{
    NW_G3D_RES_COMMON(ResSamplerVar);

public:
    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief シェーディングモデル内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    //! @brief GX2VarType を取得します。
    GX2SamplerType GetGX2SamplerType() const
    {
        return static_cast<GX2SamplerType>(ref().gx2SamplerType);
    }

    //! @brief 配列長を取得します。
    int GetArrayLength() const { return ref().arrayLength; }

    NW_G3D_RES_FIELD_STRING_DECL(Alt)

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief Uniform 変数の構造体です。
struct ResUniformVarData
{
    s32 index; //!< シェーダ内でのインデクス
    u16 arrayLength;
    u8 gx2VarType;
    u8 blockIndex;
    u16 offset;
    u8 shaderParamType; // ResShaderParam::Type
    u8 reserved;
    BinString ofsConverter;
};

//! @brief Uniform 変数です。
class ResUniformVar : private ResUniformVarData
{
    NW_G3D_RES_COMMON(ResUniformVar);

public:
    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief シェーディングモデル内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    //! @brief 所属する UniformBlock のインデクスを取得します。
    int GetBlockIndex() const { return ref().blockIndex; }

    //! @brief GX2VarType を取得します。
    GX2VarType GetGX2VarType() const { return static_cast<GX2VarType>(ref().gx2VarType); }

    //! @brief 配列長を取得します。
    int GetArrayLength() const { return ref().arrayLength; }

    //! @brief UniformBlock の先頭からのバイトオフセットを取得します。
    s32 GetOffset() const { return ref().offset - 1; }

    NW_G3D_RES_FIELD_STRING_DECL(Converter);

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief UniformBlock 変数の構造体です。
struct ResUniformBlockVarData
{
    u8 index;
    u8 type;
    u16 size;
    u16 numUniform;
    u16 reserved;

    Offset ofsUniformDic;
    Offset ofsDefault; // TYPE_MATERIAL の場合のみ値が存在する。
};

//! @brief UniformBlock 変数です。
class ResUniformBlockVar : private ResUniformBlockVarData
{
    NW_G3D_RES_COMMON(ResUniformBlockVar);

public:
    //! @brief UniformBlock の種類です。
    enum Type
    {
        //! @brief 種類が無指定です。
        TYPE_NONE,

        //! @brief マテリアル用の UniformBlock です。
        TYPE_MATERIAL,

        //! @brief シェイプ用の UniformBlock です。
        TYPE_SHAPE,

        //! @brief スケルトン用の UniformBlock です。
        TYPE_SKELETON,

        //! @brief オプション用の UniformBlock です。
        TYPE_OPTION,

        //! @briefprivate
        NUM_TYPE
    };

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief シェーディングモデル内でのインデクスを取得します。
    int GetIndex() const { return ref().index; }

    //! @brief UniformBlock のバイト単位のサイズを取得します。
    size_t GetSize() const { return ref().size; }

    //! @brief UniformBlock の種類を取得します。
    Type GetType() const { return static_cast<Type>(ref().type); }

    //! @brief UniformBlock のデフォルト値を取得します。
    const void* GetDefault() const { return ref().ofsDefault.to_ptr(); }

    NW_G3D_RES_FIELD_DIC_DECL(ResUniformVar, Uniform)

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーディングモデルの構造体です。
struct ResShadingModelData
{
    u8 staticKeyLength;
    u8 dynamicKeyLength;
    u16 numStaticOption;
    u16 numDynamicOption;
    u16 numShaderProgram;
    u8 numAttrib;
    u8 numSampler;
    u8 numUniformBlock;
    u8 maxVSRingItem; // vec4 換算
    u16 maxGSRingItem; // vec4 換算
    u16 reserved;
    u8 maxGPRs[ResShaderProgram::NUM_STAGE];
    u8 maxStackEntries[ResShaderProgram::NUM_STAGE];
    s32 numUniform;
    s8 systemBlockIndices[ResUniformBlockVar::NUM_TYPE - 1];
    s32 defaultProgramIndex;

    BinString ofsName;
    u32 revision;

    Offset ofsStaticOptionArray;
    Offset ofsStaticOptionDic;
    Offset ofsDynamicOptionArray;
    Offset ofsDynamicOptionDic;
    Offset ofsAttribArray;
    Offset ofsAttribDic;
    Offset ofsSamplerArray;
    Offset ofsSamplerDic;
    Offset ofsUniformBlockArray;
    Offset ofsUniformBlockDic;
    Offset ofsUniformArray;
    Offset ofsShaderProgramArray;
    Offset ofsKeyTable;
    Offset ofsShaderArchive; // 親への参照

    BinPtr pUserPtr;
    BinPtr pCallbackParam; // edit ライブラリ用

    Offset ofsGLShaderInfo; // GL 版シェーダ用
};

//! @brief シェーディングモデルのリソースです。
class ResShadingModel : private ResShadingModelData
{
    NW_G3D_RES_COMMON(ResShadingModel);

public:

    //! @brief GL 版シェーダ用の情報です。
    struct GLShaderInfo
    {
        u32 numStreamout;
        Offset ofsSamplerTable; // ResName[ ResShadingModelData::numSampler * 3 ]
        Offset ofsBlockTable; // ResName[ ResShadingModelData::numUniformBlock * 3 ]
        Offset ofsStreamout; // ResName[ numStreamout ]
        struct SharedSource
        {
            u32 sharedSourceSize;
            u32 compressedSharedSourceSize;
            Offset ofsCompressedSharedSource;
            BinPtr pSharedSource;
        } sharedSources[NUM_STAGE];
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief シェーディングモデルをセットアップします。
    void Setup();

    //! @brief シェーディングモデルをクリーンアップします。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)

    //! @brief 頂点シェーダからリングバッファに出力される頂点データの最大数を取得します。
    u32 GetMaxInputRingItem() const { return ref().maxVSRingItem << 2; } // 構造体サイズの節約のためリソースではシフトされています。

    //! @brief ジオメトリシェーダからリングバッファに出力される頂点データの最大数を取得します。
    u32 GetMaxOutputRingItem() const { return ref().maxGSRingItem << 2; } // 構造体サイズの節約のためリソースではシフトされています。

    //! @brief シェーダが用いる GPR の最大数を取得します。
    u32 GetMaxGPRs(ResShaderProgram::Stage stage) const { return ref().maxGPRs[stage]; }

    //! @brief シェーダが用いるスタックエントリーの最大数を取得します。
    u32 GetMaxStackEntries(ResShaderProgram::Stage stage) const { return ref().maxStackEntries[stage]; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //@}

    //----------------------------------------
    //! @name オプション
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResShaderOption, StaticOption, GetName())
    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResShaderOption, DynamicOption, GetName())

    //@}

    //----------------------------------------
    //! @name 頂点属性
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResAttribVar, Attrib, GetName())
    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL_WITH_FUNCNAME(ResAttribVar, Attr, Attrib, GetName())

    //@}

    //----------------------------------------
    //! @name サンプラー
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResSamplerVar, Sampler, GetName())

    //@}

    //----------------------------------------
    //! @name Uniform
    //@{

    NW_G3D_RES_FIELD_CLASS_NAMED_ARRAY_DECL_DETAIL(ResUniformBlockVar, UniformBlock, GetName())
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResUniformVar, Uniform, GetName())

    //! @brief システム既定の UniformBlock のインデクスを取得します。
    int GetSystemBlockIndex(ResUniformBlockVar::Type type) const
    {
        NW_G3D_ASSERT(type != ResUniformBlockVar::TYPE_NONE);
        return ref().systemBlockIndices[type - 1];
    }

    //@}

    //----------------------------------------
    //! @name シェーダプログラム
    //@{

    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResShaderProgram, ShaderProgram, GetName())

    //! @brief シェーダプログラムの検索に使用するキーの bit32 換算の長さを取得します。
    int GetKeyLength() const { return ref().staticKeyLength + ref().dynamicKeyLength; }

    //! @brief シェーダプログラムの検索に使用する static オプションのキーの bit32 換算の長さを取得します。
    int GetStaticKeyLength() const { return ref().staticKeyLength; }

    //! @brief シェーダプログラムの検索に使用する dynamic オプションのキーの bit32 換算の長さを取得します。
    int GetDynamicKeyLength() const { return ref().dynamicKeyLength; }

    //! @brief デフォルトのシェーダキーを書き込みます。
    void WriteDefaultStaticKey(bit32* pStaticKey) const;

    //! @brief デフォルトのシェーダキーを書き込みます。
    void WriteDefaultDynamicKey(bit32* pDynamicKey) const;

    //! @brief 存在しないシェーダキーを書き込みます。
    void WriteInvalidDynamicKey(bit32* pDynamicKey) const;

    //! @brief キーで検索してシェーダプログラムのインデックスを取得します。
    int FindProgramIndex(const bit32* pKey) const;

    //! @brief キーで検索してシェーダプログラムのインデックスを取得します。
    int FindProgramIndex(const ShaderRange& range, const bit32* pDynamicKey) const;

    //! @brief static オプションのキーで検索して該当するシェーダプログラムの範囲を取得します。
    bool FindProgramRange(ShaderRange* pRange, const bit32* pStaticKey) const;

    //! @brief デフォルトプログラムのインデックスを取得します。
    int GetDefaultProgramIndex() const { return ref().defaultProgramIndex; }

    //! @brief 指定したシェーダプログラムのインデックスを取得します。
    int GetProgramIndex(const ResShaderProgram* pProgram) const
    {
        // 配列に並んでいることを利用して計算します。
        int programIndex = static_cast<int>(pProgram - GetShaderProgram(0));
        NW_G3D_ASSERT_INDEX_BOUNDS(programIndex, GetShaderProgramCount());
        return programIndex;
    }

    //! @brief 指定したインデクスのシェーダプログラムに対応するキーを取得します。
    const bit32* GetKey(int programIndex) const;

    //! @brief 指定したインデクスのシェーダプログラムに対応するキーを取得します。
    const bit32* GetStaticKey(int programIndex) const;

    //! @brief 指定したインデクスのシェーダプログラムに対応するキーを取得します。
    const bit32* GetDynamicKey(int programIndex) const;

    //! @briefprivate
    void UpdateProgram(int programIndex);

    //@}

    //----------------------------------------
    //! @name デバッグ用
    //@{

    //! @brief キーを文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    static int PrintKeyTo(char* pStr, int strLength, const bit32* pKey, int keyLength);

    //! @brief キーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintStaticOptionTo(char* pStr, int strLength, const bit32* pKey) const;

    //! @brief キーのオプション表現を文字列に変換します。
    //!
    //! 戻り値は終端文字を含まない文字列長です。pStr が NULL でない場合は必ず終端されます。
    //! strLength が必要なサイズに満たない場合は戻り値は負になります。
    //! pStr を NULL、strLength を 0 にすることで終端文字を含まない文字列長を返します。
    //!
    int PrintDynamicOptionTo(char* pStr, int strLength, const bit32* pKey) const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダアーカイブの構造体です。
struct ResShaderArchiveData
{
    BinaryFileHeader fileHeader;
    u32 alignment;
    BinString ofsName; //!< バイナリ変換時に指定されたファイル名（拡張子を除く）
    u32 sizeStringPool; // !< 文字列プールのバイトサイズ
    Offset ofsStringPool;

    BinString ofsPath;

    u16 numShadingModel;
    bit16 flag;

    u32 abiVersion; //!< shaderUtils の ABI バージョン

    Offset ofsShadingModelDic;

    BinPtr pUserPtr;

    // シェーダプログラムの更新が必要な場合に呼び出されるコールバックで、内部で使用します。
    BinPtr pCallback;
};

//! @brief シェーダアーカイブのリソースです。
class ResShaderArchive : private ResShaderArchiveData
{
public:
    //! @briefprivate
    typedef ResShaderArchiveData DataType;

    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'H', 'A') };

    //! @brief シェーダアーカイブに関するフラグです。
    enum Flag
    {
        //! @brief GL 版シェーダソースをもちます。
        GL_SOURCE = 0x01 << 0,

        //! @brief  GL 版シェーダバイナリをもちます。
        GL_BINARY = 0x01 << 1,

        //! @brief シェーダが圧縮されています。
        SHADER_COMPRESSED = 0x01 << 2,

        //! @brief GL 版バイナリが使用可能です。
        GL_BINARY_AVAILABLE = 0x01 << 3,

        //! @brief コンバート時にオプションが強制的にバリエーション化されています。
        FORCE_VARIATION = 0x01 << 4,

        //! @brief シェーダがオフライン DL 化されています。
        SHADER_OFFLINE_DL = 0x01 << 5
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief ロードしたファイルを ResShaderArchive に変換します。
    //!
    //! PC 版では内部でエンディアン変換が行われます。
    //!
    static ResShaderArchive* ResCast(void* ptr);

    //! @brief シェーダアーカイブをセットアップします。
    //!
    //! PC 版では PC 版シェーダである必要があります。
    //!
    void Setup();

    //! @brief シェーダアーカイブをクリーンアップします。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief クラス名を取得します。
    static const char* GetClassName() { return "ResShaderArchive"; }

    //! @brief データへの参照を取得します。
    DataType& ref() { NW_G3D_VALIDITY_ASSERT; return *ptr(); }

    //! @brief データへの参照を取得します。
    const DataType& ref() const { NW_G3D_VALIDITY_ASSERT; return *ptr(); }

    //! @brief データのポインタを取得します。
    DataType* ptr() { return this; }

    //! @brief データのポインタを取得します。
    const DataType* ptr() const { return this; }

    //! @brief バイナリファイルが有効かどうかを取得します。
    static bool IsValid(const void* ptr);

    //! @brief リソースのヘッダを取得します。
    const BinaryFileHeader* GetFileHeader() const { return &ref().fileHeader; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief GL版のシェーダかどうかを取得します。
    bool IsGLShader() const { return HasGLSource() || HasGLBinary(); }

    //! @brief GL版のソースをもつかどうかを取得します。
    bool HasGLSource() const { return ( ref().flag & GL_SOURCE ) != 0; }

    //! @brief GL版のバイナリをもつかどうかを取得します。
    bool HasGLBinary() const { return ( ref().flag & GL_BINARY ) != 0; }

    //! @brief シェーダが圧縮されているかどうかを取得します。
    bool IsShaderCompressed() const { return ( ref().flag & SHADER_COMPRESSED ) != 0; }

    //! @brief GL版のバイナリが使用可能かどうかを取得します。Setup() が呼び出されている必要があります。
    bool IsGLBinaryAvailable() const { return ( ref().flag & GL_BINARY_AVAILABLE ) != 0; }

    //! @briefprivate
    UpdateProgramCallback GetUpdateProgramCallback() const
    {
        void* ptr = const_cast<void*>(ref().pCallback.to_ptr());
        UpdateProgramCallback pCallback = reinterpret_cast<UpdateProgramCallback>(ptr); // 警告対策
        return pCallback;
    }

    //! @briefprivate
    void SetUpdateProgramCallback(UpdateProgramCallback pCallback)
    {
        void* ptr = reinterpret_cast<void*>(pCallback); // 警告対策
        ref().pCallback.set_ptr(ptr);
    }

    //@}

    //----------------------------------------
    //! @name シェーディングモデル
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShadingModel, ShadingModel, GetName())

    //@}

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ResShaderArchive);
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESTEXTURE_H_
