﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESMODEL_H_
#define NW_G3D_RES_RESMODEL_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResSkeleton.h>
#include <nw/g3d/res/g3d_ResShape.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/res/g3d_ResUserData.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

class BindCallback;
class ResFile;

//--------------------------------------------------------------------------------------------------

//! @brief モデルの構造体です。
struct ResModelData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    Offset ofsSkeleton;
    Offset ofsVertexArray;
    Offset ofsShapeDic;
    Offset ofsMaterialDic;
    Offset ofsUserDataDic;

    u16 numVertex;
    u16 numShape;
    u16 numMaterial;
    u16 numUserData;

    u32 totalProcessVertex;

    BinPtr pUserPtr;
};

//! @brief モデルのリソースです。
class ResModel : private ResModelData
{
    NW_G3D_RES_COMMON(ResModel);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'M', 'D', 'L') };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief モデルをセットアップします。
    void Setup();

    //! @brief モデルをクリーンアップします。
    void Cleanup();

    //! @brief モデルをリセットします。
    void Reset();

    //! @brief 外部参照されているリソースの参照解決を行います。
    BindResult Bind(const ResFile* pFile);

    //! @brief 外部参照されているリソースの参照解決を行います。
    BindResult Bind(const ResFile* pFile, BindCallback* pCallback);

    //! @brief 指定した名前のテクスチャの参照を差し替えます。
    bool ForceBind(const ResTexture* pTexture, const char* name);

    //! @brief リソースの参照設定を解除します。
    void Release();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)

    //! @brief 処理頂点数を取得します。
    u32 GetTotalProcessVertex() const { return ref().totalProcessVertex; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { ref().pUserPtr.set_ptr(pUserPtr); }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return ref().pUserPtr.to_ptr(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return ref().pUserPtr.to_ptr<T>(); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return ref().pUserPtr.to_ptr<T>(); }

    //@}

    //----------------------------------------
    //! @name スケルトン
    //@{

    NW_G3D_RES_FIELD_CLASS_DECL(ResSkeleton, Skeleton)

    //@}

    //----------------------------------------
    //! @name 頂点情報
    //@{

    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResVertex, Vertex, GetName())

    //@}

    //----------------------------------------
    //! @name シェイプ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResShape, Shape, GetName())

    //@}

    //----------------------------------------
    //! @name マテリアル
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResMaterial, Material, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_RES_RESMODEL_H_
