﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESDICTIONARY_H_
#define NW_G3D_RES_RESDICTIONARY_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>

namespace nw { namespace g3d { namespace res {

//! @brief Patricia Tree で実装された辞書の構造体です。
struct ResDicPatriciaData
{
    u32 size; //!< データブロックのサイズ
    s32 numData;

    //! @brief 辞書のノード情報です。
    struct Node
    {
        u32 refBit;
        u16 idxLeft;
        u16 idxRight;
        BinString ofsName;
        Offset ofsData;
    } node[1]; // 実際には (numData + 1) 個並んでいる。1個目はルートノードで空。
};

//! @brief Patricia Tree で実装された辞書です。
//!
//! 無効な辞書に対する名前引きは NULL を返しますが、
//! インデックス引きは未定義の動作となります。
//!
class ResDicPatricia : private ResDicPatriciaData
{
    NW_G3D_RES_COMMON(ResDicPatricia);

public:
    //! @briefprivate
    using ResDicPatriciaData::Node;

    //! @brief 辞書に含まれるデータ数を取得します。
    int GetDataCount() const
    {
        return ref().numData;
    }

    //! @brief インデクスからデータのポインタを取得します。
    void* Get(int idx) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(idx, ref().numData);
        // 辞書引き関連についてはconst correctnessを維持しない。
        return const_cast<void*>(ref().node[idx + 1].ofsData.to_ptr());
    }

    //! @brief 文字列で検索してポインタを取得します。
    //!
    //! 文字列の長さが分かっている場合は len を渡す方が高速です。
    //!
    void* Find(const char* str) const;

    //! @brief 文字列で検索してポインタを取得します。
    void* Find(const char* str, size_t len) const;

    //! @brief 文字列で検索してポインタを取得します。
    void* Find(const ResName* name) const;

    //! @brief 文字列で検索してインデクスを取得します。
    //!
    //! 文字列の長さが分かっている場合は len を渡す方が高速です。
    //!
    int FindIndex(const char* str) const;

    //! @brief 文字列で検索してインデクスを取得します。
    int FindIndex(const char* str, size_t len) const;

    //! @brief 文字列で検索してインデクスを取得します。
    int FindIndex(const ResName* name) const;

    //! @brief 文字列で検索してノードを取得します。
    Node* FindNode(const char* str, size_t len) const;

    //! @brief 文字列で検索してノードを取得します。
    Node* FindNode(const ResName* name) const;

    //! @brief インデクスから名前を取得します。
    //!
    //! インデクスは辞書が持つデータ数の範囲内とする必要があります。
    //!
    const ResName* GetResName(u32 idx) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(idx, ref().numData);
        return ref().node[idx + 1].ofsName.GetResName();
    }

    //! @brief インデクスから名前を取得します。
    //!
    //! インデクスは辞書が持つデータ数の範囲内とする必要があります。
    //!
    const char* GetName(int idx) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(idx, ref().numData);
        return ref().node[idx + 1].ofsName.to_ptr();
    }

    //! @brief 辞書の構築結果です。
    enum BuildResult
    {
        //! @brief 辞書の構築が成功したことを表します。
        SUCCESS = 0,

        //! @brief 辞書に名前が重複した要素が存在して構築できなかったことを表します。
        ERR_NAME_DUPLICATION
    };

    //! @brief 辞書を再構築します。
    BuildResult Build();
};

//! @brief 辞書の定義です。
typedef ResDicPatricia ResDicType;

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESDICTIONARY_H_
