﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESANIMCURVE_H_
#define NW_G3D_RES_RESANIMCURVE_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/res/g3d_ResCommon.h>

#include <limits>

namespace nw { namespace g3d { namespace res {

//! @brief アニメーションに関するフラグです。
struct AnimFlag
{
    enum
    {
        //! @brief カーブがベイク済みであることを表します。
        CURVE_BAKED     = 0x1 << 0,

        //! @brief ループ再生を表します。
        PLAYPOLICY_LOOP = 0x1 << 2,

        //! @brief 範囲外であることを表します。
        NOT_BOUND       = 0xFFFF
    };
};

//! @brief アニメーションフレームのキャッシュ構造体です。
struct AnimFrameCache
{
    float start;
    float end;
    int keyIndex;
};

//! @brief コンスタントカーブの構造体です。
struct ResAnimConstantData
{
    u32 targetOffset;
    union
    {
        f32 fValue;
        s32 iValue;
    };
};

//! @brief コンスタントカーブのリソースです。
class ResAnimConstant : private ResAnimConstantData
{
    NW_G3D_RES_COMMON(ResAnimConstant);

public:
    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief Float として値を取得します。
    float GetFloat() const { return ref().fValue; }

    //! @brief Int として値を取得します。
    int GetInt() const { return ref().iValue; }

    //@}
};

//! @brief アニメーションカーブの構造体です。
struct ResAnimCurveData
{
    bit16 flag;
    u16 numKey;
    u32 targetOffset; //!< ターゲット構造体内におけるバイトオフセット
    f32 startFrame;
    f32 endFrame;
    union
    {
        f32 fScale; //!< float の結果に対するスケール
        s32 iScale; // 未使用
    };
    union
    {
        f32 fOffset; //!< float の結果に対するオフセット
        s32 iOffset; //!< int の結果に対するオフセット
    };
    f32 fDelta;
    Offset ofsFrameArray;
    Offset ofsKeyArray;
};

//! @brief アニメーションカーブのリソースです。
class ResAnimCurve : private ResAnimCurveData
{
    NW_G3D_RES_COMMON(ResAnimCurve);

public:
    //! @brief アニメーションカーブに関するフラグです。
    enum Flag
    {
        // FrameQuantization
        //! @briefprivate
        FRAME_SHIFT         = 0,

        //! @brief F32 浮動小数点数のフレームです。
        FRAME32             = 0x0 << FRAME_SHIFT,

        //! @brief S10.5 固定小数点数のフレームです。
        FRAME16             = 0x1 << FRAME_SHIFT,

        //! @brief U8 整数のフレームです。
        FRAME8              = 0x2 << FRAME_SHIFT,

        //! @briefprivate
        FRAME_MASK          = 0x3 << FRAME_SHIFT,

        //! @briefprivate
        NUM_FRAME           = 3,

        // KeyQuantization
        //! @briefprivate
        KEY_SHIFT           = 2,

        //! @brief F32 浮動小数点数 * fScale + fOffset のキーです。
        KEY32               = 0x0 << KEY_SHIFT,

        //! @brief S16 整数 * fScale + fOffset のキーです。
        KEY16               = 0x1 << KEY_SHIFT,

        //! @brief S8 整数 * fScale + fOffset のキーです。
        KEY8                = 0x2 << KEY_SHIFT,

        //! @briefprivate
        KEY_MASK            = 0x3 << KEY_SHIFT,

        //! @briefprivate
        NUM_KEY             = 3,

        // CureveType
        //! @briefprivate
        CURVE_SHIFT         = 4,

        // float
        //! @brief 3次多項式のカーブです。
        CURVE_CUBIC         = 0x0 << CURVE_SHIFT,

        //! @brief 1次多項式のカーブです。
        CURVE_LINEAR        = 0x1 << CURVE_SHIFT,

        //! @brief コマ形式のカーブです。
        CURVE_BAKED_FLOAT   = 0x2 << CURVE_SHIFT,

        // int
        //! @brief ステップ形式のカーブです。
        CURVE_STEP_INT      = 0x4 << CURVE_SHIFT,

        //! @brief コマ形式のカーブです。
        CURVE_BAKED_INT     = 0x5 << CURVE_SHIFT,

        //! @brief ステップ形式のカーブです。
        CURVE_STEP_BOOL     = 0x6 << CURVE_SHIFT,

        //! @brief コマ形式のカーブです。
        CURVE_BAKED_BOOL    = 0x7 << CURVE_SHIFT,

        //! @briefprivate
        CURVE_INT_BASE      = CURVE_STEP_INT,

        //! @briefprivate
        CURVE_MASK          = 0x7 << CURVE_SHIFT
    };

    //! @brief 繰り返し方法です。
    enum WrapMode
    {
        //! @brief 最初または最後のフレームにクランプします。
        WRAP_CLAMP,

        //! @brief カーブを繰り返します。
        WRAP_REPEAT,

        //! @brief カーブを反転しつつ繰り返します。
        WRAP_MIRROR,

        //! @brief 端を起点として相対的にカーブを繰り返します。
        WRAP_RELATIVE_REPEAT,

        //! @briefprivate
        WRAP_MASK = 0x3,

        //! @briefprivate
        WRAP_PRE_SHIFT = 8,

        //! @briefprivate
        WRAP_POST_SHIFT = 12,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //!@ brief カーブをコマ化します。
    void Bake(void* pBuffer, size_t bufferSize) { IsFloatCurve() ? BakeFloat(pBuffer, bufferSize) : BakeInt(pBuffer, bufferSize); }

    //! @brief Float カーブをコマ化します。
    void BakeFloat(void* pBuffer, size_t bufferSize);

    //! @brief Int カーブをコマ化します。
    void BakeInt(void* pBuffer, size_t bufferSize);

    //! @brief カーブをコマ化するのに必要なバッファサイズを返します。
    size_t CalcBakedSize() const { return IsFloatCurve() ? CalcBakedFloatSize() : CalcBakedIntSize(); }

    //! @brief Float カーブをコマ化するのに必要なバッファサイズを返します。
    size_t CalcBakedFloatSize() const;

    //! @brief Int カーブをコマ化するのに必要なバッファサイズを返します。
    size_t CalcBakedIntSize() const;

    //! @brief コマ化したアニメーションをカーブに戻します。
    void Reset() { IsFloatCurve() ? ResetFloat() : ResetInt(); }

    //! @brief コマ化したアニメーションを Float カーブに戻します。
    void ResetFloat();

    //! @brief コマ化したアニメーションを Int カーブに戻します。
    void ResetInt();

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief Float カーブを評価します。
    float EvalFloat(float frame) const
    {
        AnimFrameCache frameCache;
        frameCache.start = std::numeric_limits<float>::infinity();
        return EvalFloat(frame, &frameCache);
    }

    //! @brief EvalFloat() の別名関数です。
    float EvaluateFloat(float frame) const
    {
        return EvalFloat(frame);
    }

    //! @brief Int カーブを評価します。
    int EvalInt(float frame) const
    {
        AnimFrameCache frameCache;
        frameCache.start = std::numeric_limits<float>::infinity();
        return EvalInt(frame, &frameCache);
    }

    //! @brief EvalInt() の別名関数です。
    int EvaluateInt(float frame) const
    {
        return EvalInt(frame);
    }

    //! @brief Float カーブを評価します。
    float EvalFloat(float frame, AnimFrameCache* pFrameCache) const;

    //! @brief EvalFloat() の別名関数です。
    float EvaluateFloat(float frame, AnimFrameCache* pFrameCache) const
    {
        return EvaluateFloat(frame, pFrameCache);
    }

    //! @brief Int カーブを評価します。
    int EvalInt(float frame, AnimFrameCache* pFrameCache) const;

    //! @brief EvalInt() の別名関数です。
    int EvaluateInt(float frame, AnimFrameCache* pFrameCache) const
    {
        return EvalInt(frame, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief Float カーブかどうかを取得します。
    bool IsFloatCurve() const { return (ref().flag & CURVE_MASK) < CURVE_INT_BASE; }

    //! @brief Int カーブかどうかを取得します。
    bool IsIntCurve() const { return (ref().flag & CURVE_MASK) >= CURVE_INT_BASE; }

    //! @brief 最初のキーよりも前のフレームで評価した際の動作を取得します。
    WrapMode GetPreWrapMode() const { return WrapMode((ref().flag >> WRAP_PRE_SHIFT) & WRAP_MASK);}

    //! @brief 最後のキーよりも後のフレームで評価した際の動作を取得します。
    WrapMode GetPostWrapMode() const { return WrapMode((ref().flag >> WRAP_POST_SHIFT) & WRAP_MASK);}

    //! @brief 最初のフレームを取得します。
    float GetStartFrame() const { return ref().startFrame; }

    //! @brief 最後のフレームを取得します。
    float GetEndFrame() const { return ref().endFrame; }

    //@}
protected:
    class Impl;

    //! @briefprivate
    bit32 GetFrameType() const { return ref().flag & FRAME_MASK; }

    //! @briefprivate
    bit32 GetKeyType() const { return ref().flag & KEY_MASK; }

    //! @briefprivate
    bit32 GetCurveType() const { return ref().flag & CURVE_MASK; }

    //! @briefprivate
    float WrapFrame(float *pOutOffset, float frame) const;

    //! @briefprivate
    void UpdateFrameCache(AnimFrameCache* pFrameCache, float frame) const;

    //! @briefprivate
    template <typename T>
    void FindFrame(AnimFrameCache *pFrameCache, float frame) const;

    //! @briefprivate
    template <typename T>
    float EvalCubic(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    template <typename T>
    float EvalLinear(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    template <typename T>
    float EvalBakedFloat(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    template <typename T>
    int EvalStepInt(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    template <typename T>
    int EvalBakedInt(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    int EvalStepBool(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    int EvalBakedBool(float frame, AnimFrameCache* pFrameCache) const;

    //! @briefprivate
    template <typename T>
    void BakeImpl(void* pBuffer, float start, int numKey);
};

//--------------------------------------------------------------------------------------------------

//! @brief S10.5 固定小数点数を F32 浮動小数点数に変換します。
NW_G3D_FORCE_INLINE
float CastS10_5ToF32(s16 value)
{
    return FastCast<float>(value) * (1.0f / 32.0f);
}

//! @brief F32 浮動小数点数を S10.5 固定小数点数に変換します。
NW_G3D_FORCE_INLINE
s16 CastF32ToS10_5(float value)
{
    return Math::Floor<s16>(value * 32.0f);
}

//! @brief 三次方程式を計算します。
NW_G3D_FORCE_INLINE
float CalcCubic(float t, float c0, float c1, float c2, float c3)
{
    // 精度低下を防ぐため t 同士をかけないよう注意。
    return ((c3 * t + c2) * t) * t + (c1 * t + c0);
}

//! @brief 線形方程式を計算します。
NW_G3D_FORCE_INLINE
float CalcLinear(float t, float c0, float c1)
{
    return c1 * t + c0;
}

//--------------------------------------------------------------------------------------------------

//! @brief Frame の計算に用いる構造体です。
template <typename T>
struct Frame
{
    T frame;

    //! @brief フレームを float で取得します。
    float Get() const { return FastCast<float>(frame); }

    //! @brief フレーム区間の探索用にフレームを量子化します。
    //!
    //! フレーム区間の探索に使用するので、Round ではなく Floor で変換を行います。
    //!
    static T Quantize(float frame) { return Math::Floor<T>(frame); }
};

//! @brief float の Frame の計算に用いる構造体です。
template <>
struct Frame<float>
{
    float frame;

    //! @brief フレームを float で取得します。
    float Get() const { return frame; }

    //! @brief フレームを量子化します。
    static float Quantize(float frame) { return frame; }
};

//! @brief s16 の Frame の計算に用いる構造体です。
template <>
struct Frame<s16>
{
    s16 frame;

    //! @brief フレームを float で取得します。
    float Get() const { return CastS10_5ToF32(frame); }

    //! @brief フレームを量子化します。
    static s16 Quantize(float frame) { return CastF32ToS10_5(frame); }
};

//--------------------------------------------------------------------------------------------------

//! @brief キュービックキーの計算に用いる構造体です。
template <typename T>
struct ResCubicKey
{
    T coef[4];

    //! @brief 計算結果を取得します。
    float Get(float ratio) const
    {
        return CalcCubic(ratio, static_cast<float>(coef[0]), static_cast<float>(coef[1]),
            static_cast<float>(coef[2]), static_cast<float>(coef[3]));
    }
};

//! @brief リニアキーの計算に用いる構造体です。
template <typename T>
struct ResLinearKey
{
    T coef[2];

    //! @brief 計算結果を取得します。
    float Get(float ratio) const
    {
        return CalcLinear(ratio, static_cast<float>(coef[0]), static_cast<float>(coef[1]));
    }
};

//! @brief Float キーの計算に用いる構造体です。
template <typename T>
struct ResFloatKey
{
    T value;

    //! @brief 計算結果を取得します。
    float Get() const { return StaticCast<float>(value); }
};

//! @brief Int キーの計算に用いる構造体です。
template <typename T>
struct ResIntKey
{
    T value;

    //! @brief 計算結果を取得します。
    int Get() const { return static_cast<int>(value); }
};

#if defined( __ghs__ )

template <>
NW_G3D_FORCE_INLINE
float ResCubicKey<float>::Get(float param) const
{
    f32x2 term23 = __PSQ_LX(coef, 8, 0, 0);
    f32x2 term01 = __PSQ_LX(coef, 0, 0, 0);
    term23 = __PS_MULS0F(term23, param);
    term23 = __PS_MULS0F(term23, param);
    f32x2 one_t = { 1.0f, param };
    term01 = __PS_MUL(term01, one_t);
    term23 = __PS_MUL(term23, one_t);
    f32x2 sum = __PS_ADD(term01, term23);
    sum = __PS_SUM0(sum, sum, sum);
    return sum[0];
};

template <>
NW_G3D_FORCE_INLINE
float ResCubicKey<s16>::Get(float param) const
{
    f32x2 term23 = __PSQ_LX(coef, 4, 0, OS_FASTCAST_S16);
    f32x2 term01 = __PSQ_LX(coef, 0, 0, OS_FASTCAST_S16);
    term23 = __PS_MULS0F(term23, param);
    term23 = __PS_MULS0F(term23, param);
    f32x2 one_t = { 1.0f, param };
    term01 = __PS_MUL(term01, one_t);
    term23 = __PS_MUL(term23, one_t);
    f32x2 sum = __PS_ADD(term01, term23);
    sum = __PS_SUM0(sum, sum, sum);
    return sum[0];
};

template <>
NW_G3D_FORCE_INLINE
float ResCubicKey<s8>::Get(float param) const
{
    f32x2 term23 = __PSQ_LX(coef, 2, 0, OS_FASTCAST_S8);
    f32x2 term01 = __PSQ_LX(coef, 0, 0, OS_FASTCAST_S8);
    term23 = __PS_MULS0F(term23, param);
    term23 = __PS_MULS0F(term23, param);
    f32x2 one_t = { 1.0f, param };
    term01 = __PS_MUL(term01, one_t);
    term23 = __PS_MUL(term23, one_t);
    f32x2 sum = __PS_ADD(term01, term23);
    sum = __PS_SUM0(sum, sum, sum);
    return sum[0];
};

template <>
NW_G3D_FORCE_INLINE
float ResLinearKey<float>::Get(float param) const
{
    f32x2 coef01 = __PSQ_L(coef, 0, 0);
    return coef01[0] + coef01[1] * param;
};

template <>
NW_G3D_FORCE_INLINE
float ResLinearKey<s16>::Get(float param) const
{
    f32x2 coef01 = __PSQ_L(coef, 0, OS_FASTCAST_S16);
    return coef01[0] + coef01[1] * param;
};

template <>
NW_G3D_FORCE_INLINE
float ResLinearKey<s8>::Get(float param) const
{
    f32x2 coef01 = __PSQ_L(coef, 0, OS_FASTCAST_S8);
    return coef01[0] + coef01[1] * param;
};

#endif

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESANIMCURVE_H_
