﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_VISIBILITYANIMOBJ_H_
#define NW_G3D_VISIBILITYANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResVisibilityAnim.h>

namespace nw { namespace g3d {

class ModelObj;

//! @brief ビジビリティアニメーションインスタンスです。
class VisibilityAnimObj : public ModelAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    VisibilityAnimObj() : ModelAnimObj(), m_pRes() {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    virtual BindResult Bind(const ResModel* pModel);
    virtual BindResult Bind(const ModelObj* pModel);
    virtual void BindFast(const ResModel* pModel);

    //! @brief 指定したインデックスのターゲットにフラグを設定します。
    void SetBindFlag(int targetIndex, BindFlag flag)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        SetBindFlagImpl(targetIndex, flag);
    }

    //! @brief 指定したインデックスのターゲットのフラグを取得します。
    BindFlag GetBindFlag(int targetIndex)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return GetBindFlagImpl(targetIndex);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    virtual void ApplyTo(ModelObj* pModelObj) const;
    //! @brief Calc()の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResVisibilityAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するビジビリティアニメーションを差し替えます。
    //!
    void SetResource(ResVisibilityAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource()
    {
        m_pRes = NULL;
        SetTargetUnbound();
    }

    //! @brief アニメーション対象に対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResModel* pModel) const
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        if (m_pRes->GetVisibilityType() == ResVisibilityAnim::BONE_VISIBILITY)
        {
            return pModel->GetSkeleton()->GetBoneCount() <= GetBindTable().GetSize();
        }
        else
        {
            return pModel->GetMaterialCount() <= GetBindTable().GetSize();
        }
    }

    //! @brief アニメーションリソースに対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResVisibilityAnim* pRes) const
    {
        NW_G3D_ASSERT_NOT_NULL(pRes);
        if (pRes->GetVisibilityType() == ResVisibilityAnim::BONE_VISIBILITY)
        {
            return pRes->GetAnimCount() <= m_MaxBoneAnim;
        }
        else
        {
            return pRes->GetAnimCount() <= m_MaxMatAnim;
        }
    }

    //@}

protected:
    //! @briefprivate カーブを取得します。
    //!
    //!
    const ResAnimCurve* GetCurve(int curveIndex) const
    {
        return ResAnimCurve::ResCast(&m_pCurveArray[curveIndex]);
    }

private:
    class Sizer;
    ResVisibilityAnim* m_pRes;
    u16 m_MaxBoneAnim;
    u16 m_MaxMatAnim;
    s32 m_NumCurve;
    ResAnimCurveData* m_pCurveArray;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(VisibilityAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate VisibilityAnimObj のサイズを計算するためのクラスです。
//!
//!
class VisibilityAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief VisibilityAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        BIND_TABLE,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief VisibilityAnimObj::Init() に渡して初期化を行うパラメータです。
class VisibilityAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResVisibilityAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxBoneCount()、SetMaxMatCount()、SetMaxBoneAnimCount()、
    //! SetMaxMatAnimCount(), SetMaxCurveCount() でバッファサイズの計算に必要な
    //! パラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumBone = m_NumMat = m_NumBoneAnim = m_NumMatAnim = m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    //!
    //! ボーンビジビリティまたはマテリアルビジビリティいずれかのみの最大数を設定する場合、
    //! SetTargetCount() で正確な値を指定することができます。
    //!
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        m_NumBone = std::max(m_NumBone, pResModel->GetSkeleton()->GetBoneCount());
        m_NumMat = std::max(m_NumMat, pResModel->GetMaterialCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResVisibilityAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResVisibilityAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        if (pResAnim->GetVisibilityType() == ResVisibilityAnim::BONE_VISIBILITY)
        {
            m_NumBoneAnim = std::max(m_NumBoneAnim, pResAnim->GetAnimCount());
        }
        else
        {
            m_NumMatAnim = std::max(m_NumMatAnim, pResAnim->GetAnimCount());
        }
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のボーン数を指定します。
    void SetMaxBoneCount(int boneCount)
    {
        NW_G3D_ASSERT(boneCount >= 0);
        m_NumBone = boneCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のボーン数を取得します。
    int GetMaxBoneCount() const { return m_NumBone; }

    //! @brief 使用できる最大のマテリアル数を指定します。
    void SetMaxMatCount(int matCount)
    {
        NW_G3D_ASSERT(matCount >= 0);
        m_NumMat = matCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアル数を取得します。
    int GetMaxMatCount() const { return m_NumMat; }

    //! @brief 使用できる最大のボーンアニメーション数を指定します。
    void SetMaxBoneAnimCount(int boneAnimCount)
    {
        NW_G3D_ASSERT(boneAnimCount >= 0);
        m_NumBoneAnim = boneAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のボーンアニメーション数を取得します。
    int GetMaxBoneAnimCount() const { return m_NumBoneAnim; }

    //! @brief 使用できる最大のマテリアルアニメーション数を指定します。
    void SetMaxMatAnimCount(int matAnimCount)
    {
        NW_G3D_ASSERT(matAnimCount >= 0);
        m_NumMatAnim = matAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアルアニメーション数を取得します。
    int GetMaxMatAnimCount() const { return m_NumMatAnim; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const
    {
        return ((m_NumBone >= 0 && m_NumBoneAnim >= 0) ||
            (m_NumMat >= 0 && m_NumMatAnim >= 0)) && m_NumCurve >= 0;
    }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumBone;
    int m_NumMat;
    int m_NumBoneAnim;
    int m_NumMatAnim;
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief VisibilityAnimObj の構築を行うクラスです。
class VisibilityAnimObj::Builder : public VisibilityAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResVisibilityAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxBoneCount()、SetMaxMatCount()、SetMaxBoneAnimCount()、
    //! SetMaxMatAnimCount(), SetMaxCurveCount() でバッファサイズの計算に必要な
    //! パラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief VisibilityAnimObj を構築します。
    //!
    //! @param[in] pVisibilityAnimObj VisibilityAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(VisibilityAnimObj* pVisibilityAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pVisibilityAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief VisibilityAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief VisibilityAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  VisibilityAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief VisibilityAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_VISIBILITYANIMOBJ_H_
