﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SIZER_H_
#define NW_G3D_SIZER_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/ut/g3d_Inlines.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d {

//! @briefprivate サイズ計算ユーティリティです。
//!
//! インスタンスクラスのバッファサイズを計算するためにライブラリ内部で使用されます。
class Sizer
{
public:
    //! @briefprivate
    struct Chunk
    {
        size_t size;
        ptrdiff_t offset;
    };

    //! @brief コンストラクタです。
    Sizer() : pChunk(NULL), totalSize(0) {}

    //! @brief 合計サイズを取得します。
    size_t GetTotalSize() const { return totalSize; }

    //! @brief 有効であるかどうかを取得します。
    bool IsValid() const { return pChunk != NULL; }

    //! @brief サイズ計算を無効化します。
    void Invalidate() { pChunk = NULL; totalSize = 0; }

    //! @brief 指定したポインタにチャンクのサイズを足してポインタを取得します。
    template <typename T>
    NW_G3D_FORCE_INLINE T* GetPtr(void* ptr, int type) const
    {
        NW_G3D_ASSERT_NOT_NULL(pChunk);
        return pChunk[type].size ? AddOffset<T>(ptr, pChunk[type].offset) : NULL;
    }

    //! @brief 指定したポインタにチャンクのサイズを足してポインタを取得します。
    NW_G3D_FORCE_INLINE void* GetPtr(void* ptr, int type) const
    {
        NW_G3D_ASSERT_NOT_NULL(pChunk);
        return pChunk[type].size ? AddOffset(ptr, pChunk[type].offset) : NULL;
    }

protected:
    //! @briefprivate
    void CalcOffset(Chunk* pChunk, int count)
    {
        int idx = 0;
        pChunk[idx].offset = 0;
        for (; idx < count - 1; ++idx)
        {
            NW_G3D_ASSERT(IsAligned(pChunk[idx].size));
            pChunk[idx + 1].offset = pChunk[idx].offset + pChunk[idx].size;
        }
        totalSize = pChunk[idx].offset + pChunk[idx].size;
        this->pChunk = pChunk;
    }

private:
    Chunk* pChunk;
    size_t totalSize;
};

}} // namespace nw::g3d

NW_G3D_PRAGMA_POP_WARNINGS

#endif // NW_G3D_SIZER_H_
