﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SHAPEANIMOBJ_H_
#define NW_G3D_SHAPEANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResShapeAnim.h>

namespace nw { namespace g3d {

class ModelObj;
class ShapeObj;

//! @brief シェイプアニメーションインスタンスです。
class ShapeAnimObj : public ModelAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ShapeAnimObj() : ModelAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    virtual BindResult Bind(const ResModel* pModel);
    virtual BindResult Bind(const ModelObj* pModel);
    virtual void BindFast(const ResModel* pModel);

    //! @brief 指定したインデックスのシェイプにフラグを設定します。
    void SetBindFlag(int shapeIndex, BindFlag flag)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        SetBindFlagImpl(shapeIndex, flag);
    }

    //! @brief 指定したインデックスのシェイプのフラグを取得します。
    BindFlag GetBindFlag(int shapeIndex)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return GetBindFlagImpl(shapeIndex);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    virtual void ApplyTo(ModelObj* pModelObj) const;
    //! @brief Calc()の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResShapeAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するシェイプアニメーションを差し替えます。
    //!
    void SetResource(ResShapeAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource()
    {
        m_pRes = NULL;
        SetTargetUnbound();
    }

    //! @brief アニメーション対象に対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResModel* pModel) const
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return pModel->GetShapeCount() <= GetBindTable().GetSize();
    }

    //! @brief アニメーションリソースに対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResShapeAnim* pRes) const
    {
        NW_G3D_ASSERT_NOT_NULL(pRes);
        return pRes->GetVertexShapeAnimCount() <= m_MaxVtxShpAnim &&
            pRes->GetKeyShapeAnimCount() <= m_MaxSubBind;
    }

    //! @brief 指定したシェイプに適用されるアニメーションの対象キーシェイプ数を取得します。
    int GetKeyShapeCount(int shapeIndex) const
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(shapeIndex, GetBindTable().GetTargetCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        int idxAnim = GetBindTable().GetAnimIndex(shapeIndex);
        if (idxAnim != AnimBindTable::NOT_BOUND && GetBindTable().IsApplyEnabled(idxAnim))
        {
            return m_pRes->GetVertexShapeAnim(idxAnim)->GetKeyShapeAnimCount();
        }
        return 0;
    }

    //! @brief 指定したシェイプに適用されるアニメーションの対象キーシェイプインデックスを取得します。
    s8* GetKeyShapeIndexArray(int shapeIndex) const
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(shapeIndex, GetBindTable().GetTargetCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        int idxAnim = GetBindTable().GetAnimIndex(shapeIndex);
        if (idxAnim != AnimBindTable::NOT_BOUND && GetBindTable().IsApplyEnabled(idxAnim))
        {
            s32 beginKeyShapeAnim = m_pRes->GetVertexShapeAnim(idxAnim)->ref().beginKeyShapeAnim;
            return &m_pSubBindIndexArray[beginKeyShapeAnim];
        }
        return NULL;
    }

    //@}

protected:

    //! @briefprivate 頂点アニメーション取得します。
    //!
    //!
    ResVertexShapeAnim* GetVertexShapeAnim(int animIndex)
    {
        return ResVertexShapeAnim::ResCast(&m_pVertexShapeAnimArray[animIndex]);
    }

    //! @briefprivate 頂点アニメーション取得します。
    //!
    const ResVertexShapeAnim* GetVertexShapeAnim(int animIndex) const
    {
        return ResVertexShapeAnim::ResCast(&m_pVertexShapeAnimArray[animIndex]);
    }

    //! @briefprivate シェイプごとに関連付けを行う内部関数です。
    //!
    BindResult SubBind(const ResVertexShapeAnim* pVertexShapeAnim, const ResShape* pShape);

    //! @briefprivate シェイプごとに関連付けを行う内部関数です。
    //!
    BindResult SubBindFast(const ResVertexShapeAnim* pVertexShapeAnim);

    //! @briefprivate シェイプごとに適用を行う内部関数です。
    //!
    void ApplyTo(ShapeObj* pShapeObj, int animIndex) const;

private:
    class Sizer;
    ResShapeAnim* m_pRes;
    ResVertexShapeAnimData* m_pVertexShapeAnimArray;
    s32 m_MaxVtxShpAnim;
    s32 m_MaxSubBind;
    s8* m_pSubBindIndexArray;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShapeAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ShapeAnimObj のサイズを計算するためのクラスです。
//!
class ShapeAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ShapeAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        BIND_TABLE,
        SUBBIND_TABLE,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief ShapeAnimObj::Init() に渡して初期化を行うパラメータです。
class ShapeAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResShapeAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxShapeCount()、SetMaxVertexShapeAnimCount()、
    //! SetMaxKeyShapeAnimCount()、SetMaxCurveCount() でバッファサイズの計算に必要な
    //! パラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumShape = m_NumVertexShapeAnim = m_NumKeyShapeAnim = m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        m_NumShape = std::max(m_NumShape, pResModel->GetShapeCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResShapeAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResShapeAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumVertexShapeAnim = std::max(m_NumVertexShapeAnim, pResAnim->GetVertexShapeAnimCount());
        m_NumKeyShapeAnim = std::max(m_NumKeyShapeAnim, pResAnim->GetKeyShapeAnimCount());
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のシェイプ数を指定します。
    void SetMaxShapeCount(int shapeCount)
    {
        NW_G3D_ASSERT(shapeCount >= 0);
        m_NumShape = shapeCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のシェイプ数を取得します。
    int GetMaxShapeCount() const { return m_NumShape; }

    //! @brief 使用できる最大のシェイプアニメーション数を指定します。
    void SetMaxVertexShapeAnimCount(int vertexShapeAnimCount)
    {
        NW_G3D_ASSERT(vertexShapeAnimCount >= 0);
        m_NumVertexShapeAnim = vertexShapeAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のシェイプアニメーション数を取得します。
    int GetMaxVertexShapeAnimCount() const { return m_NumVertexShapeAnim; }

    //! @brief 使用できる最大のキーシェイプアニメーション数を指定します。
    void SetMaxKeyShapeAnimCount(int keyShapeAnimCount)
    {
        NW_G3D_ASSERT(keyShapeAnimCount >= 0);
        m_NumKeyShapeAnim = keyShapeAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のキーシェイプアニメーション数を取得します。
    int GetMaxKeyShapeAnimCount() const { return m_NumKeyShapeAnim; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const
    {
        return m_NumShape >= 0 && m_NumVertexShapeAnim >= 0 &&
            m_NumKeyShapeAnim >= 0 && m_NumCurve >= 0;
    }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

    // 以下の関数は削除予定。

    //! @brief コンストラクタです。
    //!
    //! @param[in]      shapeCount      アニメーション対象モデルのシェイプ数です。
    //!                                 ResModel::GetShapeCount() で取得できます。
    //! @param[in]      vertexShapeAnimCount アニメーションが適用されるシェイプの数です。
    //!                                 ResShapeAnim::GetVertexShapeAnimCount() で取得できます。
    //! @param[in]      keyShapeAnimCount アニメーションが適用される頂点属性の数です。
    //!                                 ResShapeAnim::GetKeyShapeAnimCount() で取得できます。
    //!
    InitArg(int shapeCount, int vertexShapeAnimCount, int keyShapeAnimCount)
        : m_NumShape(shapeCount)
        , m_NumVertexShapeAnim(vertexShapeAnimCount)
        , m_NumKeyShapeAnim(keyShapeAnimCount)
        , m_NumCurve(0)
        , m_ContextEnabled(false)
        , m_ContextAvailable(true)
    {
        NW_G3D_ASSERT(shapeCount >= 0);
        NW_G3D_ASSERT(vertexShapeAnimCount >= 0);
        NW_G3D_ASSERT(keyShapeAnimCount >= 0);
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    //!
    //! @param[in]      curveCount      アニメーションに含まれるカーブの数です。
    //!                                 ResShapeAnim::GetCurveCount() で取得できます。
    //!
    void EnableContext(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextEnabled = true;
        m_Sizer.Invalidate();
    }

private:
    int m_NumShape;
    int m_NumVertexShapeAnim;
    int m_NumKeyShapeAnim;
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief ShapeAnimObj の構築を行うクラスです。
class ShapeAnimObj::Builder : public ShapeAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResShapeAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxShapeCount()、SetMaxVertexShapeAnimCount()、
    //! SetMaxKeyShapeAnimCount()、SetMaxCurveCount() でバッファサイズの計算に必要な
    //! パラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ShapeAnimObj を構築します。
    //!
    //! @param[in] pShapeAnimObj ShapeAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ShapeAnimObj* pShapeAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pShapeAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ShapeAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ShapeAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ShapeAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief SkeletalAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_SHAPEANIMOBJ_H_
