﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_ANIMOBJ_H_
#define NW_G3D_ANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/g3d_Sizer.h>
#include <nw/g3d/ut/g3d_Flag.h>

#include <limits>

namespace nw { namespace g3d {

namespace res {

class ResModel;

} // namespace res;

class ModelObj;

//--------------------------------------------------------------------------------------------------

//! @brief アニメーションのフレームを制御するクラスです。
class AnimFrameCtrl
{
public:
    //! @brief 入力フレームを再生ポリシーに応じて加工する関数の型です。
    typedef float (*PlayPolicy)(
        float inputFrame, float startFrame, float endFrame, void* pUserPtr);

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    AnimFrameCtrl();

    //! @brief 初期化を行います。
    void Init(float startFrame, float endFrame, PlayPolicy pPlayPolicy);

    //! @brief Init() の別名関数です。
    void Initialize(float startFrame, float endFrame, PlayPolicy pPlayPolicy)
    {
        Init(startFrame, endFrame, pPlayPolicy);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief フレーム更新幅に応じて現在のフレーム位置を更新します。
    void UpdateFrame()
    {
        m_Frame = m_pPlayPolicy(m_Frame + m_Step, m_StartFrame, m_EndFrame, m_pUserPtr);
    }

    //@}

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //! @brief 現在のフレーム位置を設定します。
    void SetFrame(float frame)
    {
        m_Frame = m_pPlayPolicy(frame, m_StartFrame, m_EndFrame, m_pUserPtr);
    }

    //! @brief 現在のフレーム位置を取得します。
    float GetFrame() const { return m_Frame; }

    //! @brief 開始フレームと終了フレームを設定します。
    void SetFrameRange(float start, float end)
    {
        NW_G3D_ASSERT(end - start >= 0.0f);
        m_StartFrame = start;
        m_EndFrame = end;
    }

    //! @brief 開始フレームを取得します。
    float GetStartFrame() const { return m_StartFrame; }

    //! @brief 終了フレームを取得します。
    float GetEndFrame() const { return m_EndFrame; }

    //! @brief フレーム更新幅を設定します。
    void SetStep(float step) { m_Step = step; }

    //! @brief フレーム更新幅を取得します。
    float GetStep() const { return m_Step; }

    //! @brief アニメーション再生ポリシーを設定します。
    void SetPlayPolicy(PlayPolicy policy) { m_pPlayPolicy = policy; }

    //! @brief アニメーション再生ポリシーを取得します。
    PlayPolicy GetPlayPolicy() const { return m_pPlayPolicy; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}

    static const float INVALID_FRAME;

private:
    float m_Frame;
    float m_StartFrame;
    float m_EndFrame;
    float m_Step;
    PlayPolicy m_pPlayPolicy;
    void* m_pUserPtr;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(AnimFrameCtrl);
};

//! @brief ワンタイム再生用の標準アニメーションポリシー関数です。
float PlayPolicy_Onetime(float inputFrame, float startFrame, float endFrame, void* pUserData);

//! @brief ループ再生用の標準アニメーションポリシー関数です。
float PlayPolicy_Loop(float inputFrame, float startFrame, float endFrame, void* pUserData);

//--------------------------------------------------------------------------------------------------

//! @brief アニメーションの関連付け状態を管理するクラスです。
class AnimBindTable
{
public:
    //! @briefprivate 関連付けに関するフラグです。
    //!
    enum BindFlag
    {
        //! @brief 有効です。
        ENABLED             = 0,

        //! @brief 更新処理を無視するが、最後の結果が Apply されます。
        SKIP_CALC           = 0x1 << 0,

        //! @brief Apply を無視するので、単独では無駄な更新が行われます。
        SKIP_APPLY          = 0x1 << 1,

        //! @brief Calc も Apply もスキップします。
        DISABLED            = SKIP_CALC | SKIP_APPLY
    };

    //! @briefprivate バインド状態の管理に関するフラグです。
    //!
    enum Flag
    {
        NOT_BOUND           = 0x7FFF,   // [0 .. 0x7FFE] が有効なインデックス
        INDEX_MASK          = 0x7FFF,   // Anim から Target へのバインド
        REVERSE_SHIFT       = 15,
        REVERSE_NOT_BOUND   = NOT_BOUND     << REVERSE_SHIFT,
        REVERSE_INDEX_MASK  = INDEX_MASK    << REVERSE_SHIFT, // Target から Anim へのバインド
        FLAG_SHIFT          = REVERSE_SHIFT * 2,
        FLAG_ENABLED        = ENABLED       << FLAG_SHIFT,
        FLAG_SKIP_CALC      = SKIP_CALC     << FLAG_SHIFT,
        FLAG_SKIP_APPLY     = SKIP_APPLY    << FLAG_SHIFT,
        FLAG_DISABLED       = DISABLED      << FLAG_SHIFT,
        FLAG_MASK           = 0x3           << FLAG_SHIFT,
        TARGET_BOUND        = 0x1   // BindTable がバインド済み
    };

    //----------------------------------------
    //! @name 構築
    //@{

    //! @brief コンストラクタです。
    AnimBindTable() {}

    //! @brief 初期化を行います。
    void Init(bit32* pBindArray, int tableSize);

    //@}

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //! @brief アニメーションの総数を設定します。
    void SetAnimCount(int animCount)
    {
        NW_G3D_ASSERT(0 <= animCount && animCount <= m_Size);
        m_NumAnim = static_cast<u16>(animCount);
    }

    //! @brief 関連付け状態を記録するテーブルのサイズを取得します。
    int GetSize() const { return m_Size; }

    //! @brief 記録されているアニメーションの総数を取得します。
    int GetAnimCount() const { return m_NumAnim; }

    //! @brief 記録されているアニメーション対象の総数を取得します。
    int GetTargetCount() const { return m_NumTarget; }

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    //! @brief 関連付け状態をクリアします。
    void ClearAll(int targetCount);

    //! @brief アニメーションからアニメーション対象への片方向の関連付けを元に、
    //! 両方向の関連付けを復元して記録します。
    void BindAll(const u16* pBindIndexArray);

    //! @brief アニメーションとアニメーション対象の関連付けを行います。
    void Bind(int animIndex, int targetIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        NW_G3D_ASSERT_INDEX_BOUNDS(targetIndex, m_NumTarget);
        m_pBindArray[animIndex] &= ~(INDEX_MASK | FLAG_MASK);
        m_pBindArray[animIndex] |= (targetIndex & INDEX_MASK);
        m_pBindArray[targetIndex] &= ~REVERSE_INDEX_MASK;
        m_pBindArray[targetIndex] |= (animIndex & INDEX_MASK) << REVERSE_SHIFT;
    }

    //! @brief アニメーションとアニメーション対象の関連付けを解除します。
    void Unbind(int animIndex, int targetIndex)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        NW_G3D_ASSERT_INDEX_BOUNDS(targetIndex, m_NumTarget);
        NW_G3D_ASSERT(targetIndex == GetTargetIndex(animIndex));
        m_pBindArray[animIndex] |= (NOT_BOUND | FLAG_DISABLED);
        m_pBindArray[targetIndex] |= REVERSE_NOT_BOUND;
    }

    //! @brief アニメーションに関連付けられているアニメーション対象のインデックスを取得します。
    int GetTargetIndex(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return  m_pBindArray[animIndex] & INDEX_MASK;
    }

    //! @brief アニメーション対象に関連付けられているアニメーションのインデックスを取得します。
    int GetAnimIndex(int targetIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(targetIndex, m_NumTarget);
        return ( m_pBindArray[targetIndex] >> REVERSE_SHIFT) & INDEX_MASK;
    }

    //! @brief バインドテーブルを関連付け済み状態にします。
    void SetTargetBound() { m_Flag |= TARGET_BOUND; }

    //! @brief バインドテーブルを関連付けされていない状態にします。
    void SetTargetUnbound() { m_Flag &= ~TARGET_BOUND; }

    //! @brief バインドテーブルが関連付け済みかどうかを取得します。
    bool IsTargetBound() const { return 0 != (m_Flag & TARGET_BOUND); }

    //@}

    //----------------------------------------
    //! @name 有効/無効
    //@{

    //! @brief アニメーションの計算が有効かどうかを取得します。
    bool IsCalcEnabled(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return CheckFlag(m_pBindArray[animIndex], FLAG_SKIP_CALC, 0);
    }
    //! @brief IsCalcEnabled() の別名関数です。
    bool IsCalculateEnabled(int animIndex) const
    {
        return IsCalcEnabled(animIndex);
    }

    //! @brief アニメーションの適用が有効かどうかを取得します。
    bool IsApplyEnabled(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return CheckFlag(m_pBindArray[animIndex], FLAG_SKIP_APPLY, 0);
    }

    //! @brief アニメーションの計算と適用が共に有効かどうかを取得します。
    bool IsEnabled(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return CheckFlag(m_pBindArray[animIndex], FLAG_MASK, FLAG_ENABLED);
    }

    //! @brief アニメーションの計算と適用が共に無効かどうかを取得します。
    bool IsDisabled(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return CheckFlag(m_pBindArray[animIndex], FLAG_MASK, FLAG_DISABLED);
    }

    //! @brief アニメーションの有効/無効に関するフラグを取得します。
    BindFlag GetBindFlag(int animIndex) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        return BindFlag(m_pBindArray[animIndex] >> FLAG_SHIFT);
    }

    //! @brief アニメーションの有効/無効に関するフラグを設定します。
    void SetBindFlag(int animIndex, BindFlag flag)
    {
        SetBindFlagRaw(animIndex, CreateFlagValue<bit32>(bit32(flag), FLAG_SHIFT, FLAG_MASK));
    }

    //! @brief アニメーションの有効/無効に関するフラグを設定します。
    void SetBindFlagRaw(int animIndex, bit32 flag)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(animIndex, m_NumAnim);
        // flag のマスクもシフトも行わない
        m_pBindArray[animIndex] &= ~FLAG_MASK;
        m_pBindArray[animIndex] |= flag;
    }

    //@}

private:
    bit32* m_pBindArray;
    bit16 m_Flag;
    u16 m_Size;
    u16 m_NumAnim;
    u16 m_NumTarget;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(AnimBindTable);
};

//--------------------------------------------------------------------------------------------------

//! @brief アニメーション高速化のためのコンテクストクラスです。
//!
//! 前回評価時のキーインデックスを記録し、次回評価時には記録したキーインデックスから
//! 線形探索を始めることで、キーの探索を高速化します。
//! 逆再生時やカーブがベイクされている場合は使用されません。
//!
class AnimContext
{
public:
    //----------------------------------------
    //! @name 構築
    //@{

    //! @brief コンストラクタです。
    AnimContext() {}

    //! @brief 初期化を行います。
    void Init(AnimFrameCache* pFrameCacheArray, int arraySize);

    //! @brief Init() の別名関数です。
    void Initialize(AnimFrameCache* pFrameCacheArray, int arraySize)
    {
        Init(pFrameCacheArray, arraySize);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief アニメーションカーブの数を設定します。
    void SetCurveCount(int curveCount)
    {
        NW_G3D_WARNING(m_pFrameCacheArray == NULL || curveCount <= m_Size,
            "Too many curves to store context.\n");
        m_NumCurve = curveCount;
        Reset();
    }

    //! @brief コンテクストをリセットします。
    void Reset()
    {
        m_LastFrame = AnimFrameCtrl::INVALID_FRAME;
        if (IsFrameCacheValid())
        {
            for (int idxCurve = 0; idxCurve < m_NumCurve; ++idxCurve)
            {
                m_pFrameCacheArray[idxCurve].start = std::numeric_limits<float>::infinity();
            }
        }
    }

    //! @brief アニメーションカーブの数を取得します。
    int GetCurveCount() const { return m_NumCurve; }

    //! @brief キーインデックスの記録が有効かどうかを取得します。
    bool IsFrameCacheValid() const { return 0 < m_NumCurve && m_NumCurve <= m_Size; }

    //! @brief キーインデックスの配列を取得します。
    AnimFrameCache* GetFrameCacheArray(int startIndex) { return &m_pFrameCacheArray[startIndex]; }

    //! @brief 直近のフレームを設定します。
    void SetLastFrame(float frame) { m_LastFrame = frame; }

    //! @brief 直近のフレームを取得します。
    float GetLastFrame() const { return m_LastFrame; }

    //@}

private:
    AnimFrameCache* m_pFrameCacheArray;
    int m_Size;
    int m_NumCurve;
    float m_LastFrame;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(AnimContext);
};

//--------------------------------------------------------------------------------------------------

//! @brief アニメーションインスタンスの基底クラスです。
class AnimObj
{
public:
    //! @brief 関連付けに関するフラグです。
    enum BindFlag
    {
        //! @brief 有効です。
        ENABLED             = AnimBindTable::ENABLED,

        //! @brief 更新処理を無視するが、最後の結果が Apply されます。
        SKIP_CALC           = AnimBindTable::SKIP_CALC,

        //! @brief Apply を無視するので、単独では無駄な更新が行われます。
        SKIP_APPLY          = AnimBindTable::SKIP_APPLY,

        //! @brief Calc も Apply もスキップします。
        DISABLED            = AnimBindTable::DISABLED
    };

    //----------------------------------------
    //! @name 破棄
    //@{

    //! @brief デストラクタです。
    virtual ~AnimObj() {}

    //@}

    //----------------------------------------
    //! @name フレーム制御
    //@{

    //! @brief フレームコントローラを取得します。
    AnimFrameCtrl& GetFrameCtrl() { return *m_pFrameCtrl; }

    //! @brief フレームコントローラを取得します。
    const AnimFrameCtrl& GetFrameCtrl() const { return *m_pFrameCtrl; }

    //! @brief フレームコントローラを設定します。
    //!
    //! デフォルトのフレームコントローラはリソースの変更時に自動的に初期化されますが、
    //! ここで設定したフレームコントローラは初期化されません。
    //!
    void SetFrameCtrl(AnimFrameCtrl* pFrameCtrl)
    {
        m_pFrameCtrl = pFrameCtrl != NULL ? pFrameCtrl : &m_DefaultFrameCtrl;
    }

    //! @brief このオブジェクトが所有権を持つデフォルトのフレームコントローラを取得します。
    AnimFrameCtrl& GetDefaultFrameCtrl() { return m_DefaultFrameCtrl; }

    //! @brief このオブジェクトが所有権を持つデフォルトのフレームコントローラを取得します。
    const AnimFrameCtrl& GetDefaultFrameCtrl() const { return m_DefaultFrameCtrl; }

    //@}

    //----------------------------------------
    //! @name コンテクスト
    //@{

    //! @brief アニメーション高速化のためのコンテクストを取得します。
    AnimContext& GetContext() { return m_Context; }

    //! @brief アニメーション高速化のためのコンテクストを取得します。
    const AnimContext& GetContext() const { return m_Context; }

    //! @brief アニメーション高速化のためのコンテクストを破棄します。
    void InvalidateContext() { m_Context.Reset(); }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief アニメーション結果を初期化します。
    //!
    //! バインド時に呼ばれるため、意図的に結果を初期化したい場合以外は呼ぶ必要はありません。
    //!
    virtual void ClearResult() = 0;

    //! @brief アニメーションカーブを評価します。
    //!
    //! 前回の計算時からフレームが変わっていない場合は計算を行いません。
    //!
    virtual void Calc() = 0;

    //! @brief Calc() の別名関数です。
    virtual void Calculate() = 0;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief カーブの総数を取得します。
    int GetCurveCount() const { return m_Context.GetCurveCount(); }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //@}

protected:

    //! @briefprivate
    AnimObj();

    void SetBufferPtr(void* pBuffer) { m_pBufferPtr = pBuffer; }

    //! @briefprivate
    void SetResultBuffer(void* pBuffer) { m_pResultBuffer = pBuffer; }

    //! @briefprivate
    void* GetResultBuffer() { return m_pResultBuffer; }

    //! @briefprivate
    const void* GetResultBuffer() const { return m_pResultBuffer; }

    //! @briefprivate
    void ResetFrameCtrl(int frameCount, bool loop);

    //! @briefprivate
    bool IsFrameChanged() const { return m_Context.GetLastFrame() != m_pFrameCtrl->GetFrame(); }

    //! @briefprivate
    void UpdateLastFrame() { m_Context.SetLastFrame(m_pFrameCtrl->GetFrame()); }

    //@}

private:
    AnimFrameCtrl* m_pFrameCtrl;
    AnimFrameCtrl m_DefaultFrameCtrl;
    AnimContext m_Context;
    void* m_pResultBuffer;
    void* m_pBufferPtr; // m_pResultBuffer は NULL になる場合があるので、別途覚えておく。

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(AnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @brief モデルアニメーションインスタンスの基底クラスです。
class ModelAnimObj : public AnimObj
{
public:

    //----------------------------------------
    //! @name 関連付け
    //@{

    // アニメーション種類ごとの関数
    // 派生クラスから呼び出す場合は派生先の関数を明示して呼び出す方が効率がよい。

    //! @brief アニメーション対象に関連付けます。
    virtual BindResult Bind(const ResModel* pModel) = 0;

    //! @brief アニメーション対象に関連付けます。
    virtual BindResult Bind(const ModelObj* pModel) = 0;

    //! @brief アニメーション対象に関連付けます。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    virtual void BindFast(const ResModel* pModel) = 0;

    //! @brief アニメーション対象への関連付け状態を表すオブジェクトを取得します。
    AnimBindTable& GetBindTable() { return m_BindTable; }

    //! @brief アニメーション対象への関連付け状態を表すオブジェクトを取得します。
    const AnimBindTable& GetBindTable() const { return m_BindTable; }

    //! @brief アニメーション対象に関連付けられているかどうかを取得します。
    bool IsTargetBound() const { return GetBindTable().IsTargetBound(); }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief アニメーション結果を対象に適用します。
    virtual void ApplyTo(ModelObj* pModelObj) const = 0;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief 処理するアニメーションの個数を取得します。
    //!
    //! ボーンやマテリアルに対応するアニメーションの個数です。
    //!
    int GetAnimCount() const { return m_BindTable.GetAnimCount(); }

    //! @brief ターゲットの総数を取得します。
    //!
    //! アニメーション対象となりうるボーンやマテリアルの最大数です。
    //! この数を超えてアニメーションをバインドすることはできません。
    //!
    int GetTargetCount() const { return m_BindTable.GetTargetCount(); }

    //@}

protected:
    //! @briefprivate
    ModelAnimObj() : AnimObj() {}

    //! @briefprivate
    void SetTargetBound()
    {
        GetBindTable().SetTargetBound();
        InvalidateContext(); // バインドし直した場合、前回評価のコンテクストは無効。
    }

    //! @briefprivate
    void SetTargetUnbound() { GetBindTable().SetTargetUnbound(); }

    //! @briefprivate
    void SetBindFlagImpl(int targetIndex, BindFlag flag);

    //! @briefprivate
    BindFlag GetBindFlagImpl(int targetIndex) const;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ModelAnimObj);

    AnimBindTable m_BindTable;
};

}} // namespace nw::g3d

#endif // NW_G3D_ANIMOBJ_H_
