﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GFXSHADER_H_
#define NW_G3D_FND_GFXSHADER_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GX2Struct.h>

namespace nw { namespace g3d { namespace fnd {

//! @brief シェーダです。
class GfxShader
{
public:
    static const u32 maxBindingPointNumber = 256;

    //! @briefprivate
    GfxShader() {}

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    struct SetupArg
    {
#if NW_G3D_IS_GX2
        GX2VertexShader* pVertexShader;
        GX2GeometryShader* pGeometryShader;
        GX2PixelShader* pFragmentShader;
#if NW_G3D_COMPUTE_SHADER_ENABLE
        GX2ComputeShader* pComputeShader;
#endif
#elif NW_G3D_IS_GL
        const char* pVertexShader;
        const char* pGeometryShader;
        const char* pFragmentShader;
        const char* pComputeShader;
        const char** ppTransformFeedbackVaryings;
        int numVaryings;
        const void* pBinary;
        int binarySize;
        uint binaryFormat;

        // 複数ソース版
        bool useMultiSource;
        const char** ppVertexShader;
        const char** ppGeometryShader;
        const char** ppFragmentShader;
        const char** ppComputeShader;
        int numVertexShader;
        int numGeometryShader;
        int numFragmentShader;
        int numComputeShader;
#endif

#if NW_G3D_GFXSHADER_SETUPARG_CONSTRUCTOR
        SetupArg() :
#if NW_G3D_IS_GX2
            pVertexShader(NULL),
            pGeometryShader(NULL),
            pFragmentShader(NULL)
#if NW_G3D_COMPUTE_SHADER_ENABLE
            , pComputeShader(NULL)
#endif
#elif NW_G3D_IS_GL
            pVertexShader(NULL),
            pGeometryShader(NULL),
            pFragmentShader(NULL),
            pComputeShader(NULL),
            ppTransformFeedbackVaryings(NULL),
            numVaryings(),
            pBinary(NULL),
            binarySize(),
            binaryFormat(),
            useMultiSource(false),
            ppVertexShader(),
            ppGeometryShader(),
            ppFragmentShader(),
            ppComputeShader(),
            numVertexShader(),
            numGeometryShader(),
            numFragmentShader(),
            numComputeShader()
#endif
        {}
#endif
    };

    //! @brief シェーダを構築します。
    void Setup(const SetupArg& arg);

    //! @brief GLバイナリでのセットアップが可能かどうかを取得します。
    bool IsGLBinaryAvailable(const SetupArg& arg);

    //! @brief シェーダを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief シェーダプログラムを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief 頂点シェーダにおける頂点属性のロケーションを取得します。
    s8 GetVertexAttribLocation(const char* name) const;

    //! @brief 頂点シェーダにおけるサンプラーのロケーションを取得します。
    s8 GetVertexSamplerLocation(const char* name) const;

    //! @brief ジオメトリシェーダにおけるサンプラーのロケーションを取得します。
    s8 GetGeometrySamplerLocation(const char* name) const;

    //! @brief フラグメントシェーダにおけるサンプラーのロケーションを取得します。
    s8 GetFragmentSamplerLocation(const char* name) const;

    //! @brief 演算シェーダにおけるサンプラーのロケーションを取得します。
    s8 GetComputeSamplerLocation(const char* name) const;

    //! @brief 頂点シェーダにおける UniformBlock のロケーションを取得します。
    s8 GetVertexBlockLocation(const char* name) const;

    //! @brief ジオメトリシェーダにおける UniformBlock のロケーションを取得します。
    s8 GetGeometryBlockLocation(const char* name) const;

    //! @brief フラグメントシェーダにおける UniformBlock のロケーションを取得します。
    s8 GetFragmentBlockLocation(const char* name) const;

    //! @brief 演算シェーダにおける UniformBlock のロケーションを取得します。
    s8 GetComputeBlockLocation(const char* name) const;

    //! @brief 頂点シェーダにおける UniformBlock のサイズを取得します。
    u32 GetVertexBlockSize(const char* name) const;

    //! @brief ジオメトリシェーダにおける UniformBlock のサイズを取得します。
    u32 GetGeometryBlockSize(const char* name) const;

    //! @brief フラグメントシェーダにおける UniformBlock のサイズを取得します。
    u32 GetFragmentBlockSize(const char* name) const;

    //! @brief 演算シェーダにおける UniformBlock のサイズを取得します。
    u32 GetComputeBlockSize(const char* name) const;

    //! @brief 頂点シェーダにおける Uniform の UniformBlock 内バイトオフセットを取得します。
    s32 GetVertexUniformOffset(const char* name) const;

    //! @brief ジオメトリシェーダにおける Uniform の UniformBlock 内バイトオフセットを取得します。
    s32 GetGeometryUniformOffset(const char* name) const;

    //! @brief フラグメントシェーダにおける Uniform の UniformBlock 内バイトオフセットを取得します。
    s32 GetFragmentUniformOffset(const char* name) const;

    //! @brief 演算シェーダにおける Uniform の UniformBlock 内バイトオフセットを取得します。
    s32 GetComputeUniformOffset(const char* name) const;

    //! @brief 頂点シェーダにおける Uniform のロケーションを取得します。
    s32 GetVertexUniformLocation(const char* name) const;

    //! @brief ジオメトリシェーダにおける Uniform のロケーションを取得します。
    s32 GetGeometryUniformLocation(const char* name) const;

    //! @brief フラグメントシェーダにおける Uniform のロケーションを取得します。
    s32 GetFragmentUniformLocation(const char* name) const;

    //! @brief 演算シェーダにおける Uniform のロケーションを取得します。
    s32 GetComputeUniformLocation(const char* name) const;

    //@}

#if NW_G3D_IS_GX2
    GX2VertexShader* pVertexShader;
    GX2GeometryShader* pGeometryShader;
    GX2PixelShader* pFragmentShader;
#if NW_G3D_COMPUTE_SHADER_ENABLE
    GX2ComputeShader* pComputeShader;
#endif
#else
    u32 handle;
#endif

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(GfxShader);
};

//--------------------------------------------------------------------------------------------------

//! @brief ジオメトリシェーダの入出力に用いるリングバッファです。
class GfxRingBuffer
{
public:

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    GfxRingBuffer() : m_pInputBuffer(NULL), m_pOutputBuffer(NULL), m_InputSize(0), m_OutputSize(0)
    {
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! ジオメトリシェーダの入力に用いるリングバッファのサイズを計算します。
    static size_t CalcInputBufferSize(u32 ringItem);

    //! ジオメトリシェーダの出力に用いるリングバッファのサイズを計算します。
    static size_t CalcOutputBufferSize(u32 ringItem);

    //! ジオメトリシェーダの入力に用いるリングバッファを設定します。
    void SetInputBuffer(void* ptr, size_t size);

    //! ジオメトリシェーダの入力に用いるリングバッファを取得します。
    void* GetInputBuffer() { return m_pInputBuffer; }

    //! ジオメトリシェーダの入力に用いるリングバッファを取得します。
    const void* GetInputBuffer() const { return m_pInputBuffer; }

    //! ジオメトリシェーダの入力に用いるリングバッファのサイズを取得します。
    size_t GetInputBufferSize() const { return m_InputSize; }

    //! ジオメトリシェーダの出力に用いるリングバッファを設定します。
    void SetOutputBuffer(void* ptr, size_t size);

    //! ジオメトリシェーダの出力に用いるリングバッファを取得します。
    void* GetOutputBuffer() { return m_pOutputBuffer; }

    //! ジオメトリシェーダの出力に用いるリングバッファを取得します。
    const void* GetOutputBuffer() const { return m_pOutputBuffer; }

    //! ジオメトリシェーダの入力に用いるリングバッファのサイズを取得します。
    size_t GetOutputBufferSize() const { return m_OutputSize; }

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! リングバッファを GPU に設定します。
    void Load();

    //@}

private:
    void* m_pInputBuffer;
    void* m_pOutputBuffer;
    size_t m_InputSize;
    size_t m_OutputSize;
};

//--------------------------------------------------------------------------------------------------

//! @brief シェーダモードを設定します。
void SetShaderMode(GX2ShaderMode mode);

//! @brief シェーダモードを設定します。
void SetShaderModeEx(GX2ShaderMode mode,
                     u32 vsGprs,
                     u32 vsStackSize,
                     u32 gsGprs,
                     u32 gsStackSize,
                     u32 psGprs,
                     u32 psStackSize);

}}} // namespace nw::g3d::fnd

#endif // NW_G3D_FND_GFXSHADER_H_
