﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GX2UTILITY_H_
#define NW_G3D_FND_GX2UTILITY_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GX2Struct.h>

namespace nw { namespace g3d { namespace fnd {

//! @brief GL 形式でのプリミティブタイプです。
struct GLPrimitiveType
{
    u32 drawType;
    u32 varyingType;
};

//! @brief GL 形式でのサーフェスフォーマットです。
struct GLSurfaceFormat
{
    GX2SurfaceFormat gx2Format;
    u32 detailFormat;
    u32 format;
    u32 type;
};

//! @brief GL 形式での頂点属性フォーマットです。
struct GLAttribFormat
{
    GX2AttribFormat gx2Format;
    u32 type;
    u32 size;
    u8 normalized;
};

const GLPrimitiveType& FindGLPrimitiveType(GX2PrimitiveType type);

//! @brief GX2SurfaceFormat に対応する GLSurfaceFormat を返します。
//!
//! @return 変換後のフォーマットです。変換できない場合は GX2_SURFACE_FORMAT_LAST が返ります。
//!
const GLSurfaceFormat& FindGLFormat(GX2SurfaceFormat format);

//! @brief GX2SurfaceFormat に対応するデプス用の GLSurfaceFormat を返します。
//!
//! @return 変換後のフォーマットです。変換できない場合は GX2_SURFACE_FORMAT_LAST が返ります。
//!
const GLSurfaceFormat& FindGLDepthFormat(GX2SurfaceFormat format);

//! @brief GX2AttribFormat に対応する GLAttribFormat を返します。
//!
//! @return 変換後のフォーマットです。変換できない場合は GX2_ATTRIB_FORMAT_LAST が返ります。
//!
const GLAttribFormat& FindGLFormat(GX2AttribFormat format);

//! @brief 実機フォーマットのテクスチャを OpenGL で読み込むことができる形式に変換します。
//!        元のデータを破壊することに注意してください。
//!
void ConvertToGLSurface(GX2Surface& surface, int arrayLength);

//! @brief GX2Surface のサイズとアライメントを計算して代入します。
//!        事前に size, alignment 以外を代入しておく必要があります。
//!
void CalcSurfaceSizeAndAlignment(GX2Surface& surface);

//! @brief 圧縮フォーマットであるかどうかを返します。
//!
//! @return 圧縮フォーマットの場合に true を返します。
//!
bool IsCompressed(GX2SurfaceFormat format);

//! @brief 画像のフォーマットと横幅、高さ、深度からバイトサイズを計算して返します。
//!
//! @return 画像のバイトサイズです。
//!
u32 CalcImageSize(GX2SurfaceFormat format, u32 width, u32 height, u32 depth);

//! @brief 指定されたミップマップレベルのピクセルイメージを返します。
//!
//! @return ピクセルイメージへのポインタです。
//!
void* GetImagePtr(GX2Surface& surface, u32 mipLevel);

//! @brief 指定されたミップマップレベルのピクセルイメージを返します。
//!
//! @return ピクセルイメージへのポインタです。
//!
const void* GetImagePtr(const GX2Surface& surface, u32 mipLevel);

}}} // namespace nw::g3d::fnd

#endif // NW_G3D_FND_GX2UTILITY_H_
