﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// edit と同じ状態（Release ビルドで無効化）するために include しています。
// edit から分離することになった際には対処します。
#include <nw/g3d/edit/g3d_EditDefs.h>

#if !defined(NW_G3D_CONFIG_USE_INPUTCAPTURE)
#define NW_G3D_CONFIG_USE_INPUTCAPTURE (0)
#endif

#if NW_G3D_CONFIG_USE_HOSTIO || ( NW_G3D_CONFIG_USE_INPUTCAPTURE && !defined(ANDROID) && !defined(__APPLE__) )

#include <cafe/os/OSCore.h> // $CAFE_ROOT/system/include を参照

#if NW_G3D_IS_HOST_CAFE
#include <cafe/hio.h>
#endif

namespace nw { namespace g3d { namespace edit {

//---------------------------------------------------------------------------
//! @brief キーボードの仮想キーコードです。
//---------------------------------------------------------------------------
enum Keycode
{
    KC_BACKSPACE            = 0x08, //!< BackSpace キー
    KC_TAB                  = 0x09, //!< Tab キー
    KC_LINEFEED             = 0x0A, //!< ライン フィード キー
    KC_CLEAR                = 0x0C, //!< Clear キー
    KC_ENTER                = 0x0D, //!< Enter キー
    KC_SHIFT                = 0x10, //!< Shift キー
    KC_CONTROL              = 0x11, //!< Ctrl キー
    KC_ALT                  = 0x12, //!< Alt キー
    KC_PAUSE                = 0x13, //!< Pause キー
    KC_CAPS_LOCK            = 0x14, //!< CapsLock キー
    KC_KANA                 = 0x15, //!< IME かなモード キー
    KC_HANGUL               = 0x15, //!< IME ハングル モード キー(KANAキーと同じ)
    KC_JUNJA                = 0x17, //!< IME Junja モード キー
    KC_FINAL                = 0x18, //!< IME Final モード キー
    KC_KANJI                = 0x19, //!< IME 漢字モード キー
    KC_HANJA                = 0x19, //!< IME Hanja モード キー (KANJIキーと同じ)
    KC_ESCAPE               = 0x1B, //!< Esc キー
    KC_IME_CONVERT          = 0x1C, //!< IME 変換キー
    KC_IME_NONCONVERT       = 0x1D, //!< IME 無変換キー
    KC_IME_ACCEPT           = 0x1E, //!< IME Accept キー
    KC_IME_MODECHANGE       = 0x1F, //!< IME モード変更キー
    KC_SPACE                = 0x20, //!< Space キー
    KC_PAGEUP               = 0x21, //!< PageUp キー
    KC_PAGEDOWN             = 0x22, //!< PageDown キー
    KC_END                  = 0x23, //!< End キー
    KC_HOME                 = 0x24, //!< Home キー
    KC_LEFT                 = 0x25, //!< ← キー
    KC_UP                   = 0x26, //!< ↑ キー
    KC_RIGHT                = 0x27, //!< → キー
    KC_DOWN                 = 0x28, //!< ↓ キー
    KC_SELECT               = 0x29, //!< Select キー
    KC_PRINT                = 0x2A, //!< Print キー
    KC_EXECUTE              = 0x2B, //!< Execute キー
    KC_PRINTSCREEN          = 0x2C, //!< PrintScreen キー
    KC_INSERT               = 0x2D, //!< Insert キー
    KC_DELETE               = 0x2E, //!< Delele キー
    KC_HELP                 = 0x2F, //!< Help キー
    KC_0                    = 0x30, //!< 0 キー
    KC_1                    = 0x31, //!< 1 キー
    KC_2                    = 0x32, //!< 2 キー
    KC_3                    = 0x33, //!< 3 キー
    KC_4                    = 0x34, //!< 4 キー
    KC_5                    = 0x35, //!< 5 キー
    KC_6                    = 0x36, //!< 6 キー
    KC_7                    = 0x37, //!< 7 キー
    KC_8                    = 0x38, //!< 8 キー
    KC_9                    = 0x39, //!< 9 キー
    KC_A                    = 0x41, //!< A キー
    KC_B                    = 0x42, //!< B キー
    KC_C                    = 0x43, //!< C キー
    KC_D                    = 0x44, //!< D キー
    KC_E                    = 0x45, //!< E キー
    KC_F                    = 0x46, //!< F キー
    KC_G                    = 0x47, //!< G キー
    KC_H                    = 0x48, //!< H キー
    KC_I                    = 0x49, //!< I キー
    KC_J                    = 0x4A, //!< J キー
    KC_K                    = 0x4B, //!< K キー
    KC_L                    = 0x4C, //!< L キー
    KC_M                    = 0x4D, //!< M キー
    KC_N                    = 0x4E, //!< N キー
    KC_O                    = 0x4F, //!< O キー
    KC_P                    = 0x50, //!< P キー
    KC_Q                    = 0x51, //!< Q キー
    KC_R                    = 0x52, //!< R キー
    KC_S                    = 0x53, //!< S キー
    KC_T                    = 0x54, //!< T キー
    KC_U                    = 0x55, //!< U キー
    KC_V                    = 0x56, //!< V キー
    KC_W                    = 0x57, //!< W キー
    KC_X                    = 0x58, //!< X キー
    KC_Y                    = 0x59, //!< Y キー
    KC_Z                    = 0x5A, //!< Z キー
    KC_LWIN                 = 0x5B, //!< 左の Windows ロゴ キー
    KC_RWIN                 = 0x5C, //!< 右の Windows ロゴ キー
    KC_APPS                 = 0x5D, //!< アプリケーション キー
    KC_SLEEP                = 0x5F, //!< Sleep キー
    KC_NUM_0                = 0x60, //!< 数値パッドの 0 キー
    KC_NUM_1                = 0x61, //!< 数値パッドの 1 キー
    KC_NUM_2                = 0x62, //!< 数値パッドの 2 キー
    KC_NUM_3                = 0x63, //!< 数値パッドの 3 キー
    KC_NUM_4                = 0x64, //!< 数値パッドの 4 キー
    KC_NUM_5                = 0x65, //!< 数値パッドの 5 キー
    KC_NUM_6                = 0x66, //!< 数値パッドの 6 キー
    KC_NUM_7                = 0x67, //!< 数値パッドの 7 キー
    KC_NUM_8                = 0x68, //!< 数値パッドの 8 キー
    KC_NUM_9                = 0x69, //!< 数値パッドの 9 キー
    KC_NUM_MULT             = 0x6A, //!< 数値パッドの * キー
    KC_NUM_PLUS             = 0x6B, //!< 数値パッドの + キー
    KC_NUM_COMMA            = 0x6C, //!< 数値パッドの , キー
    KC_NUM_MINUS            = 0x6D, //!< 数値パッドの - キー
    KC_NUM_PERIOD           = 0x6E, //!< 数値パッドの . キー
    KC_NUM_SLASH            = 0x6F, //!< 数値パッドの / キー
    KC_F1                   = 0x70, //!< F1 キー
    KC_F2                   = 0x71, //!< F2 キー
    KC_F3                   = 0x72, //!< F3 キー
    KC_F4                   = 0x73, //!< F4 キー
    KC_F5                   = 0x74, //!< F5 キー
    KC_F6                   = 0x75, //!< F6 キー
    KC_F7                   = 0x76, //!< F7 キー
    KC_F8                   = 0x77, //!< F8 キー
    KC_F9                   = 0x78, //!< F9 キー
    KC_F10                  = 0x79, //!< F10 キー
    KC_F11                  = 0x7A, //!< F11 キー
    KC_F12                  = 0x7B, //!< F12 キー
    KC_F13                  = 0x7C, //!< F13 キー
    KC_F14                  = 0x7D, //!< F14 キー
    KC_F15                  = 0x7E, //!< F15 キー
    KC_F16                  = 0x7F, //!< F16 キー
    KC_F17                  = 0x80, //!< F17 キー
    KC_F18                  = 0x81, //!< F18 キー
    KC_F19                  = 0x82, //!< F19 キー
    KC_F20                  = 0x83, //!< F20 キー
    KC_F21                  = 0x84, //!< F21 キー
    KC_F22                  = 0x85, //!< F22 キー
    KC_F23                  = 0x86, //!< F23 キー
    KC_F24                  = 0x87, //!< F24 キー
    KC_NUM_LOCK             = 0x90, //!< Num Lock キー
    KC_SCROLL_LOCK          = 0x91, //!< Scroll Lock キー
    KC_LSHIFT               = 0xA0, //!< 左 Shift キー
    KC_RSHIFT               = 0xA1, //!< 右 Shift キー
    KC_LCONTROL             = 0xA2, //!< 左 Ctrl キー
    KC_RCONTROL             = 0xA3, //!< 右 Ctrl キー
    KC_LALT                 = 0xA4, //!< 左 Alt キー
    KC_RALT                 = 0xA5, //!< 右 Alt キー
    KC_OEM_SEMICOLON        = 0xBA, //!< US キーボードのセミコロン キー
    KC_OEM_PLUS             = 0xBB, //!< US キーボードの + キー
    KC_OEM_COMMA            = 0xBC, //!< US キーボードの . キー
    KC_OEM_MINUS            = 0xBD, //!< US キーボードの - キー
    KC_OEM_PERIOD           = 0xBE, //!< US キーボードの . キー
    KC_OEM_QUESTION         = 0xBF, //!< US キーボードの ? キー
    KC_OEM_TILDE            = 0xC0, //!< US キーボードの ~ キー
    KC_OEM_OPEN_BRACKETS    = 0xDB, //!< US キーボードの [ キー
    KC_OEM_PIPE             = 0xDC, //!< US キーボードの | キー
    KC_OEM_CLOSE_BRACKETS   = 0xDD, //!< US キーボードの ] キー
    KC_OEM_QUOTE            = 0xDE, //!< US キーボードの " キー
    KC_OEM_8                = 0xDF, //!< US キーボードの 8 キー
    KC_OEM_BACKSLASH        = 0xE2  //!< US キーボードの \ キー
};

//---------------------------------------------------------------------------
//! @brief マウスボタンの列挙子です。
//---------------------------------------------------------------------------
enum MOUSE_BUTTON
{
    MOUSE_BUTTON_LEFT    = 0x01,    //!< 左ボタンです。
    MOUSE_BUTTON_RIGHT   = 0x02,    //!< 右ボタンです。
    MOUSE_BUTTON_MIDDLE  = 0x04,    //!< 中ボタンです。
    MOUSE_BUTTON_X1      = 0x08,    //!< X ボタン 1 です。
    MOUSE_BUTTON_X2      = 0x10     //!< X ボタン 2 です。
};

//---------------------------------------------------------------------------
//! @brief InputCapture を行うクラスです。
//---------------------------------------------------------------------------
class InputCapture
{
public:
    //! @brief コンストラクタに渡す引数です。
    struct InitArg
    {
        //! @brief コンストラクタです。
        InitArg()
            : width(1280)
            , height(720)
            , isPointerClampEnabled(false)
        {}

        //! @brief コンストラクタです。
        InitArg(u32 width, u32 height)
            : width(width)
            , height(height)
            , isPointerClampEnabled(false)
        {}

        u32 width;                  //!< 画面の幅です。
        u32 height;                 //!< 画面の高さです。
        bool isPointerClampEnabled; //!< マウスポインタの位置を画面にあわせてクランプするかどうかです。
    };

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ InputCapture( const InitArg& arg = InitArg() );

    //---------------------------------------------------------------------------
    //! @brief    InputCapture を開始します。
    //---------------------------------------------------------------------------
    bool    Open();

    //---------------------------------------------------------------------------
    //! @brief    入力データの Read を行います。
    //---------------------------------------------------------------------------
    void    Read();

    //---------------------------------------------------------------------------
    //! @brief    InputCapture を終了します。
    //---------------------------------------------------------------------------
    void    Close();

    //---------------------------------------------------------------------------
    //! @brief    画面の幅を取得します。
    //---------------------------------------------------------------------------
    u32     GetWidth() const
    {
        return m_ScreenWidth;
    }

    //---------------------------------------------------------------------------
    //! @brief    画面の高さを取得します。
    //---------------------------------------------------------------------------
    u32     GetHeight() const
    {
        return m_ScreenHeight;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスポインタの位置を画面にあわせてクランプするかを設定します。
    //---------------------------------------------------------------------------
    void    SetIsPointerClampEnabled( bool enabled )
    {
        m_IsPointerClampEnabled = enabled;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスポインタの位置を画面にあわせてクランプするかを取得します。
    //---------------------------------------------------------------------------
    bool    GetIsPointerClampEnabled() const
    {
        return m_IsPointerClampEnabled;
    }

    //! @name キー入力の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    キーが押されているかを取得します。
    //---------------------------------------------------------------------------
    bool    IsKeyHold( int keyCode ) const
    {
        return m_KeyState.Test( keyCode );
    }

    //---------------------------------------------------------------------------
    //! @brief    キーがこのフレーム押されているかを取得します。
    //---------------------------------------------------------------------------
    bool    IsKeyTriggered( int keyCode ) const
    {
        return m_KeyTriggered.Test( keyCode );
    }

    //---------------------------------------------------------------------------
    //! @brief    キーがこのフレームで離されているかを取得します。
    //---------------------------------------------------------------------------
    bool    IsKeyReleased( int keyCode ) const
    {
        return m_KeyReleased.Test( keyCode );
    }

    //---------------------------------------------------------------------------
    //! @brief    キーボードの言語識別子を取得します。
    //!
    //! キーボードレイアウト識別子(KLID)のうち、下位 16bit の言語識別子を取得します。
    //! KLID の値は、レジストリ上の
    //! HKEY_LOCAL_MACHINE/SYSTEM/CurrentControlSet/Control/Keyboard Layouts
    //! に定義されています。それらのうち言語識別子の例として
    //! 日本語:0x0411, 英語(米国):0x0409, 英語(英国):0x0809
    //! などが該当します。
    //!
    //---------------------------------------------------------------------------
    u32     GetKeyboardLanguageId() const
    {
        return m_KeyboardLanguageId;
    }

    //@}

    //! @name マウス入力の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    マウスボタンが押されているかを取得します。
    //---------------------------------------------------------------------------
    bool    IsMouseButtonHold( int mouseButton ) const
    {
        return ( m_MouseButton & mouseButton ) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスボタンがこのフレームで押されたかを取得します。
    //---------------------------------------------------------------------------
    bool    IsMouseButtonTriggered( int mouseButton ) const
    {
        return ( m_MouseButtonTriggered & mouseButton ) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスボタンがこのフレームで離されているかを取得します。
    //---------------------------------------------------------------------------
    bool    IsMouseButtonReleased( int mouseButton ) const
    {
        return ( m_MouseButtonReleased & mouseButton ) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスポインタの X 座標を取得します。
    //---------------------------------------------------------------------------
    int     GetMousePosX() const
    {
        return m_MousePosX;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスポインタの Y 座標を取得します。
    //---------------------------------------------------------------------------
    int     GetMousePosY() const
    {
        return m_MousePosY;
    }

    //---------------------------------------------------------------------------
    //! @brief    ホイールが回転したときの回転量を取得します。
    //---------------------------------------------------------------------------
    int     GetMouseWheelRaw() const
    {
        return m_MouseWheelRaw;
    }

    //---------------------------------------------------------------------------
    //! @brief    ホイールを回転したときにスクロールする行数を取得します。
    //---------------------------------------------------------------------------
    int     GetMouseWheel() const
    {
        return m_MouseWheelScrollLines;
    }

    //---------------------------------------------------------------------------
    //! @brief    マウスポインタが画面内にあるかどうかを取得します。
    //---------------------------------------------------------------------------
    bool    GetIsPointerOnScreen() const
    {
        return m_IsPointerOnScreen;
    }

    //@}

    //! @name キャプチャー状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      キャプチャーが開始されているかを取得します。
    //---------------------------------------------------------------------------
    bool    GetIsCapturing() const
    {
        return m_IsCapturing;
    }

    //@}

private:
    class BitSet256
    {
    public:
        /* ctor */ BitSet256()
        {
            this->Reset();
        }

        BitSet256& Reset()
        {
            for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
            {
                m_Flags[ i ] = 0;
            }
            return *this;
        }

        BitSet256& Set( int index, int value = 1 )
        {
            if ( value )
            {
                m_Flags[ index / 32 ] |= 0x1 << (index % 32);
            }
            else
            {
                m_Flags[ index / 32 ] &= ~(0x1 << (index % 32));
            }
            return *this;
        }

        bool Test( int index ) const
        {
            return (m_Flags[ index / 32 ] & (1 << (index % 32))) != 0;
        }

        bool Any() const
        {
            for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
            {
                if (this->m_Flags[ i ] > 0)
                {
                    return true;
                }
            }
            return false;
        }

        BitSet256 operator &(const BitSet256& bitSet)
        {
            BitSet256 result;
            for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
            {
                result.m_Flags[i] = this->m_Flags[ i ] & bitSet.m_Flags[ i ];
            }
            return result;
        }

        BitSet256 operator ^(const BitSet256& bitSet)
        {
            BitSet256 result;
            for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
            {
                result.m_Flags[i] = this->m_Flags[ i ] ^ bitSet.m_Flags[ i ];
            }
            return result;
        }

    private:
        u32 m_Flags[ 8 ];
    };

private:
    void Write(void* buf, u32 size);
    void Analyze();
    void AdjustMousePos();
    void SendStopCaptureMsg( int pointerPosX, int pointerPosY );
    void SendAckMsg();
    void ResetState();

#if NW_G3D_IS_HOST_CAFE
    static void ConnectionCallback(HIOStatus status, void* context);
    static void ReadCallback(HIOStatus status, void* context);
    static void WriteCallback(HIOStatus status, void* context);

    static HIOHandle s_Handle;
#endif

    static const int BUFFER_SIZE = 256;
    static const int BUFFER_COUNT = 2;

    struct ReadBuffer
    {
        bool    isReadStarted;
        bool    isReadDone;
        u8      pReadBuffer[BUFFER_SIZE];
        u32     readSize;

        explicit ReadBuffer()
            : isReadStarted(false)
            , isReadDone(false)
            , readSize(0)
        {}
    };

    bool    m_IsConnected;
    bool    m_IsCapturing;

    ReadBuffer m_ReadBuffer[BUFFER_COUNT];
    int     m_ReadBufferCount;
    int     m_AnalyzeBufferCount;

    BitSet256 m_KeyState;
    BitSet256 m_LastKeyState;
    BitSet256 m_KeyTriggered;
    BitSet256 m_KeyReleased;
    u32     m_KeyboardLanguageId;

    u32     m_MouseButton;
    u32     m_LastMouseButton;
    u32     m_MouseButtonTriggered;
    u32     m_MouseButtonReleased;

    int     m_MousePosX;
    int     m_MousePosY;
    int     m_MouseWheelRaw;
    int     m_MouseWheelScrollLines;

    u32     m_ScreenWidth;
    u32     m_ScreenHeight;

    u32     m_BorderType;
    u32     m_BorderLength;

    bool    m_IsPointerClampEnabled;
    bool    m_IsPointerOnScreen;
};

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
