﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/g3d_config.h>
#include <cafe/gx2/gx2Enum.h>

// リリースビルド時は、g3d::editの機能を無効にします。
#if defined(NW_RELEASE) && !defined(NW_USE_G3D_EDIT) || defined(NW_NO_USE_G3D_EDIT)

#define NW_G3D_CONFIG_USE_HOSTIO (0) // NOLINT

#else

#if !defined(ANDROID) && !defined(__APPLE__)
#define NW_G3D_CONFIG_USE_HOSTIO (1) // NOLINT
#endif

#endif

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResMaterial.h>

//--------------------------------------------------------------------------------------------------
// バージョン
enum {
    NW_G3D_VERSION_EDITMAJOR  = 3
};
enum {
    NW_G3D_VERSION_EDITMINOR  = 6,
};
enum {
    NW_G3D_VERSION_EDITMICRO  = 0,
};
enum {
    NW_G3D_VERSION_EDITBUGFIX = 0,
};


#define NW_G3D_EDIT_MAGIC               \
    NW_G3D_MAKE_U8X4_AS_U32('N','W','3','E')

#define NW_G3D_EDIT_VERSION             \
    NW_G3D_MAKE_U8X4_AS_U32(            \
    NW_G3D_VERSION_EDITMAJOR,           \
    NW_G3D_VERSION_EDITMINOR,           \
    NW_G3D_VERSION_EDITMICRO,           \
    NW_G3D_VERSION_EDITBUGFIX           \
    )

namespace nw { namespace g3d {

class ModelObj;
class ShapeAnimObj;
class CameraAnimObj;
class LightAnimObj;
class FogAnimObj;

namespace res {

class ResModel;
class ResShaderParamAnim;
class ResShaderArchive;

} // namespace res

namespace edit {

#if defined( _MSC_VER )
#pragma warning(push)
#pragma warning(disable:4201)
#endif

//--------------------------------------------------------------------------------------------------
// enum

/**
    @briefprivate 編集アニメーション種類
 */
enum EditAnimKind
{
    EDIT_SKELETAL_ANIM,
    EDIT_SHADER_PARAM_ANIM,
    EDIT_COLOR_ANIM,
    EDIT_TEX_SRT_ANIM,
    EDIT_TEX_PATTERN_ANIM,
    EDIT_BONE_VIS_ANIM,
    EDIT_MAT_VIS_ANIM,
    EDIT_SHAPE_ANIM
};

//--------------------------------------------------------------------------------------------------
// 引数用構造体

//! @brief LoadFileコールバック実行時に渡される構造体です。
struct LoadFileArg
{
    u32 key;            //!< @briefprivate
    void*   resFile;    //!< ロードされた ResFile です。
    size_t  fileSize;   //!< ファイルサイズです。
    size_t  align;      //!< ファイルに必要なアライメントです。
};

//! @brief UnloadFileコールバック実行時に渡される構造体です。
struct UnloadFileArg
{
    ResFile* resFile; //!< アンロード対象のResFile です。
};

//! @brief UpdateRenderInfoコールバック実行時に渡される構造体です。
struct RenderInfoArg
{
    const ModelObj* modelObj;   //!< 3DEditorが編集した対象のModelObjです。
    s32 materialIndex;          //!< 3DEditorが編集した対象のModelObjのマテリアルインデックスです。
    const char* name;           //!< 編集対象の描画情報名です。
};

//! @brief SendRenderInfoコールバック実行時に渡される構造体です。
struct SendRenderInfoArg
{
    const ModelObj* modelObj;   //!< 3DEditorが情報を要求している対象のModelObjです。
    s32 materialIndex;          //!< 3DEditorが情報を要求している対象のModelObjのマテリアルインデックスです。
};

//! @brief UpdateMaterialsコールバック実行時に渡される構造体です。
struct UpdateMaterialsArg
{
    //! @brief Stateを示すenumです。
    enum State
    {
        STATE_BEGIN,    //!< 初期状態時に格納される値です。
        STATE_UPDATE,   //!< 更新が行われた時に格納される値です。
        STATE_END,      //!< 終了状態時に格納される値です。
        STATE_ERROR
    };

    ModelObj*   modelObj;   //!< マテリアル更新対象のModelObjです。
    State       state;      //!< 更新状態です。
};

//! @brief UpdateShaderAssignコールバック実行時に渡される構造体です。
struct UpdateShaderAssignArg
{
    //! @brief Stateを示すenumです。
    enum State
    {
        STATE_MODEL_BEGIN,              //!< モデル初期状態時に格納される値です。
        STATE_MODEL_UPDATE,             //!< モデル更新が行われた時に格納される値です。
        STATE_MODEL_END,                //!< モデル終了状態時に格納される値です。
        STATE_ERROR
    };

    ModelObj* modelObj;                             //!< @briefprivate 更新対象のModelObjです。
    ResModel* oldResModel;                          //!< 適用前のResModelです。
    ResModel* newResModel;                          //!< 更新時に適用されるResModelです。
    ResShaderArchive** shaderArchivePtrs;           //!< 各マテリアルに対する変更されたシェーダアーカイブです。NULL のときは変更されたシェーダアーカイブがありません。
    int numShaderArchive;                           //!< シェーダアーカイブの数です。更新対象モデルのマテリアル数と一致します。次回廃止予定です。
    State state;                                    //!< 更新状態です。
};

//! @brief UpdateBoneBindコールバック実行時に渡される構造体です。
struct UpdateBoneBindArg
{
    ModelObj* parentModelObj;   //!< バインド対象の親 ModelObj です。
    int parentBoneIndex;        //!< バインド対象の親 ModelObj のボーンインデックスです。
    ModelObj* modelObj;         //!< バインド対象の ModelObj です。
};

//! @brief UpdateModelLayoutコールバック実行時に渡される構造体です。
struct UpdateModelLayoutArg
{
    ModelObj* modelObj; //!< モデル配置対象のModelObj です。
    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } scale; //!< モデル配置対象に設定するScaleです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } rotate; //!< モデル配置対象に設定するRotateです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } translate; //!< モデル配置対象に設定するTranslateです。
};

//! @brief SendModelLayoutコールバック実行時に渡される構造体です。
struct SendModelLayoutArg
{
    const ModelObj* modelObj; //!< モデル配置情報を送信する対象のModelObj です。
};

//! @brief SendModelLayoutコールバック実行後に送信される配置情報を格納する構造体です。
struct SendModelLayoutData
{
    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } scale; //!< 3DEditorに反映させるモデル配置情報のScaleです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } rotate; //!< 3DEditorに反映させるモデル配置情報のRotateです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } translate; //!< 3DEditorに反映させるモデル配置情報のTranslateです。
};

//! @brief BindSceneAnimコールバック実行時に渡される構造体です。
struct BindSceneAnimArg
{
    nw::g3d::CameraAnimObj**    cameraAnimObjPtrs; //!< バインド対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     lightAnimObjPtrs;  //!< バインド対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       fogAnimObjPtrs;    //!< バインド対象のフォグアニメのポインタ配列です。
    int                         numCameraAnimObj;  //!< バインド対象のカメラアニメのポインタ配列の要素数です。
    int                         numLightAnimObj;   //!< バインド対象のライトアニメのポインタ配列の要素数です。
    int                         numFogAnimObj;     //!< バインド対象のフォグアニメのポインタ配列の要素数です。
};

//! @brief UnbindSceneAnimコールバック実行時に渡される構造体です。
struct UnbindSceneAnimArg
{
    nw::g3d::CameraAnimObj**    cameraAnimObjPtrs; //!< アンバインド対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     lightAnimObjPtrs;  //!< アンバインド対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       fogAnimObjPtrs;    //!< アンバインド対象のフォグアニメのポインタ配列です。
    int                         numCameraAnimObj;  //!< アンバインド対象のカメラアニメのポインタ配列の要素数です。
    int                         numLightAnimObj;   //!< アンバインド対象のライトアニメのポインタ配列の要素数です。
    int                         numFogAnimObj;     //!< アンバインド対象のフォグアニメのポインタ配列の要素数です。
};

//! @brief ApplySceneAnimコールバック実行時に渡される構造体です。
struct ApplySceneAnimArg
{
    nw::g3d::CameraAnimObj**    cameraAnimObjPtrs; //!< アニメーション適用対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     lightAnimObjPtrs;  //!< アニメーション適用対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       fogAnimObjPtrs;    //!< アニメーション適用対象のフォグアニメのポインタ配列です。
    int                         numCameraAnimObj;  //!< アニメーション適用対象のカメラアニメのポインタ配列の要素数です。
    int                         numLightAnimObj;   //!< アニメーション適用対象のライトアニメのポインタ配列の要素数です。
    int                         numFogAnimObj;     //!< アニメーション適用対象のフォグアニメのポインタ配列の要素数です。
};

//! @brief SelectTargetコールバック実行時に渡される構造体です。
struct SelectTargetArg
{
    //! @brief 選択対象の種類です。
    enum TargetKind
    {
        TARGET_MODEL,      //!< モデルです。
        TARGET_MATERIAL,   //!< マテリアルです。
        TARGET_BONE,       //!< ボーンです。
        TARGET_SHAPE       //!< シェイプです。
    };

    TargetKind      targetKind;                   //!< 3DEditorで選択された対象の種類です。
    const ModelObj* modelObj;                     //!< 3DEditorで選択された対象のModelObjです。
    s32*            index;                        //!< 3DEditorで選択されたマテリアル、ボーン、シェイプのインデックスの配列です。
    u32             indexSize;                    //!< 3DEditorで選択されたマテリアル、ボーン、シェイプのインデックスの数です。
};

//! @brief AttachShader コールバック実行時に渡される構造体です。
struct AttachShaderArg
{
    nw::g3d::ResShaderArchive* shaderArchive; //! アタッチされたシェーダアーカイブへのポインタです。
};

//! @brief DetachShader コールバック実行時に渡される構造体です。
struct DetachShaderArg
{
    nw::g3d::ResShaderArchive* shaderArchive; //! デタッチされるシェーダアーカイブへのポインタです。
};

//! @brief AttachModel コールバック実行時に渡される構造体です。
struct AttachModelArg
{
    ModelObj* modelObj; //!< アタッチされたModelObjです。
};

//! @brief DetachModel コールバック実行時に渡される構造体です。
struct DetachModelArg
{
    ModelObj* modelObj; //!< デタッチされたModelObjです。
};

//! @brief UpdateShaderProgramコールバック実行時に渡される構造体です。
struct UpdateShaderProgramArg
{
    ResShaderArchive* shaderArchive;        //!< 更新対象のシェーダアーカイブです。
    int                     shadingModelIndex;    //!< 更新対象のシェーダアーカイブのシェーディングモデルインデックスです。
    int                     shaderProgramIndex;   //!< 更新対象のシェーダアーカイブのシェーダプログラムインデックスです。
};

//! @brief EditShadingModel コールバック実行時に渡される構造体です。
struct EditShadingModelArg
{
    ResShaderArchive* shaderArchive; //!< 更新対象のシェーダアーカイブです。
    int* shadingModelIndices; //!< 更新対象のシェーダアーカイブのシェーディングモデルインデックスの配列です。
    int numShadingModel; //! < 更新対象のシェーダアーカイブのシェーディングモデルインデックスの数です。
};

//! @brief ExecuteUserScript コールバック実行時に渡される構造体です。
struct ExecuteModelUserScriptArg
{
    ModelObj* targetModelObj; //!< スクリプト適用対象のモデルです。
    const char* scriptText; //!< スクリプト文字列です。
    const int* selectedMaterialIndices; //!< 選択されているマテリアルインデックスのリストです。
    const int* selectedShapeIndices; //!< 選択されているシェイプインデックスのリストです。
    const int* selectedBoneIndices; //!< 選択されているボーンインデックスのリストです。
    int selectedMaterialCount; //!< 選択されているマテリアルインデックスのリストの要素数です。
    int selectedShapeCount; //!< 選択されているシェイプインデックスのリストの要素数です。
    int selectedBoneCount; //!< 選択されているボーンインデックスのリストの要素数です。
};


//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelFileLoadedArg
{
    nw::g3d::ResFile* pResFile; //!< @brief 3DEditor から送信された ResFile です。
    size_t fileSize; //!< @brief ファイルサイズです。
    size_t alignment; //!< @brief ファイルに必要なアライメントです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelFileLoadedOutArg
{
    nw::g3d::ResFile* pResFile; //!< @brief コールバック内で入力引数から取得した ResFile をコピーした ResFile を渡します。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelFileUnloadedArg
{
    nw::g3d::ResFile* pResFile; //!< アンロード対象のResFile です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct TextureFileLoadedArg
{
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct TextureFileUnloadedArg
{
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct RenderInfoUpdatedArg
{
    const nw::g3d::ModelObj* pModelObj;   //!< 3DEditorが編集した対象のModelObjです。
    int materialIndex;          //!< 3DEditorが編集した対象のModelObjのマテリアルインデックスです。
    const char* name;           //!< 編集対象の描画情報名です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct SendRenderInfoRequestedArg
{
    const nw::g3d::ModelObj* pModelObj;   //!< 3DEditorが情報を要求している対象のModelObjです。
    int materialIndex;          //!< 3DEditorが情報を要求している対象のModelObjのマテリアルインデックスです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct MaterialUpdatedArg
{
    //! @brief Stateを示すenumです。
    enum State
    {
        State_Begin,    //!< 初期状態時に格納される値です。
        State_Update,   //!< 更新が行われた時に格納される値です。
        State_End,      //!< 終了状態時に格納される値です。
        State_Error
    };

    nw::g3d::ModelObj*   pModelObj;   //!< マテリアル更新対象のModelObjです。
    State state;      //!< 更新状態です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShaderAssignUpdatedArg
{
    //! @brief Stateを示すenumです。
    enum State
    {
        State_ModelBegin,              //!< モデル初期状態時に格納される値です。
        State_ModelUpdate,             //!< モデル更新が行われた時に格納される値です。
        State_ModelEnd,                //!< モデル終了状態時に格納される値です。
        State_Error
    };

    nw::g3d::ModelObj* pModelObj;                             //!< @briefprivate 更新対象のModelObjです。
    nw::g3d::ResModel* pOldResModel;                          //!< 適用前のResModelです。
    nw::g3d::ResModel* pNewResModel;                          //!< 更新時に適用されるResModelです。
    nw::g3d::ResShaderArchive** pShaderArchives;           //!< 各マテリアルに対する変更されたシェーダアーカイブの配列です。配列の長さはマテリアル数と一致します。NULL のときは変更されたシェーダアーカイブがありません。
    State state;                                    //!< 更新状態です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct BoneBindUpdatedArg
{
    nw::g3d::ModelObj* pParentModelObj;   //!< バインド対象の親 nw::g3d::ModelObj です。
    int parentBoneIndex;        //!< バインド対象の親 nw::g3d::ModelObj のボーンインデックスです。
    nw::g3d::ModelObj* pModelObj;         //!< バインド対象の nw::g3d::ModelObj です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelLayoutUpdatedArg
{
    nw::g3d::ModelObj* pModelObj; //!< モデル配置対象の ModelObj です。
    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } scale; //!< モデル配置対象に設定するScaleです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } rotate; //!< モデル配置対象に設定するRotateです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } translate; //!< モデル配置対象に設定するTranslateです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct SendModelLayoutRequestedArg
{
    const nw::g3d::ModelObj* pModelObj; //!< モデル配置情報を送信する対象のModelObj です。
};

    //! @briefprivate nn::g3d::viewer 互換用です。
struct SendModelLayoutRequestedOutArg
{
    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } scale; //!< モデル配置対象に設定するScaleです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } rotate; //!< モデル配置対象に設定するRotateです。

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } translate; //!< モデル配置対象に設定するTranslateです。
    bool isEnabled; //!< 3DEditor にモデル配置情報を反映させる場合は true、そうでない場合は false を指定します。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct SceneAnimBoundArg
{
    nw::g3d::CameraAnimObj**    pCameraAnimObjs; //!< バインド対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     pLightAnimObjs;  //!< バインド対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       pFogAnimObjs;    //!< バインド対象のフォグアニメのポインタ配列です。
    int                         cameraAnimObjCount;  //!< バインド対象のカメラアニメのポインタ配列の要素数です。
    int                         lightAnimObjCount;   //!< バインド対象のライトアニメのポインタ配列の要素数です。
    int                         fogAnimObjCount;     //!< バインド対象のフォグアニメのポインタ配列の要素数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct SceneAnimUnboundArg
{
    nw::g3d::CameraAnimObj**    pCameraAnimObjs; //!< アンバインド対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     pLightAnimObjs;  //!< アンバインド対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       pFogAnimObjs;    //!< アンバインド対象のフォグアニメのポインタ配列です。
    int                         cameraAnimObjCount;  //!< アンバインド対象のカメラアニメのポインタ配列の要素数です。
    int                         lightAnimObjCount;   //!< アンバインド対象のライトアニメのポインタ配列の要素数です。
    int                         fogAnimObjCount;     //!< アンバインド対象のフォグアニメのポインタ配列の要素数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ApplySceneAnimRequestedArg
{
    nw::g3d::CameraAnimObj**    pCameraAnimObjs; //!< アニメーション適用対象のカメラアニメのポインタ配列です。
    nw::g3d::LightAnimObj**     pLightAnimObjs;  //!< アニメーション適用対象のライトアニメのポインタ配列です。
    nw::g3d::FogAnimObj**       pFogAnimObjs;    //!< アニメーション適用対象のフォグアニメのポインタ配列です。
    int                         cameraAnimObjCount;  //!< アニメーション適用対象のカメラアニメのポインタ配列の要素数です。
    int                         lightAnimObjCount;   //!< アニメーション適用対象のライトアニメのポインタ配列の要素数です。
    int                         fogAnimObjCount;     //!< アニメーション適用対象のフォグアニメのポインタ配列の要素数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct TargetSelectedArg
{
    //! @brief 選択対象の種類です。
    enum TargetKind
    {
        TargetKind_Model,      //!< モデルです。
        TargetKind_Material,   //!< マテリアルです。
        TargetKind_Bone,       //!< ボーンです。
        TargetKind_Shape       //!< シェイプです。
    };

    TargetKind      targetKind;                   //!< 3DEditorで選択された対象の種類です。
    const nw::g3d::ModelObj* pModelObj;                     //!< 3DEditorで選択された対象のModelObjです。
    int*            index;                        //!< 3DEditorで選択されたマテリアル、ボーン、シェイプのインデックスの配列です。
    int             indexCount;                    //!< 3DEditorで選択されたマテリアル、ボーン、シェイプのインデックスの数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShaderAttachedArg
{
    nw::g3d::ResShaderArchive* pShaderArchive; //! アタッチされたシェーダアーカイブへのポインタです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShaderDetachedArg
{
    nw::g3d::ResShaderArchive* pShaderArchive; //! デタッチされるシェーダアーカイブへのポインタです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelAttachedArg
{
    nw::g3d::ModelObj* pModelObj; //!< アタッチされたModelObjです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelDetachedArg
{
    nw::g3d::ModelObj* pModelObj; //!< デタッチされたModelObjです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShaderProgramUpdatedArg
{
    nw::g3d::ResShaderArchive* pShaderArchive;        //!< 更新対象のシェーダアーカイブです。
    int                     shadingModelIndex;    //!< 更新対象のシェーダアーカイブのシェーディングモデルインデックスです。
    int                     shaderProgramIndex;   //!< 更新対象のシェーダアーカイブのシェーダプログラムインデックスです。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShadingModelUpdatedArg
{
    nw::g3d::ResShaderArchive* pShaderArchive; //!< 更新対象のシェーダアーカイブです。
    int* shadingModelIndices; //!< 更新対象のシェーダアーカイブのシェーディングモデルインデックスの配列です。
    int shadingModelCount; //! < 更新対象のシェーダアーカイブのシェーディングモデルインデックスの数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ModelUserScriptExecutedArg
{
    nw::g3d::ModelObj* pTargetModelObj; //!< スクリプト適用対象のモデルです。
    const char* scriptText; //!< スクリプト文字列です。
    const int* pSelectedMaterialIndices; //!< 選択されているマテリアルインデックスのリストです。
    const int* pSelectedShapeIndices; //!< 選択されているシェイプインデックスのリストです。
    const int* pSelectedBoneIndices; //!< 選択されているボーンインデックスのリストです。
    int selectedMaterialCount; //!< 選択されているマテリアルインデックスのリストの要素数です。
    int selectedShapeCount; //!< 選択されているシェイプインデックスのリストの要素数です。
    int selectedBoneCount; //!< 選択されているボーンインデックスのリストの要素数です。
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct ShapeUpdatedArg
{
    ModelObj* pModelObj;
};

//! @briefprivate nn::g3d::viewer 互換用です。
struct SamplerParamUpdatedArg
{
    ModelObj* pModelObj;
};

//! @brief edit の各イベントを独自拡張をするためのコールバッククラスです。
class EditCallback
{
public:
    virtual ~EditCallback() {}

    //! @brief ファイルをロード時の処理です。ファイルに対応するオブジェクトのインスタンスを返します。
    virtual nw::g3d::res::ResFile* LoadFile(const LoadFileArg& arg)
    {
        ModelFileLoadedOutArg outArg;
        outArg.pResFile = NULL;

        ModelFileLoadedArg inArg;
        inArg.alignment = arg.align;
        inArg.fileSize = arg.fileSize;
        inArg.pResFile = static_cast<ResFile*>(arg.resFile);
        ModelFileLoaded(outArg, inArg);
        return outArg.pResFile;
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelFileLoaded(ModelFileLoadedOutArg&, const ModelFileLoadedArg&)
    {
    }

    //! @brief ファイルをアンロード時の処理です。
    virtual void UnloadFile(const UnloadFileArg& arg)
    {
        ModelFileUnloadedArg inArg;
        inArg.pResFile = arg.resFile;
        ModelFileUnloaded(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelFileUnloaded(const ModelFileUnloadedArg&)
    {
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void TextureFileLoaded(const TextureFileLoadedArg&)
    {
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void TextureFileUnloaded(const TextureFileUnloadedArg&)
    {
    }

    //! @brief マテリアルの構造が変更された時に実行される処理です。
    virtual void UpdateMaterials(const UpdateMaterialsArg& arg)
    {
        MaterialUpdatedArg inArg;
        inArg.pModelObj = arg.modelObj;
        inArg.state = static_cast<MaterialUpdatedArg::State>(arg.state);
        MaterialUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void MaterialUpdated(const MaterialUpdatedArg&)
    {
    }

    //! @brief モデルの割り当て情報が変更された時に実行される処理です。
    virtual void UpdateShaderAssign(const UpdateShaderAssignArg& arg)
    {
        ShaderAssignUpdatedArg inArg;
        inArg.pModelObj = arg.modelObj;
        inArg.pNewResModel = arg.newResModel;
        inArg.pOldResModel = arg.oldResModel;
        inArg.pShaderArchives = arg.shaderArchivePtrs;
        inArg.state = static_cast<ShaderAssignUpdatedArg::State>(arg.state);
        ShaderAssignUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShaderAssignUpdated(const ShaderAssignUpdatedArg&)
    {
    }

    //! @brief 描画情報の状態が変更された時に実行される処理です。
    virtual void UpdateRenderInfo(const RenderInfoArg& arg)
    {
        RenderInfoUpdatedArg inArg;
        inArg.materialIndex = arg.materialIndex;
        inArg.name = arg.name;
        inArg.pModelObj = arg.modelObj;
        RenderInfoUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void RenderInfoUpdated(const RenderInfoUpdatedArg&)
    {
    }

    //! @brief 3DEditorから描画情報データが要求された時に実行される処理です。対応する描画情報データを3DEditor側に送信します。
    virtual void SendRenderInfo(const SendRenderInfoArg& arg)
    {
        SendRenderInfoRequestedArg inArg;
        inArg.materialIndex = arg.materialIndex;
        inArg.pModelObj = arg.modelObj;
        SendRenderInfoRequested(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void SendRenderInfoRequested(const SendRenderInfoRequestedArg&)
    {
    }

    //! @brief ボーンバインドの状態が変更された時に実行される処理です。
    virtual void UpdateBoneBind(const UpdateBoneBindArg& arg)
    {
        BoneBindUpdatedArg inArg;
        inArg.parentBoneIndex = arg.parentBoneIndex;
        inArg.pModelObj = arg.modelObj;
        inArg.pParentModelObj = arg.parentModelObj;
        BoneBindUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void BoneBindUpdated(const BoneBindUpdatedArg&)
    {
    }

    //! @brief モデル配置の状態が変更された時に実行される処理です。
    virtual void UpdateModelLayout(const UpdateModelLayoutArg& arg)
    {
        ModelLayoutUpdatedArg inArg;
        inArg.pModelObj = arg.modelObj;
        inArg.rotate.x = arg.rotate.x;
        inArg.rotate.y = arg.rotate.y;
        inArg.rotate.z = arg.rotate.z;
        inArg.scale.x = arg.scale.x;
        inArg.scale.y = arg.scale.y;
        inArg.scale.z = arg.scale.z;
        inArg.translate.x = arg.translate.x;
        inArg.translate.y = arg.translate.y;
        inArg.translate.z = arg.translate.z;
        ModelLayoutUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelLayoutUpdated(const ModelLayoutUpdatedArg&)
    {
    }

    //! @brief 3DEditorからモデル配置データが要求された時に実行される処理です。対応するモデル配置データを3DEditor側に送信します。
    virtual bool SendModelLayout(SendModelLayoutData* outputData, const SendModelLayoutArg& arg)
    {
        SendModelLayoutRequestedOutArg outArg;
        SendModelLayoutRequestedArg inArg;
        inArg.pModelObj = arg.modelObj;
        SendModelLayoutRequested(outArg, inArg);

        outputData->rotate.x = outArg.rotate.x;
        outputData->rotate.y = outArg.rotate.y;
        outputData->rotate.z = outArg.rotate.z;
        outputData->scale.x = outArg.scale.x;
        outputData->scale.y = outArg.scale.y;
        outputData->scale.z = outArg.scale.z;
        outputData->translate.x = outArg.translate.x;
        outputData->translate.y = outArg.translate.y;
        outputData->translate.z = outArg.translate.z;
        return outArg.isEnabled;
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void SendModelLayoutRequested(SendModelLayoutRequestedOutArg& arg, const SendModelLayoutRequestedArg&)
    {
        arg.isEnabled = false;
    }

    //! @brief 3DEditorからシーンアニメをバインド対象に設定した時に実行される処理です。
    virtual void BindSceneAnim(const BindSceneAnimArg& arg)
    {
        SceneAnimBoundArg inArg;
        inArg.cameraAnimObjCount = arg.numCameraAnimObj;
        inArg.fogAnimObjCount = arg.numFogAnimObj;
        inArg.lightAnimObjCount = arg.numLightAnimObj;
        inArg.pCameraAnimObjs = arg.cameraAnimObjPtrs;
        inArg.pFogAnimObjs = arg.fogAnimObjPtrs;
        inArg.pLightAnimObjs = arg.lightAnimObjPtrs;
        SceneAnimBound(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void SceneAnimBound(const SceneAnimBoundArg&)
    {
    }

    //! @brief 3DEditorからシーンアニメをバインド対象から外した時に実行される処理です。
    virtual void UnbindSceneAnim(const UnbindSceneAnimArg& arg)
    {
        SceneAnimUnboundArg inArg;
        inArg.cameraAnimObjCount = arg.numCameraAnimObj;
        inArg.fogAnimObjCount = arg.numFogAnimObj;
        inArg.lightAnimObjCount = arg.numLightAnimObj;
        inArg.pCameraAnimObjs = arg.cameraAnimObjPtrs;
        inArg.pFogAnimObjs = arg.fogAnimObjPtrs;
        inArg.pLightAnimObjs = arg.lightAnimObjPtrs;
        SceneAnimUnbound(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void SceneAnimUnbound(const SceneAnimUnboundArg&)
    {
    }

    /** @brief バインド対象に設定されているシーンアニメの計算結果を反映させる時に実行される処理です。
               このタイミングで各アニメの計算結果を適用したい対象に設定する事ができます。
     */
    virtual void ApplySceneAnim(const ApplySceneAnimArg& arg)
    {
        ApplySceneAnimRequestedArg inArg;
        inArg.cameraAnimObjCount = arg.numCameraAnimObj;
        inArg.fogAnimObjCount = arg.numFogAnimObj;
        inArg.lightAnimObjCount = arg.numLightAnimObj;
        inArg.pCameraAnimObjs = arg.cameraAnimObjPtrs;
        inArg.pFogAnimObjs = arg.fogAnimObjPtrs;
        inArg.pLightAnimObjs = arg.lightAnimObjPtrs;
        ApplySceneAnimRequested(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ApplySceneAnimRequested(const ApplySceneAnimRequestedArg&)
    {
    }

    /** @brief 3DEditorからマテリアルなどが選択された時に実行される処理です。
               モデルの選択の場合には SelectTargetArg の index は常に NULL になります。
               SelectTargetArg の index の指している内容はこのコールバックが終了した時点で無効になります。
               遅延処理を行う場合などは index の内容をコピーして下さい。
     */
    virtual void SelectTarget(const SelectTargetArg& arg)
    {
        TargetSelectedArg inArg;
        inArg.index = arg.index;
        inArg.indexCount = arg.indexSize;
        inArg.pModelObj = arg.modelObj;
        inArg.targetKind = static_cast<TargetSelectedArg::TargetKind>(arg.targetKind);
        TargetSelected(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void TargetSelected(const TargetSelectedArg&)
    {
    }

    //! @brief モデルがアタッチされた時に実行される処理です。
    virtual void AttachModel(const AttachModelArg& arg)
    {
        ModelAttachedArg inArg;
        inArg.pModelObj = arg.modelObj;
        ModelAttached(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelAttached(const ModelAttachedArg&)
    {
    }

    //! @brief モデルがデタッチされた時に実行される処理です。
    virtual void DetachModel(const DetachModelArg& arg)
    {
        ModelDetachedArg inArg;
        inArg.pModelObj = arg.modelObj;
        ModelDetached(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelDetached(const ModelDetachedArg&)
    {
    }

    //! @brief シェーダがアタッチされた後に実行される処理です。
    virtual void AttachShader(const AttachShaderArg& arg)
    {
        ShaderAttachedArg inArg;
        inArg.pShaderArchive = arg.shaderArchive;
        ShaderAttached(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShaderAttached(const ShaderAttachedArg&)
    {
    }

    //! @brief シェーダがデタッチされた後に実行される処理です。
    virtual void DetachShader(const DetachShaderArg& arg)
    {
        ShaderDetachedArg inArg;
        inArg.pShaderArchive = arg.shaderArchive;
        ShaderDetached(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShaderDetached(const ShaderDetachedArg&)
    {
    }

    //! @brief レンダーステートが変更された時に実行される処理です。
    virtual void UpdateRenderState(ModelObj* modelObj)
    {
        (void)modelObj;
    }

    //! @brief サンプラのパラメータが変更された時に実行される処理です。
    virtual void UpdateSamplerParam(ModelObj* modelObj)
    {
        SamplerParamUpdatedArg inArg;
        inArg.pModelObj = modelObj;
        SamplerParamUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void SamplerParamUpdated(const SamplerParamUpdatedArg&)
    {
    }

    /**
        @briefprivate シェイプが変更された時に実行される処理です。
    */
    virtual void UpdateShape(ModelObj* modelObj)
    {
        ShapeUpdatedArg inArg;
        inArg.pModelObj = modelObj;
        ShapeUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShapeUpdated(const ShapeUpdatedArg&)
    {
    }

    /**
        @briefprivate アタッチシェーダ時、3DEditor からシェーディングモデルが編集されたときに実行される処理です。
    */
    virtual void EditShadingModel(const EditShadingModelArg& arg)
    {
        ShadingModelUpdatedArg inArg;
        inArg.pShaderArchive = arg.shaderArchive;
        inArg.shadingModelCount = arg.numShadingModel;
        inArg.shadingModelIndices = arg.shadingModelIndices;
        ShadingModelUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShadingModelUpdated(const ShadingModelUpdatedArg&)
    {
    }

    /**
        @briefprivate アタッチシェーダ時、遅延コンパイル後にシェーダプログラムを変更した後に実行される処理です。
     */
    virtual void UpdateShaderProgram(const UpdateShaderProgramArg& arg)
    {
        ShaderProgramUpdatedArg inArg;
        inArg.pShaderArchive = arg.shaderArchive;
        inArg.shaderProgramIndex = arg.shaderProgramIndex;
        inArg.shadingModelIndex = arg.shadingModelIndex;
        ShaderProgramUpdated(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ShaderProgramUpdated(const ShaderProgramUpdatedArg&)
    {
    }

    /**
        @briefprivate モデルに対するユーザスクリプトが実行された時に呼ばれる処理です。
     */
    virtual void ExecuteModelUserScript(const ExecuteModelUserScriptArg& arg)
    {
        ModelUserScriptExecutedArg inArg;
        inArg.pSelectedBoneIndices = arg.selectedBoneIndices;
        inArg.pSelectedMaterialIndices = arg.selectedMaterialIndices;
        inArg.pSelectedShapeIndices = arg.selectedShapeIndices;
        inArg.pTargetModelObj = arg.targetModelObj;
        inArg.scriptText = arg.scriptText;
        inArg.selectedBoneCount = arg.selectedBoneCount;
        inArg.selectedMaterialCount = arg.selectedMaterialCount;
        inArg.selectedShapeCount = arg.selectedShapeCount;
        ModelUserScriptExecuted(inArg);
    }

    //! @briefprivate nn::g3d::viewer 互換用です。
    virtual void ModelUserScriptExecuted(const ModelUserScriptExecutedArg&)
    {
    }

protected:
    EditCallback() {}
};

#if defined( _MSC_VER )
#pragma warning(pop)
#endif

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
