﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_TAGPROCESSORBASE_H_
#define NW_FONT_TAGPROCESSORBASE_H_

#include <nw/types.h>
#include <nw/ut/ut_Rect.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

namespace nw {
namespace font {



template <typename CharType>
class TextWriterBase;


template <typename CharType>
struct PrintContext
{
    TextWriterBase<CharType>*   writer;     //!< 文字列描画に使用しているTextWriterへのポインタ。
    const CharType*             str;        //!< 描画中の文字列へのポインタ。
    const CharType*             strEnd;     //!< 描画中の文字列の終端のポインタ。
    const f32                   xOrigin;    //!< 描画開始X座標。
    const f32                   yOrigin;    //!< 描画開始Y座標。
    const f32                   hScale;     //!< 描画開始時点の水平方向スケール。
    const f32                   vScale;     //!< 描画開始時点の垂直方向スケール。
    u32                         flags;      //!< TextWriter::ContextFlag のOR。
    u16                         prevCode;   //!< ひとつ前の文字。カーニングの値を求めるために使用します。

    PrintContext(
        TextWriterBase<CharType>*   aWriter,
        const CharType*             aStr,
        const CharType*             aStrEnd,
        const f32                   aXOrigin,
        const f32                   aYOrigin,
        const f32                   aHScale,
        const f32                   aVScale,
        u32                         aFlags
    )
    :   writer(aWriter),
        str(aStr),
        strEnd(aStrEnd),
        xOrigin(aXOrigin),
        yOrigin(aYOrigin),
        hScale(aHScale),
        vScale(aVScale),
        flags(aFlags),
        prevCode(0)
    {}

private:
    // コピー演算子を禁止します。
    const PrintContext&   operator=( const PrintContext& rhs);
};

#if defined( __CC_ARM ) || defined( __ARMCC_VERSION )
#pragma push
#pragma diag_suppress 2819  // class XXX has an implicitly instantiated key function YYY
#endif

//---------------------------------------------------------------------------
//! @brief        グリフグループが定義されているフォントリソースを扱うクラスです。
//!
//! @tparam       CharType 文字の型。
//---------------------------------------------------------------------------
template <typename CharType>
class TagProcessorBase
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef PrintContext<CharType>  ContextType;

    //! 呼び出し元 (TextWriter) が行うべき処理の列挙子です。
    enum Operation
    {
        //! @brief      行頭では次の文字との間に文字間を開けません。
        //!             行頭以外の位置では次の文字との間に文字間を開けます。
        //!
        OPERATION_DEFAULT,

        //! @brief      行頭以外の位置でタグ処理が行われた場合、
        //!             デフォルトでは次の文字との間に文字間を空けますが、
        //!             これを空けないようにします。
        //!
        OPERATION_NO_CHAR_SPACE,

        //! @brief      行頭でタグ処理が行われた場合、
        //!             デフォルトでは次の文字との間に文字間を空けませんが、
        //!             これを空けるようにします。
        //!
        OPERATION_CHAR_SPACE,

        //! @brief      改行時に必要な処理のうち X 座標の調整のみを行います。
        //!             Y 座標の調整は行いません。
        //!             また、次の文字との間に文字間を開けません。
        //!
        OPERATION_NEXT_LINE,

        //! @brief      文字列の途中で文字列描画を終了する場合に使用します。
        //!
        OPERATION_END_DRAW,

        //! @brief 内部用機能のため使用禁止です。
        NUM_OF_OPERATION
    };

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ / デストラクタ
    //@{

    //! コンストラクタです。
                        TagProcessorBase();

    //! デストラクタです。
    virtual             ~TagProcessorBase();

    //@}


    //! @name タグ処理
    //@{

    //! @brief      タグに従って処理を行います。
    //!
    //! @param[in]      code      タグ処理の起点となった制御文字コード。(0x0000 ～ 0x001F)
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    //! @return     呼び出し元 (TextWriter) が行うべき処理を返します。
    //!
    virtual Operation   Process(
                            u16                     code,
                            PrintContext<CharType>* pContext);

    //! @brief      タグの影響範囲を計算します。
    //!
    //! @param[out]     pRect     タグの影響範囲を格納する矩形構造体へのポインタ。
    //! @param[in]      code      タグ処理の起点となった制御文字コード。(0x0000 ～ 0x001F)
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    //! @return     呼び出し元 (TextWriter) が行うべき処理を返します。
    //!             同じタグに対して Process() 関数が返す値と同じ値を返すべきです。
    //!
    virtual Operation   CalcRect(
                            ut::Rect*               pRect,
                            u16                     code,
                            PrintContext<CharType>* pContext);

    //@}

    //! @name 処理開始/終了コールバック
    //@{

    //! @brief      テキストのプリントを行う前に呼び出されるコールバックです。
    //!
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    virtual void BeginPrint(PrintContext<CharType>* pContext);

    //! @brief      テキストのプリントを行った後に呼び出されるコールバックです。
    //!
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    virtual void EndPrint(PrintContext<CharType>* pContext);

    //! @brief      テキストの占める領域の計算を行う前に呼び出されるコールバックです。
    //!
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    virtual void BeginCalcRect(PrintContext<CharType>* pContext);

    //! @brief      テキストの占める領域の計算を行った後に呼び出されるコールバックです。
    //!
    //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
    //!
    virtual void EndCalcRect(PrintContext<CharType>* pContext);

    //@}

private:
    void                ProcessLinefeed(ContextType* pContext);
    void                ProcessTab(ContextType* pContext);
};

#if defined( __CC_ARM ) || defined( __ARMCC_VERSION )
#pragma pop
#endif

}   // namespace font
}   // namespace nw

#endif //  NW_FONT_TAGPROCESSORBASE_H_
