﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_FONT_CHARWRITER_H_
#define NW_FONT_CHARWRITER_H_

#if defined(_MSC_VER) && _MSC_VER >= 1500
#pragma once
#endif

// 左上原点
#define COORDINATE_LT 1

#include <nw/types.h>
#include <nw/ut/ut_Color.h>
#include <nw/font/font_Types.h>
#include <nw/font/font_Font.h>
#include <nw/math/math_Types.h>

namespace nw {
namespace font {

class DispStringBuffer;

//---------------------------------------------------------------------------
//! @brief        Fontクラスを用いて文字の描画を行うためのクラスです。
//---------------------------------------------------------------------------
class CharWriter
{
public:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */

    //! グラデーションの種類を表す enum です。
    enum GradationMode
    {
        GRADMODE_NONE,                          //!< グラデーションなし
        GRADMODE_H,                             //!< 非サポートです。
        GRADMODE_V,                             //!< 垂直方向グラデーション

        NUM_OF_GRADMODE
    };



    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const u32 DEFAULT_COLOR_MAPPING_MIN = 0x00000000UL;
    static const u32 DEFAULT_COLOR_MAPPING_MAX = 0xFFFFFFFFUL;
    #if defined(NW_PLATFORM_CAFE)
    static const int GRAPHICS_BUFFER_ALIGNMENT = GX2_VERTEX_BUFFER_ALIGNMENT;
    #else
    static const int GRAPHICS_BUFFER_ALIGNMENT = 4;
    #endif

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */

    //! @name コンストラクタ／デストラクタ
    //@{

    //! コンストラクタです。
                        CharWriter();

    //! デストラクタです。
                        ~CharWriter();

    //@}

    //! @name フォント
    //@{

    //! @brief      フォントを設定します。
    //!
    //! @param[in]  pFont  新しく設定するフォントへのポインタです。
    //!
    void                SetFont(const Font* pFont)  { m_pFont = pFont; }

    //! @brief      フォントを取得します。
    //!
    //! @return     フォントが設定されている場合にはフォントへのポインタを返します。
    //!             設定されていない場合には NULL を返します。
    //!
    const Font*         GetFont() const             { return m_pFont; }

    //@}

    //! @name 文字色／グラデーション
    //@{

    //! @brief      グラデーションモードを設定します。
    //!
    //! 廃止予定の関数です。
    //! 内部的に未対応の機能だったので、代わりになる機能等が追加される予定はありません。
    //!
    //! @param[in]  mode    新しく設定するグラデーションモードです。
    //!
    NW_DEPRECATED_FUNCTION(void                SetGradationMode(GradationMode mode))
    {
        NW_FONT_MINMAX_ASSERT(mode, GRADMODE_NONE, NUM_OF_GRADMODE - 1);
        m_GradationMode = mode;
    }

    //! @brief      グラデーションモードを取得します。
    //!
    //! 廃止予定の関数です。
    //! 内部的に未対応の機能だったので、代わりになる機能等が追加される予定はありません。
    //!
    //! @return     グラデーションモードを返します。
    //!
    NW_DEPRECATED_FUNCTION(GradationMode       GetGradationMode() const)    { return m_GradationMode; }

    //! @brief      文字色またはグラデーション色を設定します。
    //!
    //! @param[in]  color  文字色。
    //!
    void                SetTextColor(ut::Color4u8 color)
    {
        m_TextColors[internal::TEXTCOLOR_START] = color;
        m_TextColors[internal::TEXTCOLOR_END] = color;
    }

    //! @brief      文字色またはグラデーション色を設定します。
    //!
    //! @param[in]  start  グラデーション開始色です。
    //! @param[in]  end    グラデーション終了色です。
    //!
    void                SetTextColor(
                            ut::Color4u8  start,
                            ut::Color4u8  end
    )
    {
        m_TextColors[internal::TEXTCOLOR_START] = start;
        m_TextColors[internal::TEXTCOLOR_END] = end;
    }

    //! @brief      文字色を取得します。
    //!
    //! @return     文字色を返します。
    //!
    const ut::Color4u8     GetTextColor() const           { return m_TextColors[internal::TEXTCOLOR_START]; }

    //! @brief      グラデーション開始色を取得します。
    //!
    //! @return     グラデーション開始色を返します。
    //!
    const ut::Color4u8     GetGradationStartColor() const { return m_TextColors[internal::TEXTCOLOR_START]; }

    //! @brief      グラデーション終了色を取得します。
    //!
    //! @return     グラデーション終了色を返します。
    //!
    const ut::Color4u8     GetGradationEndColor() const   { return m_TextColors[internal::TEXTCOLOR_END]; }

    //! @brief      影のアルファ値を設定します。
    //!
    //! @param[in]  alpha   アルファ値です。
    //!
    void                   SetShadowAlpha(u8 alpha)      { m_ShadowAlpha = alpha; }

    //! @brief      影のアルファ値を取得します。
    //!
    //! @return     アルファ値を返します。
    //!
    u8                     GetShadowAlpha()              { return m_ShadowAlpha; }

    //@}

    //! @name 文字の拡大縮小
    //@{

    //! @brief      文字の拡大率を指定します。
    //!
    //! @param[in]  hScale  横方向拡大率。
    //! @param[in]  vScale  縦方向拡大率。
    //!
    void                SetScale(
                            f32 hScale,
                            f32 vScale
    )
    {
        m_Scale.x = hScale;
        m_Scale.y = vScale;
    }

    //! @brief      文字の拡大率を指定します。
    //!
    //! @param[in]  hvScale 拡大率。
    //!
    void                SetScale(f32 hvScale)
    {
        m_Scale.x = hvScale;
        m_Scale.y = hvScale;
    }

    //! @brief  文字の横方向の拡大率を取得します。
    //!
    //! @return 文字の横方向の拡大率を返します。
    //!
    f32                 GetScaleH() const   { return m_Scale.x; }

    //! @brief  文字の縦方向の拡大率を取得します。
    //!
    //! @return 文字の縦方向の拡大率を返します。
    //!
    f32                 GetScaleV() const   { return m_Scale.y; }

    //! @brief      文字のサイズを指定します。
    //!
    //! @param[in]  width   拡大後のセルの幅。
    //! @param[in]  height  拡大後のセルの高さ。
    //!
    void                SetFontSize(
                            f32 width,
                            f32 height);

    //! @brief      文字のサイズを指定します。
    //!
    //! @param[in]  height  拡大後のセルの高さ。
    //!
    void                SetFontSize(f32 height);

    //! @brief      拡大適用後のセル幅を取得します。
    //!
    //! @return     フォントが設定されている場合は拡大適用後のセル幅をピクセル単位で取得します。
    //!             フォントが設定されていない場合の返り値は未定義です。
    //!
    f32                 GetFontWidth() const;

    //! @brief      拡大適用後のセル高さを取得します。
    //!
    //! @return     フォントが設定されている場合は拡大適用後のセル高さをピクセル単位で取得します。
    //!             フォントが設定されていない場合の返り値は未定義です。
    //!
    f32                 GetFontHeight() const;

    //! @brief      拡大適用後のアセントを取得します。
    //!
    //! @return     拡大適用後のアセントを返します。
    //!
    f32                 GetFontAscent() const;

    //! @brief      拡大適用後のディセントを取得します。
    //!
    //! @return     拡大適用後のディセントを返します。
    //!
    f32                 GetFontDescent() const;

    //@}

    //! @name 文字装飾
    //@{

    //! @brief        イタリック体の傾き度合を設定します。
    //!
    //! @param[in]  ratio   傾き度合です。
    //!
    void                SetItalicRatio(f32 ratio)   { m_ItalicRatio = ratio; }

    //! @brief        イタリック体の傾き度合を取得します。
    //!
    //! @return       傾き度合です。
    //!
    f32                 GetItalicRatio() const { return m_ItalicRatio; }

    //@}

    //! @name 等幅描画
    //@{

    //! @brief      強制等幅描画を行うように設定します。
    //!
    //! @param[in]  isFixed 強制等幅描画を行うのであれば true を指定します。
    //!
    void                EnableFixedWidth(bool isFixed)  { m_IsWidthFixed = isFixed; }

    //! @brief  強制等幅描画を行うかどうかを取得します。
    //!
    //! @return 強制等幅描画を行う場合は true を、行わない場合は false を返します。
    //!
    bool                IsWidthFixed() const            { return m_IsWidthFixed; }

    //! @brief      強制等幅描画の文字幅を指定します。
    //!
    //! @param[in]  width   新しい等幅描画幅。
    //!
    void                SetFixedWidth(f32 width)        { m_FixedWidth = width; }

    //! @brief      強制等幅描画の文字幅を取得します。
    //!
    //! @return     強制等幅描画の文字幅を返します。
    //!
    f32                 GetFixedWidth() const           { return m_FixedWidth; }

    //@}

    //! @name 文字描画
    //@{

    //! @brief      グリフを描画します。
    //!
    //! @param[in]  glyph   描画するグリフ。
    //!
    //! @return     描画したグリフの文字幅を返します。
    //!
    f32                 PrintGlyph(const Glyph& glyph);

    //! @brief      文字を描画します。
    //!
    //! @param[in]  code    描画する文字の文字コード。
    //!
    //! @return     描画した文字の文字幅を返します。
    //!
    f32                 Print(CharCode code);

    //! @brief      グリフを描画します。
    //!
    //! @param[in]  glyph   描画する文字のグリフデータ。
    //!
    void                DrawGlyph(const Glyph& glyph);

    //@}

    //! @name カーソル
    //@{

    //! @brief      カーソル位置を指定した新しい座標に設定します。
    //!
    //! @param[in]  x   カーソルの新しい X 座標。
    //! @param[in]  y   カーソルの新しい Y 座標。
    //!
    void                SetCursor(
                            f32 x,
                            f32 y
    )
    {
        m_CursorPos.x = x;
        m_CursorPos.y = y;
    }

    //! @brief      カーソル位置を指定した新しい座標に設定します。
    //!
    //! @param[in]  x   カーソルの新しい X 座標。
    //! @param[in]  y   カーソルの新しい Y 座標。
    //! @param[in]  z   カーソルの新しい Z 座標。
    //!
    void                SetCursor(
                            f32 x,
                            f32 y,
                            f32 z
    )
    {
        m_CursorPos.x = x;
        m_CursorPos.y = y;
        m_CursorPos.z = z;
    }

    //! @brief      カーソルを現在位置から指定された差分だけ移動します。
    //!
    //! @param[in]  dx  X 軸方向のカーソル移動量。
    //! @param[in]  dy  Y 軸方向のカーソル移動量。
    //!
    void                MoveCursor(
                            f32 dx,
                            f32 dy
    )
    {
        m_CursorPos.x += dx;
        m_CursorPos.y += dy;
    }

    //! @brief      カーソルを現在位置から指定された差分だけ移動します。
    //!
    //! @param[in]  dx  X 軸方向のカーソル移動量。
    //! @param[in]  dy  Y 軸方向のカーソル移動量。
    //! @param[in]  dz  Z 軸方向のカーソル移動量。
    //!
    void                MoveCursor(
                            f32 dx,
                            f32 dy,
                            f32 dz
    )
    {
        m_CursorPos.x += dx;
        m_CursorPos.y += dy;
        m_CursorPos.z += dz;
    }

    //! @brief      カーソルの X 座標を設定します。
    //!
    //! @param[in]  x   カーソルの新しい X 座標。
    //!
    void                SetCursorX(f32 x)           { m_CursorPos.x = x; }

    //! @brief      カーソルの Y 座標を設定します。
    //!
    //! @param[in]  y   カーソルの新しい Y 座標。
    //!
    void                SetCursorY(f32 y)           { m_CursorPos.y = y; }

    //! @brief      カーソルの Z 座標を設定します。
    //!
    //! @param[in]  z   カーソルの新しい Z 座標。
    //!
    void                SetCursorZ(f32 z)           { m_CursorPos.z = z; }

    //! @brief      カーソルを現在位置から指定された差分だけ移動します。
    //!
    //! @param[in]  dx  X 軸方向のカーソル移動量。
    //!
    void                MoveCursorX(f32 dx)         { m_CursorPos.x += dx; }

    //! @brief      カーソルを現在位置から指定された差分だけ移動します。
    //!
    //! @param[in]  dy  Y 軸方向のカーソル移動量。
    //!
    void                MoveCursorY(f32 dy)         { m_CursorPos.y += dy; }

    //! @brief      カーソルを現在位置から指定された差分だけ移動します。
    //!
    //! @param[in]  dz  Z 軸方向のカーソル移動量。
    //!
    void                MoveCursorZ(f32 dz)         { m_CursorPos.z += dz; }

    //! @brief      カーソルのX座標を取得します。
    //!
    //! @return     カーソルのX座標を返します。
    //!
    f32                 GetCursorX() const          { return m_CursorPos.x; }

    //! @brief      カーソルのY座標を取得します。
    //!
    //! @return     カーソルのY座標を返します。
    //!
    f32                 GetCursorY() const          { return m_CursorPos.y; }

    //! @brief      カーソルのZ座標を取得します。
    //!
    //! @return     カーソルのZ座標を返します。
    //!
    f32                 GetCursorZ() const          { return m_CursorPos.z; }

    //@}

    //! @name 文字列表示用バッファ
    //@{

    //! @brief      文字列表示用バッファのサイズを計算します。
    //!
    //! @param[in]  charNum         最大描画文字数です。
    //!
    //! @return     文字列表示用バッファのサイズを返します。
    //!
    static u32          GetDispStringBufferSize(u32 charNum);

    //! @brief      共有頂点属性配列使用時の文字列表示用バッファのサイズを計算します。
    //!
    //! @return     文字列表示用バッファのサイズを返します。
    //!
    static u32          GetDispStringBufferSizeWithSharedCharAttrs();

    //! @brief      グラフィックス用バッファのサイズを計算します。
    //!
    //! @param[in]  charNum         最大描画文字数です。
    //!
    //! @return     文字列表示用バッファのサイズを返します。
    //!
    static u32          GetGraphicsBufferSize(u32 charNum);

    //! @brief      文字列表示用バッファを初期化します。
    //!
    //! @param[in]  drawBuffer      描画用バッファです。
    //! @param[in]  graphicsBuffer  グラフィックス用バッファです。GRAPHICS_BUFFER_ALIGNMENT アライメントになっている必要があります。
    //! @param[in]  charNum         最大描画文字数です。
    //! @param[in]  shadowEnabled   影が有効か否かです。省略した場合は無効となります。
    //!
    //! @return     初期化した文字列表示用バッファを返します。
    //!
    static DispStringBuffer*
                        InitDispStringBuffer(
                            void* drawBuffer,
                            void* graphicsBuffer,
                            u32 charNum,
                            bool shadowEnabled = false);

    //! @brief      共有頂点属性配列を使用して文字列表示用バッファを初期化します。
    //!
    //! @param[in]  drawBuffer      描画用バッファです。
    //! @param[in]  graphicsBuffer  グラフィックス用バッファです。GRAPHICS_BUFFER_ALIGNMENT アライメントになっている必要があります。
    //! @param[in]  charNum         最大描画文字数です。
    //! @param[in]  shadowEnabled   影が有効か否かです。省略した場合は無効となります。
    //!
    //! @return     初期化した文字列表示用バッファを返します。
    //!
    static DispStringBuffer*
                        InitDispStringBufferWithSharedCharAttrs(
                            void* drawBuffer,
                            void* graphicsBuffer,
                            u32 charNum,
                            bool shadowEnabled = false);

    //! @brief      文字列表示用バッファを取得します。
    //!
    //! @return     文字列表示用バッファを返します。
    //!
    DispStringBuffer*   GetDispStringBuffer() const
    {
        return m_pDispStringBuffer;
    }

    //! @brief      文字列表示用バッファをセットします。
    //!             NULLを設定した場合は文字列表示用バッファを使用しなくなります。
    //!
    //! @param[in]  buffer 文字列表示用バッファです。
    //!
    void                SetDispStringBuffer(DispStringBuffer* buffer)
    {
        m_pDispStringBuffer = buffer;
    }

    //! @brief      文字列の描画を開始します。
    //!             文字列表示用バッファが設定されている場合に使用します。
    //!
    void                StartPrint();

    //! @brief      文字列の描画を終了します。
    //!             文字列表示用バッファが設定されている場合に使用します。
    //!
    void                EndPrint();

    //! @brief      文字列表示用のコマンドバッファをカレントのコマンドリストに追加します。
    //!
    void                UseCommandBuffer();

    //@}

private:

    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef math::VEC2      CharScale;
    typedef math::VEC3      CursorPos;

    struct ColorMapping
    {
        ut::Color4u8          min;
        ut::Color4u8          max;
    };

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- 描画準備

    //! @brief      グラフィックスエンジンを文字描画用のデフォルト設定にします。
    //!
    //! @param[in]  bAlphaTex テクスチャがアルファのみのテクスチャの場合は真。
    //!
    void                SetupGXDefault(bool bAlphaTex = false);

    //! @brief      グラフィックスエンジンをテクスチャカラーの線形変換ありの
    //!             文字描画用に設定します。
    //!
    //! @param[in]  bAlphaTex テクスチャがアルファのみのテクスチャの場合は真。
    //!
    void                SetupGXWithColorMapping(bool bAlphaTex = false);

    //! 頂点フォーマットを文字描画用に設定します。
    void                SetupVertexFormat();

    //! グラフィックスエンジンに対して共通の設定を行います。
    void                SetupGXCommon();

    //! @brief      グリフを指定位置に描画します。
    //!             カーソル移動は行いません。
    //!
    //! @param[in]  x      描画位置のX座標。
    //! @param[in]  glyph  描画するグリフデータ。
    //!
    void                PrintGlyph(
                            f32             x,
                            const Glyph&    glyph);

    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    ut::Color4u8                  m_TextColors[internal::TEXTCOLOR_MAX];  //!< 文字色
    GradationMode               m_GradationMode;
    CharScale                   m_Scale;                //!< 文字拡大率
    CursorPos                   m_CursorPos;            //!< カーソル位置
    f32                         m_FixedWidth;           //!< 等幅時の幅
    f32                         m_ItalicRatio;          //!< イタリック体傾き度合
    const Font*                 m_pFont;                //!< フォント

    //! 頂点バッファオブジェクト用構造体へのポインタ
    DispStringBuffer*           m_pDispStringBuffer;

    bool                        m_IsWidthFixed;         //!< 等幅描画するかどうか
    u8                          m_ShadowAlpha;          //!< 影のアルファ
};

}   // namespace font
}   // namespace nw
#endif //  NW_FONT_CHARWRITER_H_
