﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eftcom_Message.h>
#include <nw/eft/eftcom_Protocol.h>
#include <nw/eft/eft2_Heap.h>
#include <nw/ut.h>

namespace nw    {
namespace eftvw2  {

class ViewerSystem;
typedef void ( *ViewerDataCallback )( nw::eft2::Heap* heap, void* pthis, void* pBin );
typedef void ( *ViewerMessageCallback )( nw::eft2::Heap* heap, void* pthis, void* pBin );

//------------------------------------------------------------------------------
//! @brief      パケット処理するクラスです.
//------------------------------------------------------------------------------
class PacketProcedure
{
public:
    //--------------------------------------------------------------------------
    //! @brief      コンストラクタ
    //--------------------------------------------------------------------------
    PacketProcedure() : mViewerSystem(NULL) {}

    //--------------------------------------------------------------------------
    //! @brief      デストラクタ
    //--------------------------------------------------------------------------
    virtual ~PacketProcedure()
    {
        if(mViewerTempData)
        {
            mHeap->Free(mViewerTempData);
            mViewerTempData = NULL;
            mViewerTempDataSize = 0;
        }
    }

    //--------------------------------------------------------------------------
    //! @brief      初期化
    //!
    //! @param[in]      viewerSystem        ビューアシステムです.
    //! @param[in]      heap      TBD
    //--------------------------------------------------------------------------
    void Initialize( ViewerSystem* viewerSystem, nw::eft2::Heap* heap )
    {
        mViewerSystem          = viewerSystem;
        mHeap                  = heap;
        mViewerTempData        = NULL;
        mViewerTempDataSize    = 0;
        mViewerDataCallback    = NULL;
        mViewerMessageCallback = NULL;
    }

    //--------------------------------------------------------------------------
    //! @brief      渡されたパケットを処理します
    //!
    //! @param[in]      buffer      パケットデータ.
    //! @param[in]      szBuff      パケットサイズです.
    //--------------------------------------------------------------------------
    void DoPacket( char* buffer, size_t szBuff);

    //--------------------------------------------------------------------------
    //! @brief      ビューアデータ用のデータコールバックを設定します。
    //! @param[in] pThis     TBD
    //! @param[in] callback     TBD
    //--------------------------------------------------------------------------
    void SetViewerDataCallback( void* pThis, ViewerDataCallback callback )
    {
        mViewerDataCallbackObject = pThis;
        mViewerDataCallback       = callback;
    }

    //--------------------------------------------------------------------------
    //! @brief      ビューアメッセージ用のデータコールバックを設定します。
    //! @param[in] pThis     TBD
    //! @param[in] callback     TBD
    //--------------------------------------------------------------------------
    void SetViewerMessageCallback( void* pThis, ViewerMessageCallback callback )
    {
        mViewerMessageCallbackObject = pThis;
        mViewerMessageCallback       = callback;
    }
private:
    //--------------------------------------------------------------------------
    //! @brief      指定されたemitterの変数を更新します。
    //!
    //! @param[in]      pData       データ.
    //--------------------------------------------------------------------------
    void UpdateAssetEmitter( void* pData );

    //--------------------------------------------------------------------------
    //! @brief      ビューアデータの変数を更新します。
    //!
    //! @param[in]      offset             オフセット.
    //! @param[in]      overrideData       上書きデータ.
    //! @param[in]      overrideDataSize   上書きサイズ.
    //--------------------------------------------------------------------------
    void UpdateViewerData( u32            offset,
                           void*          overrideData,
                           u32            overrideDataSize );

    //--------------------------------------------------------------------------
    //! @brief      制御メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータ.
    //--------------------------------------------------------------------------
    void HandleMessageControl( char* pBuffer );

    //--------------------------------------------------------------------------
    //! @brief      バイナリ転送メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータ.
    //! @param[in]      bufferSize  バッファサイズです.
    //--------------------------------------------------------------------------
    void HandleMessageSendBinary( char* pBuffer, u32 bufferSize );

    //--------------------------------------------------------------------------
    //! @brief      インスタンス生成メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータ.
    //--------------------------------------------------------------------------
    void HandleMessageCreate( char* pBuffer );

    //--------------------------------------------------------------------------
    //! @brief      リアルタイム編集メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータです.
    //--------------------------------------------------------------------------
    void HandleMessageModify( char* pBuffer );

    //--------------------------------------------------------------------------
    //! @brief      インスタンス削除メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータです.
    //--------------------------------------------------------------------------
    void HandleMessageDelete( char* pBuffer );

    //--------------------------------------------------------------------------
    //! @brief      バイナリ削除メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータです.
    //--------------------------------------------------------------------------
    void HandleMessageRemoveBinary( char* pBuffer );

    //--------------------------------------------------------------------------
    //! @brief      ビューアの再生制御メッセージを処理します.
    //!
    //! @param[in]      pBuffer     パケットデータです.
    //--------------------------------------------------------------------------
    void HandleMessageViewerControl( char* pBuffer );

#if 0
    //------------------------------------------------
    //! リロード処理を行います。
    //------------------------------------------------
    void PacketProcedure::ReloadProcess( nw::eftcom::Message* header, void* pData );
#endif

    ViewerSystem*      mViewerSystem;              //!< ビューアシステムです.
    nw::eft2::Heap*    mHeap;                      //!< ヒープ

    void*              mViewerTempData;            //!< ビューアテンポラリデータ
    u32                mViewerTempDataSize;        //!< ビューアテンポラリデータサイズ

    ViewerDataCallback mViewerDataCallback;        //!< ビューアデータコールバック関数
    void*              mViewerDataCallbackObject;  //!< ビューアデータコールバックオブジェクト

    ViewerDataCallback mViewerMessageCallback;        //!< ビューアメッセージコールバック関数
    void*              mViewerMessageCallbackObject;  //!< ビューアメッセージコールバックオブジェクト
};

} // namespace eftvw2
} // namespace nw
