﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Heap.h>

namespace nw     {
namespace eftut2 {

//---------------------------------------------------------------------------
//! @brief  エフェクトヒープ Nwラッパ版
//---------------------------------------------------------------------------
class NwHeap : public nw::eft2::Heap
{
    // ヒープ一貫性チェック
    #define _HEAP_COHERENCE_CHECK2 (0)

public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    NwHeap()
    {
        m_Allocator  = NULL;
        m_Counter    = 0;
        m_IsCreated  = false;

#if _HEAP_COHERENCE_CHECK2
        for( u32 i = 0; i < 1024; i++ )
        {
            m_Adress[i] = NULL;
        }
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief                  コンストラクタです。
    //! @param[in] allocator    アロケータへのポインタ
    //---------------------------------------------------------------------------
    explicit NwHeap( nw::ut::MemoryAllocator* allocator )
    {
        m_Allocator  = allocator;

#if _HEAP_COHERENCE_CHECK2
        for( u32 i = 0; i < 1024; i++ )
        {
            m_Adress[i] = NULL;
        }
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~NwHeap()
    {
        Finalize();
    }

    //---------------------------------------------------------------------------
    //! @brief                  nw::ut::IAllocatorインスタンスを設定します。
    //! @param[in] allocator    アロケータへのポインタ
    //---------------------------------------------------------------------------
    void SetNwAllocator( nw::ut::IAllocator* allocator )
    {
        m_Allocator  = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief  nw::ut::IAllocatorインスタンスを取得します。
    //! @return アロケータへのポインタ
    //---------------------------------------------------------------------------
    nw::ut::IAllocator* GetNwAllocator() const
    {
        return m_Allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief              ヒープを生成する
    //! @param[in] buffer   先頭アドレス
    //! @param[in] size     バッファサイズ
    //---------------------------------------------------------------------------
    void Initialize( void* buffer, u32 size )
    {
        m_CreatedAllocator.Initialize( buffer, size );
        SetNwAllocator( &m_CreatedAllocator );
        m_IsCreated = true;
    }

    //---------------------------------------------------------------------------
    //! @brief                  メモリを確保します。
    //! @param[in] size         メモリサイズ
    //! @param[in] alignment    アライメント
    //! @return                 先頭アドレス
    //---------------------------------------------------------------------------
    virtual void* Alloc( u32 size, s32 alignment = nw::eft2::Heap::EFT_HEAP_DEAFULT_ALIGNMENT )
    {
        void* ptr = m_Allocator->Alloc( size, alignment );
//        void* ptr =  _aligned_malloc( size, alignment );
//        EFT_ASSERT( ptr );
        m_Counter++;

#if _HEAP_COHERENCE_CHECK2
        for( u32 i = 0; i < 1024; i++ )
        {
            if ( m_Adress[i] == NULL )
            {
                m_Adress[i] = ptr;
                if ( i == 32 )
                {
                    m_Adress[i] = ptr;
                }
                break;
            }
        }
#endif

        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief          メモリを解放します。
    //! @param[in] ptr  先頭アドレス
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr )
    {
        EFT_ASSERT( ptr );

#if _HEAP_COHERENCE_CHECK2
        bool ret = false;

        for( u32 i = 0; i < 1024; i++ )
        {
            if ( m_Adress[i] == ptr )
            {
                m_Adress[i] = NULL;
                ret = true;
                break;
            }
        }

        if( !ret )
        {
             EFT_ERR( "[EFTUT] Heap Free Error! " );
        }
#endif
        if ( m_Counter > 0 )
        {
            m_Counter--;
        }
        else
        {
            EFT_ERR( "[EFTUT] Heap Free Error! " );
        }

        return m_Allocator->Free( ptr );
//        return _aligned_free( ptr );
    }


    //---------------------------------------------------------------------------
    //! @brief  メモリ確保カウンターを取得します。
    //! @return メモリ確保カウンターの値
    //---------------------------------------------------------------------------
    s32 GetCounter() const
    {
        return m_Counter;
    }


    //---------------------------------------------------------------------------
    //! @brief        終了処理
    //---------------------------------------------------------------------------
    void Finalize()
    {
        EFT_ASSERT( m_Counter == 0 );

        if ( m_IsCreated )
        {
            m_CreatedAllocator.Finalize();
            m_IsCreated = false;
        }
    }

private:
    nw::ut::IAllocator*      m_Allocator;           //!< アロケータへのポインタ
    nw::ut::MemoryAllocator  m_CreatedAllocator;    //!< メモリアロケータ
    u32                      m_Counter;             //!< メモリ確保カウンター
    bool                     m_IsCreated;           //!< 初期化済みチェック

#if _HEAP_COHERENCE_CHECK2
    void*                    m_Adress[1024];        //!< メモリアドレス保管用配列
#endif
};


} // namespace eftut2
} // namespace nw
