﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nw   {
namespace eft2 {

//---------------------------------------------------
//! @brief ビュー情報の管理構造体です。
//!
//!        シェーダ上の "viewUniformBlock" ユニフォームブロックに設定されます。
//---------------------------------------------------
struct ViewParam
{
    static const float DefaultZOffset;  //!< zOffset の規定値

    nw::math::MTX44   viewMat;          //!< view        マトリクス
    nw::math::MTX44   projMat;          //!< proj        マトリクス
    nw::math::MTX44   vpMat;            //!< view * proj マトリクス
    nw::math::MTX44   bldMat;           //!< ビルボード  マトリクス
    nw::math::VEC4    eyeVec;           //!< 視線ベクトル
    nw::math::VEC4    eyePos;           //!< 視点位置
    nw::math::VEC4    viewParam;        //!< near / far / near * far / far  - near /
    nw::math::VEC4    viewParam2;       //!< baseFovy / currentFovy / zOffset (Avoid Z-Fighting)
    // 32 byte align is required.

    //---------------------------------------------------------------------------
    //! @brief                  TBD
    //! @param[in] view         TBD
    //! @param[in] proj         TBD
    //! @param[in] camPos       TBD
    //! @param[in] screenNear   TBD
    //! @param[in] screenFar    TBD
    //---------------------------------------------------------------------------
    void Set(   nw::math::MTX34* view,
                nw::math::MTX44* proj,
                nw::math::VEC3*  camPos,
                f32 screenNear,
                f32 screenFar )
    {
        // Fovy 関連のパラメータは 1.0f を入れておけば無害
        Set( view, proj, camPos, screenNear, screenFar, 1.0f, 1.0f, DefaultZOffset );
    }

    //---------------------------------------------------------------------------
    //! @brief                  TBD
    //! @param[in] view         TBD
    //! @param[in] proj         TBD
    //! @param[in] camPos       TBD
    //! @param[in] screenNear   TBD
    //! @param[in] screenFar    TBD
    //! @param[in] zOffset      TBD
    //---------------------------------------------------------------------------
    void Set(   nw::math::MTX34* view,
                nw::math::MTX44* proj,
                nw::math::VEC3*  camPos,
                f32 screenNear,
                f32 screenFar,
                f32 zOffset )
    {
        // Fovy 関連のパラメータは 1.0f を入れておけば無害
        Set( view, proj, camPos, screenNear, screenFar, 1.0f, 1.0f, zOffset );
    }

    //---------------------------------------------------------------------------
    //! @brief                  TBD
    //! @param[in] view         TBD
    //! @param[in] proj         TBD
    //! @param[in] camPos       TBD
    //! @param[in] screenNear   TBD
    //! @param[in] screenFar    TBD
    //! @param[in] baseFovy     TBD
    //! @param[in] currentFovy  TBD
    //---------------------------------------------------------------------------
    void Set(   nw::math::MTX34* view,
                nw::math::MTX44* proj,
                nw::math::VEC3*  camPos,
                f32              screenNear,
                f32              screenFar,
                f32              baseFovy,
                f32              currentFovy )
    {
        // Fovy 関連のパラメータは 1.0f を入れておけば無害
        Set( view, proj, camPos, screenNear, screenFar, baseFovy, currentFovy, DefaultZOffset );
    }

    //---------------------------------------------------------------------------
    //! @brief                  TBD
    //! @param[in] view         TBD
    //! @param[in] proj         TBD
    //! @param[in] camPos       TBD
    //! @param[in] screenNear   TBD
    //! @param[in] screenFar    TBD
    //! @param[in] baseFovy     TBD
    //! @param[in] currentFovy  TBD
    //! @param[in] zOffset      TBD
    //---------------------------------------------------------------------------
    void Set(   nw::math::MTX34* view,
                nw::math::MTX44* proj,
                nw::math::VEC3*  camPos,
                f32              screenNear,
                f32              screenFar,
                f32              baseFovy,
                f32              currentFovy,
                f32              zOffset )
    {
        viewParam2.x = baseFovy;
        viewParam2.y = currentFovy;
        viewParam2.z = zOffset;
        viewParam2.w = 0.0f;

        viewMat = nw::math::MTX44( *view );
        projMat = *proj;
        vpMat.SetMult( projMat, viewMat );

        nw::math::MTX34 billboardMtx;
        MTX34Transpose( &billboardMtx, *view );
        bldMat.m[0][0] = billboardMtx.m[0][0];
        bldMat.m[0][1] = billboardMtx.m[0][1];
        bldMat.m[0][2] = billboardMtx.m[0][2];
        bldMat.m[0][3] = billboardMtx.m[0][3];
        bldMat.m[1][0] = billboardMtx.m[1][0];
        bldMat.m[1][1] = billboardMtx.m[1][1];
        bldMat.m[1][2] = billboardMtx.m[1][2];
        bldMat.m[1][3] = billboardMtx.m[1][3];
        bldMat.m[2][0] = billboardMtx.m[2][0];
        bldMat.m[2][1] = billboardMtx.m[2][1];
        bldMat.m[2][2] = billboardMtx.m[2][2];
        bldMat.m[2][3] = billboardMtx.m[2][3];
        bldMat.m[3][0] = 0.0f;
        bldMat.m[3][1] = 0.0f;
        bldMat.m[3][2] = 0.0f;
        bldMat.m[3][3] = 1.0f;

        eyeVec.x = billboardMtx.m[0][2];
        eyeVec.y = billboardMtx.m[1][2];
        eyeVec.z = billboardMtx.m[2][2];

        eyePos.x = camPos->x;
        eyePos.y = camPos->y;
        eyePos.z = camPos->z;

        viewParam.x = screenNear;
        viewParam.y = screenFar;
        viewParam.z = screenNear * screenFar;
        viewParam.w = screenFar  - screenNear;
    }
};

} // namespace eft2
} // namespace nw
