﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nw   {
namespace eft2 {

class System;

//---------------------------------------------------------------------------
//! @brief  テンポラリユニフォームブロックバッファ
//---------------------------------------------------------------------------
class UniformBlockBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    UniformBlockBuffer()
    {
        m_UniformBlockBuffer        = NULL;
        m_UniformBlockBufferSize    = 0;
        m_IsValidate                = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  テンポラリユニフォームブロックバッファを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    void* GetBuffer() const { return m_UniformBlockBuffer; }

    //---------------------------------------------------------------------------
    //! @brief  テンポラリユニフォームブロックバッファサイズを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetBufferSize() const { return m_UniformBlockBufferSize; }

    //---------------------------------------------------------------------------
    //! @brief  有効化されているかどうか？
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsValidate() const { return m_IsValidate; }

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを有効化する
    //! @param[in] endianSwap     TBD
    //---------------------------------------------------------------------------
    void Validate( bool endianSwap = true );

    //---------------------------------------------------------------------------
    //! @brief  Cpu / Gpu キャッシュフラッシュの有効/無効を切り替えます。
    //! @param[in] cpuCacheFlush     TBD
    //! @param[in] gpuCacheFlush     TBD
    //---------------------------------------------------------------------------
    static void SetCpuGpuChacheFlushSetting( bool cpuCacheFlush, bool gpuCacheFlush )
    {
        g_IsCpuCacheFlush = cpuCacheFlush;
        g_IsGpuCacheFlush = gpuCacheFlush;
    }

    //---------------------------------------------------------------------------
    //! @brief  エンディアンスワップの有効/無効を切り替えます。
    //! @param[in] endianSwap     TBD
    //---------------------------------------------------------------------------
    static void SetEndianSwapSetting( bool endianSwap )
    {
        g_IsEndianSwap = endianSwap;
    }

protected:
    void*           m_UniformBlockBuffer;       //!< ユニフォームブロックバッファ
    u32             m_UniformBlockBufferSize;   //!< ユニフォームブロックバッファサイズ
    bool            m_IsValidate;               //!< ユニフォームブロックバッファが有効かどうか

    static bool     g_IsCpuCacheFlush;          //!< Cpuキャッシュフラッシュを行うかどうか
    static bool     g_IsGpuCacheFlush;          //!< Gpuキャッシュフラッシュを行うかどうか
    static bool     g_IsEndianSwap;             //!< エンディアンスワップを行うかどうか
};

//---------------------------------------------------------------------------
//! @brief  静的ユニフォームブロックバッファ
//!
//!         中身が書き換わらないようなユニフォームブロックに利用。
//---------------------------------------------------------------------------
class StaticUniformBlockBuffer : public UniformBlockBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    StaticUniformBlockBuffer() : UniformBlockBuffer(){}

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを設定する
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //---------------------------------------------------------------------------
    void Set( void* buffer, u32 bufferSize )
    {
        m_UniformBlockBuffer        = buffer;
        m_UniformBlockBufferSize    = bufferSize;
        m_IsValidate                = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを有効化する
    //---------------------------------------------------------------------------
    void Validate();
};

//---------------------------------------------------------------------------
//! @brief  動的ユニフォームブロックバッファ
//!
//!         毎フレーム中身が書き換わるようなユニフォームブロックに利用。
//---------------------------------------------------------------------------
class DynamicUniformBlockBuffer : public UniformBlockBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    DynamicUniformBlockBuffer() : UniformBlockBuffer(){}

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを設定する
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //---------------------------------------------------------------------------
    void Set( void* buffer, u32 bufferSize )
    {
        m_UniformBlockBuffer        = buffer;
        m_UniformBlockBufferSize    = bufferSize;
        m_IsValidate                = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを無効化する
    //---------------------------------------------------------------------------
    void InValidate();
};

//---------------------------------------------------------------------------
//! @brief  テンポラリユニフォームブロックバッファ
//!
//!         テンポラリ領域に書き出すユニフォームブロックに利用。
//---------------------------------------------------------------------------
class TemporaryUniformBlockBuffer : public UniformBlockBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    TemporaryUniformBlockBuffer() : UniformBlockBuffer(){}

    //---------------------------------------------------------------------------
    //! @brief  テンポラリユニフォームブロックバッファを確保する
    //! @param[in] system     TBD
    //! @param[in] size     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    void* Alloc( System* system, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックバッファを設定する
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //---------------------------------------------------------------------------
    void Set( void* buffer, u32 bufferSize )
    {
        m_UniformBlockBuffer        = buffer;
        m_UniformBlockBufferSize    = bufferSize;
        m_IsValidate                = false;
    }
};

//---------------------------------------------------------------------------
//! @brief  ユニフォームブロックID
//---------------------------------------------------------------------------
class UniformBlockID
{
public:
    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックの初期化
    //!
    //! @param[in]    bindPoint ユニフォームブロックのバインドポイント。
    //! @param[in]    name      ユニフォームブロックの名前。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Initialize( u32 bindPoint, const char* name );

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックへのバッファセット
    //! @param[in] uniformBlockBuffer     TBD
    //! @param[in] vertex     TBD
    //! @param[in] fragment     TBD
    //---------------------------------------------------------------------------
    void BindUniformBlock( const UniformBlockBuffer* uniformBlockBuffer, bool vertex, bool fragment );

    //---------------------------------------------------------------------------
    //! @brief  ユニフォームブロックへのバッファセット
    //! @param[in] system     TBD
    //! @param[in] buffer     TBD
    //! @param[in] bufferSize     TBD
    //! @param[in] endianSwap     TBD
    //! @param[in] vertex     TBD
    //! @param[in] fragment     TBD
    //---------------------------------------------------------------------------
    void BindUniformBlock( System* system, const void* buffer, u32 bufferSize, bool endianSwap, bool vertex, bool fragment );

private:
#if EFT_OPENGL
    GLint   m_BufferBindPoint;  //!< TBD
    GLuint  m_BufferId;         //!< TBD
#endif
#if EFT_GX2
    u32     m_BufferBindPoint;  //!< TBD
#endif
};


} // namespace nw
} // namespace eft2
