﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_Enum.h>

namespace nw   {
namespace eft2 {

struct ShaderResource;
struct ResTextureSampler;

//---------------------------------------------------
//! @brief サンプラーをバインド
//! @param[in] sampler     TBD
//! @param[in] slot     TBD
//! @param[in] vertLoc     TBD
//! @param[in] fragLoc     TBD
//! @return TBD
//---------------------------------------------------
bool BindSampler( TextureSamplerInterface sampler, s32 slot, u32 vertLoc, u32 fragLoc );

//---------------------------------------------------------------------------
//! @brief      テクスチャサンプラロケーション
//---------------------------------------------------------------------------
struct TextureSmplerLoc
{
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] shaderRes     TBD
    //! @param[in] name     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Initialize( ShaderResource* shaderRes, const char* name );

    //---------------------------------------------------------------------------
    //! @brief  フラグメント サンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetFragmentSmpLocation() const
    {
        return fragSamplerLocation;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点 サンプラロケーションを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetVertexSmpLocation() const
    {
        return vertSamplerLocation;
    }

    //---------------------------------------------------------------------------
    //! @brief  無効かどうか。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsValid() const
    {
        if ( fragSamplerLocation == EFT_INVALID_TEXTURE_SAMPLER ) return false;
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate()
    {
        fragSamplerLocation = static_cast<u32>( EFT_INVALID_TEXTURE_SAMPLER );
    }

private:
    u32 fragSamplerLocation;                        //!< フラグメントシェーダテクスチャサンプラロケーション
    u32 vertSamplerLocation;                        //!< 頂点シェーダテクスチャサンプラロケーション
};


//---------------------------------------------------------------------------
//! @brief  テクスチャサンプラー
//---------------------------------------------------------------------------
struct TextureSampler
{
public:
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] resSampler     TBD
    //---------------------------------------------------------------------------
    void Initialize( ResTextureSampler* resSampler );

    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] filter     TBD
    //! @param[in] wrapU     TBD
    //! @param[in] wrapV     TBD
    //---------------------------------------------------------------------------
    void Initialize( TextureFilterMode filter, TextureWrapMode wrapU, TextureWrapMode wrapV );

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief  有効かどうか
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsValidate();

    //---------------------------------------------------------------------------
    //! @brief  ResTextureParamからセットアップ
    //! @param[in] sampler     TBD
    //---------------------------------------------------------------------------
    void Setup( const ResTextureSampler* sampler );

    //---------------------------------------------------------------------------
    //! @brief  セットアップ
    //! @param[in] filter     TBD
    //! @param[in] wrapU     TBD
    //! @param[in] wrapV     TBD
    //---------------------------------------------------------------------------
    void Setup( TextureFilterMode filter, TextureWrapMode wrapU, TextureWrapMode wrapV );

    //---------------------------------------------------------------------------
    //! @brief  LODバイアスの設定
    //! @param[in] maxMip     TBD
    //! @param[in] bais     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool SetupLOD( f32 maxMip, f32 bais );

    //---------------------------------------------------------------------------
    //! @brief  バインド
    //! @param[in] fragLoc     TBD
    //! @param[in] vertLoc     TBD
    //! @param[in] slot     TBD
    //---------------------------------------------------------------------------
    void Bind( u32 fragLoc, u32 vertLoc, s32 slot ) const;

    //---------------------------------------------------------------------------
    //! @brief  テクスチャサンプラリソースを取得する
    //! @return TBD
    //---------------------------------------------------------------------------
    const ResTextureSampler* GetTextureSamplerRes() const
    {
        return textureSamplerRes;
    }


private:
#if EFT_OPENGL
    GLuint                  textureSampler;     //!< TBD
#endif
#if EFT_GX2
    GX2Sampler              textureSampler;     //!< TBD
#endif
    ResTextureSampler*      textureSamplerRes;  //!< TBD
};


} // namespace nw
} // namespace eft2
