﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>

namespace nw {
namespace eft2 {

struct Emitter;

//---------------------------------------------------------------------------
//! @brief ストリームアウト バッファ
//---------------------------------------------------------------------------
struct StreamOutAttributeBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief  バッファID
    //---------------------------------------------------------------------------
    enum
    {
        EFT_STREAM_OUT_ATTRB_0   = 0,
        EFT_STREAM_OUT_ATTRB_1   = 1,
        EFT_STREAM_OUT_ATTRB_MAX = 2,
    };

    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] size バッファサイズ
    //---------------------------------------------------------------------------
    void Initialize( u32 size );

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate();

    //---------------------------------------------------------------------------
    //! @brief  入力バッファをバインド
    //! @param[in] input TBD
    //! @param[in] flip TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindInBuffer( u32 input, u32 flip );

    //---------------------------------------------------------------------------
    //! @brief  入力バッファをバインド
    //! @param[in] input TBD
    //! @param[in] flip TBD
    //! @param[in] start TBD
    //! @param[in] end TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindInBuffer( u32 input, u32 flip, u32 start, u32 end );

    //---------------------------------------------------------------------------
    //! @brief  出力バッファをバインド
    //! @param[in] index TBD
    //! @param[in] flip TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool BindOutBuffer( u32 index, u32 flip );

    //---------------------------------------------------------------------------
    //! @brief  バインド解除
    //! @param[in] index インデックス
    //---------------------------------------------------------------------------
    void UnBind( u32 index );

    //---------------------------------------------------------------------------
    //! @brief  バッファサイズを取得
    //! @return バッファサイズ
    //---------------------------------------------------------------------------
    u32 GetBufferSize() const { return bufferSize; }

private:
    bool                        isError;                                        //!< エラー状態かどうか
    u32                         bufferSize;                                     //!< バッファサイズ

#if EFT_OPENGL
    //---------------------------------------------------------------------------
    //! @brief  入力バッファのハンドルを取得
    //! @return 入力バッファのハンドル
    //---------------------------------------------------------------------------
    GLuint GetInHandle()  { return uBuffer[EFT_STREAM_OUT_ATTRB_0]; }
    //---------------------------------------------------------------------------
    //! @brief  出力バッファのハンドルを取得
    //! @return 出力バッファのハンドル
    //---------------------------------------------------------------------------
    GLuint GetOutHandle() { return uBuffer[EFT_STREAM_OUT_ATTRB_1]; }

    GLuint                      uBuffer[EFT_STREAM_OUT_ATTRB_MAX];              //!< バッファID
#endif
#if EFT_GX2
    nw::math::VEC4*             pBuffer[EFT_STREAM_OUT_ATTRB_MAX];              //!< バッファ
    GX2StreamOutBuffer          streamOutBuf[EFT_STREAM_OUT_ATTRB_MAX];         //!< ストリームアウトバッファ
    GX2StreamOutContext         streamOutContext[EFT_STREAM_OUT_ATTRB_MAX];     //!< ストリームアウトコンテキスト
    u8                          outputBuffer;                                   //!< 出力されるバッファ
#endif
    friend Emitter;
public:
    void DumpSOBuffer( const char* text ) const;
};

}   // namespace nw
}   // namespace eft2
