﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_ShaderKey.h>
//#include <nw/eft/eft2_Heap.h>
//#include <nw/eft/eft2_Shader.h>

namespace nw   {
namespace eft2 {

class Heap;
struct Shader;
struct EmitterResource;

//---------------------------------------------------------------------------
//! @brief  シェーダマネージャ
//---------------------------------------------------------------------------
class ShaderManager
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    ShaderManager();

    //---------------------------------------------------------------------------
    //! @brief                  初期化処理
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] shaderMaxNum シェーダの最大数
    //---------------------------------------------------------------------------
    void Initialize( Heap* heap, u32 shaderMaxNum );

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //! @param[in] heap ヒープへのポインタ
    //---------------------------------------------------------------------------
    void Finalize( Heap* heap );

    //---------------------------------------------------------------------------
    //! @brief  シェーダを取得します。
    //! @param[in] index インデックス
    //! @return シェーダへのポインタ
    //---------------------------------------------------------------------------
    Shader* GetShader( u32 index );

    //---------------------------------------------------------------------------
    //! @brief  管理しているシェーダ数を取得します。
    //! @return 管理しているシェーダ数
    //---------------------------------------------------------------------------
    u32 GetShaderNum() { return m_ShaderNum; }

#ifdef EFT_OGL
    //---------------------------------------------------------------------------
    //! @brief                      エミッタリソースセットを追加する。
    //! @param[in] emResSet         EmitterResourceへのポインタ
    //! @param[in] addCompileDef    コンパイル時定義へのポインタ
    //! @param[in] delayCompile     遅延コンパイルのOn/Off
    //! @return                     TBD
    //---------------------------------------------------------------------------
    u32 AddEmitterResSet( const EmitterResource* emResSet, const char* addCompileDef, bool delayCompile );
#endif
#if EFT_GX2
    //---------------------------------------------------------------------------
    //! @brief              シェーダバイナリを設定する。
    //! @param[in] heap     ヒープへのポインタ
    //! @param[in] binary   バイナリへのポインタ
    //---------------------------------------------------------------------------
    void SetShaderBinary( Heap* heap, void* binary );

    //---------------------------------------------------------------------------
    //! @brief                  シェーダセットをシェーダ管理配列に追加する
    //! @param[in] heap         ヒープへのポインタ
    //! @param[in] vertexIdx    TBD
    //! @param[in] fragIdx      TBD
    //! @return                 TBD
    //---------------------------------------------------------------------------
    u32 AddShaderSet( Heap* heap, u32 vertexIdx, u32 fragIdx );

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダ数を所得する。
    //! @return 頂点シェーダ数
    //---------------------------------------------------------------------------
    u32 GetVertexShaderNum()    { return m_TotalVertexShader; }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダサイズを所得する。
    //! @return 頂点シェーダサイズ
    //---------------------------------------------------------------------------
    u32 GetVertexShaderSize()   { return m_TotalVertexShaderSize; }

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダ数を取得する。
    //! @return フラグメントシェーダ数
    //---------------------------------------------------------------------------
    u32 GetFragmentShaderNum()  { return m_TotalPixelShader; }

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダサイズを取得する。
    //! @return フラグメントシェーダサイズ
    //---------------------------------------------------------------------------
    u32 GetFragmentShaderSize() { return m_TotalPixelShaderSize; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダバイナリを取得する。
    //! @return シェーダバイナリへのポインタ
    //---------------------------------------------------------------------------
    void* GetShaderBinary()     { return m_ShaderBinary; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダバイナリをアンリロケート。
    //! @param[in] pData      シェーダバイナリ
    //---------------------------------------------------------------------------
    static void Unrelocate( void* pData );

#endif

private:
    u32                     m_ShaderNum;            //!< 管理しているシェーダ数
    u32                     m_ShaderMaxNum;         //!< 管理可能なシェーダ数
    Shader*                 m_ShaderArray;          //!< シェーダ配列
#ifdef EFT_OGL
    ShaderKey               m_ShaderKey;            //!< シェーダキー
#endif
#if EFT_GX2
    void*                   m_ShaderBinary;         //!< シェーダバイナリ
    u32                     m_TotalVertexShader;    //!< 総頂点シェーダ数
    u32                     m_TotalPixelShader;     //!< 総フラグメントシェーダ数
    u32                     m_TotalVertexShaderSize;    //!< 総頂点シェーダサイズ
    u32                     m_TotalPixelShaderSize;     //!< 総フラグメントシェーダサイズ
#endif
};

} // namespace eft2
} // namespace nw
