﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_EmitterRes.h>
#include <nw/eft/eft2_ShaderManager.h>
#include <nw/eft/eft2_ShaderRes.h>
#include <nw/eft/eft2_Shader.h>

namespace nw   {
namespace eft2 {

class System;
struct BinaryData;

//---------------------------------------------------------------------------
//! @brief  リソース用ヒープ
//---------------------------------------------------------------------------
class ResourceHeap : public nw::eft2::Heap
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    ResourceHeap()
    {
        m_BaseHeap       = NULL;
        m_AllocatedSize  = 0;
        m_AllocatedCount = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //! @param[in] heap ヒープへのポインタ
    //---------------------------------------------------------------------------
    explicit ResourceHeap( nw::eft2::Heap* heap )
    {
        m_BaseHeap       = heap;
        m_AllocatedSize  = 0;
        m_AllocatedCount = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief  ヒープを設定します。
    //! @param[in] heap ヒープへのポインタ
    //---------------------------------------------------------------------------
    void SetHeap( nw::eft2::Heap* heap )
    {
        m_BaseHeap = heap;
    }

    //---------------------------------------------------------------------------
    //! @brief                  メモリを確保します。
    //! @param[in] size         確保するメモリサイズ
    //! @param[in] alignment    アライメントサイズ
    //! @return                 確保した領域の先頭アドレス
    //---------------------------------------------------------------------------
    virtual void* Alloc( u32 size, s32 alignment = nw::eft2::Heap::EFT_HEAP_DEAFULT_ALIGNMENT )
    {
        void* ptr = m_BaseHeap->Alloc( size, alignment );
        m_AllocatedSize += size;
        m_AllocatedCount++;
        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief  メモリを解放します。
    //! @param[in] ptr 解放する領域の先頭アドレス
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr )
    {
        EFT_ASSERT( ptr );
        EFT_ASSERT( m_AllocatedCount > 0 );
        m_AllocatedCount--;
        return m_BaseHeap->Free( ptr );
    }

    //---------------------------------------------------------------------------
    //! @brief  メモリ確保サイズを取得します。
    //! @return 確保済みのメモリサイズ
    //---------------------------------------------------------------------------
    u32 GetAllocatedSize() const
    {
        return m_AllocatedSize;
    }

    //---------------------------------------------------------------------------
    //! @brief  メモリ確保回数を取得します。
    //! @return メモリ確保回数
    //---------------------------------------------------------------------------
    u32 GetAllocatedCount() const
    {
        return m_AllocatedCount;
    }

private:
    nw::eft2::Heap* m_BaseHeap;         //!< ヒープ
    u32             m_AllocatedSize;    //!< 確保済みメモリサイズ
    u32             m_AllocatedCount;   //!< 確保回数
};

//---------------------------------------------------
//! @brief エフェクトリソース管理クラス
//---------------------------------------------------
class Resource
{
public:
    //---------------------------------------------------------------------------
    //! @brief        エミッタセットリソースを取得します。
    //!
    //! @param[in]    emitterSetID          エミッタセットID。
    //! @return       エミッタセットリソース。取得に失敗した場合は、NULLが返ります。
    //---------------------------------------------------------------------------
    EmitterSetResource* GetEmitterSetResource( s32 emitterSetID ) const
    {
        if ( emitterSetID >= static_cast<s32>( m_EmitterSetCount ) ) { return NULL; }

        if ( m_BindEmitterSetResArray[emitterSetID] )
        {
            return m_BindEmitterSetResArray[emitterSetID];
        }
        return &m_EmitterSetResArray[emitterSetID];
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットリソースを取得します。
    //!
    //! @param[in]    emitterSetName          エミッタセット名。
    //! @return       エミッタセットリソース。取得に失敗した場合は、NULLが返ります。
    //---------------------------------------------------------------------------
    EmitterSetResource* GetEmitterSetResource( const char* emitterSetName ) const
    {
        s32 emitterSetID = SearchEmitterSetID( emitterSetName );
        return GetEmitterSetResource( emitterSetID );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットリソースを取得します。
    //!
    //! @param[in]    emitterSetID          エミッタセットID。
    //! @return       エミッタセットリソース。取得に失敗した場合は、NULLが返ります。
    //---------------------------------------------------------------------------
    EmitterSetResource* GetEmitterSetResourceRom( s32 emitterSetID ) const
    {
        if ( emitterSetID >= static_cast<s32>( m_EmitterSetCount ) ) { return NULL; }
        return &m_EmitterSetResArray[emitterSetID];
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットリソースを取得します。
    //!
    //! @param[in]    emitterSetName          エミッタセット名。
    //! @return       エミッタセットリソース。取得に失敗した場合は、NULLが返ります。
    //---------------------------------------------------------------------------
    EmitterSetResource* GetEmitterSetResourceRom( const char* emitterSetName ) const
    {
        s32 emitterSetID = SearchEmitterSetID( emitterSetName );
        return GetEmitterSetResourceRom( emitterSetID );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定EmitterSetが無限寿命、または、ループエフェクトかを取得する。
    //!
    //! @param[in]    emitterSetID          エミッタセット名。
    //! @return       無限寿命、または、ループエフェクトかどうか
    //---------------------------------------------------------------------------
    bool IsNeedFade( s32 emitterSetID ) const;

    //---------------------------------------------------------------------------
    //! @brief        エミッタセット名の取得を行います。
    //!
    //! @param[in]    emitterSetID          エミッタセットID。
    //! @return       エミッタセット名。
    //---------------------------------------------------------------------------
    const char* GetEmitterSetName( s32 emitterSetID ) const
    {
        EmitterSetResource* setRes = GetEmitterSetResource( emitterSetID );
        if ( !setRes ) return NULL;
        return setRes->emitterSetRes->name;
    }

    //---------------------------------------------------------------------------
    //! @brief  リソースに含まれるエミッタセット数を取得します。
    //! @return リソースに含まれるエミッタセット数
    //---------------------------------------------------------------------------
    u32 GetNumEmitterSet()  const { return GetEmitterSetCount(); }
    //---------------------------------------------------------------------------
    //! @brief  リソースに含まれるエミッタセット数を取得します。
    //! @return リソースに含まれるエミッタセット数
    //---------------------------------------------------------------------------
    u32 GetEmitterSetNum()  const { return GetEmitterSetCount(); }
    //---------------------------------------------------------------------------
    //! @brief  リソースに含まれるエミッタセット数を取得します。
    //! @return リソースに含まれるエミッタセット数
    //---------------------------------------------------------------------------
    u32 GetEmitterSetCount()const { return m_EmitterSetCount; }

    //---------------------------------------------------------------------------
    //! @brief                  指定エミッタセット内に含まれるエミッタ数を取得します。
    //! @param[in] emitterSetID エミッタセットID
    //! @return                 指定エミッタセット内に含まれるエミッタ数
    //---------------------------------------------------------------------------
    u32 GetNumEmitter( s32 emitterSetID ) const { return GetEmitterCount( emitterSetID ); }
    //---------------------------------------------------------------------------
    //! @brief                  指定エミッタセット内に含まれるエミッタ数を取得します。
    //! @param[in] emitterSetID エミッタセットID
    //! @return                 指定エミッタセット内に含まれるエミッタ数
    //---------------------------------------------------------------------------
    u32 GetEmitterNum( s32 emitterSetID ) const { return GetEmitterCount( emitterSetID ); }
    //---------------------------------------------------------------------------
    //! @brief                  指定エミッタセット内に含まれるエミッタ数を取得します。
    //! @param[in] emitterSetID エミッタセットID
    //! @return                 指定エミッタセット内に含まれるエミッタ数
    //---------------------------------------------------------------------------
    u32 GetEmitterCount( s32 emitterSetID ) const
    {
        EmitterSetResource* setRes = GetEmitterSetResource( emitterSetID );
        if ( !setRes ) return 0;
        return setRes->emitterNum;
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタセットIDの検索を行います。
    //!
    //! @param[in]    emitterSetName          検索するエミッタセット名。
    //! @return       エミッタセットID（無ければ EFT_INVALID_EMITTER_SET_ID が返ります）。
    //---------------------------------------------------------------------------
    s32 SearchEmitterSetID( const char* emitterSetName ) const;
    s32 SearchEmitterSetId( const char* emitterSetName ) const
    {
        return SearchEmitterSetID( emitterSetName );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタIDの検索を行います。
    //!               この関数は非常に低速なため、リアルタイムではなるべく使用しないで下さい。
    //!
    //! @param[in]    emitterSetName          検索するエミッタセット名。
    //! @param[in]    emitterName      検索するエミッタ名。
    //! @return       エミッタセットID（無ければ EFT_INVALID_EMITTER_ID が返ります）。
    //---------------------------------------------------------------------------
    s32 SearchEmitterID( const char* emitterSetName , const char* emitterName ) const
    {
        s32 emSetID = SearchEmitterSetID( emitterSetName );
        if ( emSetID == EFT_INVALID_EMITTER_SET_ID ) return EFT_INVALID_EMITTER_ID;
        return SearchEmitterID( emSetID, emitterName );
    }
    s32 SearchEmitterId( const char* emitterSetName , const char* emitterName ) const
    {
        return SearchEmitterID( emitterSetName, emitterName );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタIDの検索を行います。この関数は非常に低速なため、リアルタイムではなるべく使用しないで下さい。
    //!
    //! @param[in]    emitterSetID     検索するエミッタセットID。
    //! @param[in]    emitterName      検索するエミッタ名。
    //!
    //! @return       エミッタセットID（無ければEFT_INVALID_EMITTER_ID が返ります）。
    //---------------------------------------------------------------------------
    s32 SearchEmitterID( s32 emitterSetID , const char* emitterName ) const
    {
        EmitterSetResource* setRes = GetEmitterSetResource( emitterSetID );
        if ( !setRes ) return EFT_INVALID_EMITTER_ID;

        for( u32 i=0 ; i < setRes->emitterNum ; i++ )
        {
            if ( strcmp( emitterName, setRes->emitterResSet[i].emitterData->name ) == 0 )
            {
                return i;
            }
        }
        return EFT_INVALID_EMITTER_ID;
    }
    s32 SearchEmitterId( s32 emitterSetID , const char* emitterName ) const
    {
        return SearchEmitterID( emitterSetID, emitterName );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタIDの検索を行います。この関数は非常に低速なため、リアルタイムではなるべく使用しないで下さい。
    //!
    //! @param[in]    emitterSetID     検索するエミッタセットID。
    //! @param[in]    emitterName      検索するエミッタ名。
    //! @param[in]    emitterIdx       検索するエミッタIdx(同名エミッタが存在する場合に利用)。
    //!
    //! @return       エミッタセットID（無ければ EFT_INVALID_EMITTER_ID が返ります）。
    //---------------------------------------------------------------------------
    s32 SearchEmitterID( s32 emitterSetID , const char* emitterName, u32 emitterIdx ) const
    {
        EmitterSetResource* setRes = GetEmitterSetResource( emitterSetID );
        if ( !setRes ) return EFT_INVALID_EMITTER_ID;

        for( u32 i=0 ; i < setRes->emitterNum ; i++ )
        {
            if ( strcmp( emitterName, setRes->emitterResSet[i].emitterData->name ) == 0 )
            {
                if ( i == emitterIdx )
                {
                    return i;
                }
            }
        }
        return EFT_INVALID_EMITTER_ID;
    }
    s32 SearchEmitterId( s32 emitterSetID , const char* emitterName, u32 emitterIdx ) const
    {
        return SearchEmitterID( emitterSetID, emitterName, emitterIdx );
    }

    //------------------------------------------------------------------------------
    //! @brief  リソースIDを取得します。
    //! @return リソースID
    //------------------------------------------------------------------------------
    u32 GetResourceID() const { return m_ResourceID; }
    u32 GetResourceId() const { return m_ResourceID; }

    //---------------------------------------------------------------------------
    //! @brief  チャイルドエミッタが存在するかどうか。
    //! @param[in] targetSetID エミッタセットID
    //! @return チャイルドエミッタが存在するかどうか
    //---------------------------------------------------------------------------
    bool IsExistChildEmitter( s32 targetSetID );

    //------------------------------------------------------------------------------
    //! @brief  シェーダを取得します。
    //! @param[in] index インデックス値
    //! @return シェーダへのポインタ
    //------------------------------------------------------------------------------
    Shader* GetShader( u32 index ) { return m_ShaderManager.GetShader( index ); }

    //---------------------------------------------------------------------------
    //! @brief  バイナリの先頭を取得します。
    //! @return バイナリの先頭アドレス
    //---------------------------------------------------------------------------
    void* GetBinaryTop() { return m_BinaryTop; }

    //---------------------------------------------------------------------------
    //! @brief  プロジェクト名を取得します。
    //! @return プロジェクト名
    //---------------------------------------------------------------------------
    const char* GetProjectName() const
    {
        return GetBinaryName();
    }

    //---------------------------------------------------------------------------
    //! @brief  バイナリ名を取得します。
    //! @return バイナリ名
    //---------------------------------------------------------------------------
    const char* GetBinaryName() const
    {
        ResHeader* header = reinterpret_cast<ResHeader*>( m_BinaryTop );
        return reinterpret_cast<const char*>( header->name );
    }

    //---------------------------------------------------------------------------
    //! @brief  ストリームアウトシェーダを取得します。
    //! @return ストリームアウトシェーダへのポインタ
    //---------------------------------------------------------------------------
    Shader* GetStreamOutShader() { return &m_StreamOutShader; }

    //---------------------------------------------------------------------------
    //! @briefprivate targetSetIDで指定されるEmitterSetリソースを引数のリソースにバインドします。
    //! @param[in] targetSetID エミッタセットID
    //! @param[in] emitterSetResource エミッタセットリソースへのポインタ
    //! @return 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool BindResource( s32 targetSetID, EmitterSetResource* emitterSetResource );

    //---------------------------------------------------------------------------
    //! @briefprivate targetSetIDで指定されるエミッタセットリソースの繋ぎ換えを解除します。
    //! @param[in] targetSetID エミッタセットID
    //! @return 成功した場合trueを返す。
    //---------------------------------------------------------------------------
    bool UnbindResource( s32 targetSetID );

    //---------------------------------------------------------------------------
    //! @brief                  エミッタの名前（バインドしても変化しない）を取得します。
    //! @param[in] emitterSetID エミッタセットID
    //! @param[in] emitterID    エミッタID
    //! @return                 エミッタの名前
    //---------------------------------------------------------------------------
    const char* GetEmitterNameROM( int emitterSetID, s32 emitterID ) const
    {
        EmitterSetResource* setRes = GetEmitterSetResourceRom( emitterSetID );
        if ( !setRes ) return NULL;
        if ( emitterID >= static_cast<s32>( setRes->emitterNum ) ) return NULL;

        return  setRes->emitterResSet[emitterID].emitterData->name;
    }

#if EFT_OPENGL
    //---------------------------------------------------------------------------
    //! @brief  シェーダコードを設定します。
    //---------------------------------------------------------------------------
    void SetShaderCode()
    {
        // シェーダコードを設定
        ShaderResource::SetShaderCode( m_VshArg, m_FshArg );
    }
#endif

#if EFT_GX2

    //---------------------------------------------------------------------------
    //! @brief  シェーダマネージャーを取得。
    //!@return  シェーダマネージャーのポインタ
    //---------------------------------------------------------------------------
    ShaderManager* GetShaderManager() { return &m_ShaderManager; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダバイナリをアンリロケート。
    //! @param[in] pData      シェーダバイナリ
    //---------------------------------------------------------------------------
    static void Unrelocate( void* pData );
#endif

public:
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //!
    //! @param[in]    heap              ヒープ。
    //! @param[in]    bin               エフェクトバイナリ。
    //! @param[in]    resourceID        リソースID。
    //! @param[in]    system            遅延コンパイル( Windows版のみ動作 )。
    //! @param[in]    delayCompile      遅延コンパイル( Windows版のみ動作 )。
    //! @param[in]    residentResource  共通で利用されるテクスチャを含んだリソース。
    //!
    //! @return       生成されたエフェクトリソースのインスタンス。
    //---------------------------------------------------------------------------
    Resource( Heap *heap, void *bin, u32 resourceID, System* system, bool delayCompile = false, Resource* residentResource = NULL );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトリソースの破棄を行います。
    //---------------------------------------------------------------------------
    virtual ~Resource();

    //@}

    //---------------------------------------------------------------------------
    //! @brief        エフェクトリソースの初期化を行います。
    //! @param[in]    heap          ヒープ。
    //! @param[in]    bin           エフェクトバイナリ。
    //! @param[in]    resourceID    リソースID。
    //! @param[in]    delayCompile  遅延コンパイル(Windows版のみ動作)。
    //---------------------------------------------------------------------------
    void Initialize( Heap *heap, void *bin, u32 resourceID, bool delayCompile );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトリソースの初期化を行います。
    //! @param[in]    heap          ヒープ。
    //---------------------------------------------------------------------------
    void Finalize( Heap *heap );

    //---------------------------------------------------------------------------
    //! @brief        リソースサイズの詳細をログ出力します。
    //---------------------------------------------------------------------------
    void OutputResourceInfo();

    //---------------------------------------------------------------------------
    //! @brief        リソースの削除フラグを立てます。
    //---------------------------------------------------------------------------
    void SetKillMark() { m_IsAlive = false; }

    //---------------------------------------------------------------------------
    //! @brief  リソースの生存フラグを取得します。
    //! @return 生存フラグ
    //---------------------------------------------------------------------------
    bool IsAlive() const { return m_IsAlive; }

private:
    //------------------------------------------------------------------------------
    //! @brief              バイナリをトレースする
    //! @param[in] topData  バイナリのトップアドレス
    //------------------------------------------------------------------------------
    void Trace( BinaryData* topData );

    //------------------------------------------------------------------------------
    //! @brief          EmitterSetArray バイナリをトレースする
    //! @param[in] data EmitterSetArray バイナリのトップアドレス
    //------------------------------------------------------------------------------
    void TraceEmitterSetArray( BinaryData* data );

    //------------------------------------------------------------------------------
    //! @brief          TextureArray バイナリをトレースする
    //! @param[in] data TextureArray バイナリのトップアドレス
    //------------------------------------------------------------------------------
    void TraceTextureArray( BinaryData* data );

    //------------------------------------------------------------------------------
    //! @brief          PrimitiveArray バイナリをトレースする
    //! @param[in] data PrimitiveArray バイナリのトップアドレス
    //------------------------------------------------------------------------------
    void TracePrimitiveArray( BinaryData* data );

    //------------------------------------------------------------------------------
    //! @brief                  ShaderArray バイナリをトレースする
    //! @param[in] data         ShaderArray バイナリのトップアドレス
    //! @param[in] shaderNum    シェーダ数
    //------------------------------------------------------------------------------
    void TraceShaderArray( BinaryData* data, u32 shaderNum );

#ifdef EFT_ONLINE_SHADER_COMPILE
    //------------------------------------------------------------------------------
    //! @brief          ShaderCodeArray をトレースする
    //! @param[in] data ShaderCodeArrayのトップアドレス
    //------------------------------------------------------------------------------
    void TraceShaderCodeArray( BinaryData* data );
#endif

    //------------------------------------------------------------------------------
    //! @brief                  EmitterSetResource を初期化する
    //! @param[in] emSetRes     エミッタセットリソースへのポインタ
    //! @param[in] delayCompile 遅延コンパイルの有無
    //------------------------------------------------------------------------------
    void InitializeEmitterSetResource( EmitterSetResource* emSetRes, bool delayCompile );

    //------------------------------------------------------------------------------
    //! @brief                  EmitterResource を初期化する
    //! @param[in] emResSet     エミッタリソースへのポインタ
    //! @param[in] resEmitter   バイナリへのポインタ
    //! @param[in] idx          インデックス
    //------------------------------------------------------------------------------
    void InitializeEmitterResource( EmitterResource* emResSet, BinaryData* resEmitter, u32 idx );

    //------------------------------------------------------------------------------
    //! @brief  EmitterResource の終了処理をする
    //! @param[in] emRes エミッタリソースへのポインタ
    //------------------------------------------------------------------------------
    void FinalizeEmitterResource( EmitterResource* emRes );

    ////---------------------------------------------------------------------------
    ////! @brief  リソースヒープからメモリを確保します。
    ////---------------------------------------------------------------------------
    //void* AllocFromResourceHeap( u32 size, u32 align = 128 )
    //{
    //    u32 allocSize = nw::ut::RoundUp( size, 32 );
    //    void* ptr = m_ResHeap.Alloc( allocSize, align );
    //    EFT_NULL_ASSERT( ptr );
    //    m_AllocedSize += size;
    //    return ptr;
    //}

    ////---------------------------------------------------------------------------
    ////! @brief  リソースヒープからメモリを解放します。
    ////---------------------------------------------------------------------------
    //void FreeFromResourceHeap( void* ptr )
    //{
    //    m_ResHeap.Free( ptr );
    //}

#ifdef EFT_OGL
public:
    //------------------------------------------------------------------------------
    //! @brief  頂点シェーダコードを取得
    //! @return 頂点シェーダコード
    //------------------------------------------------------------------------------
    const ShaderCodeArg& GetVertexShaderCode()   const { return m_VshArg; }
    //------------------------------------------------------------------------------
    //! @brief  フラグメントシェーダコードを取得
    //! @return フラグメントシェーダコード
    //------------------------------------------------------------------------------
    const ShaderCodeArg& GetFragmentShaderCode() const { return m_FshArg; }
#endif

private:
    //------------------------------------------------------------------------------
    //! @brief          テクスチャを検索する
    //! @param[in] guid テクスチャのGUID
    //! @return T       テクスチャリソースへのポインタ
    //------------------------------------------------------------------------------
    TextureResource* GetTexture( u64 guid );

    //------------------------------------------------------------------------------
    //! @brief              プリミティブを検索する
    //! @param[in] index    プリミティブのGUID
    //! @return             プリミティブへのポインタ
    //------------------------------------------------------------------------------
    Primitive* GetPrimitive( u64 guid );


private:
    bool                        m_IsAlive;                              //!< 生存フラグ
    System*                     m_System;                               //!< システムクラス
    void*                       m_BinaryTop;                            //!< バイナリ先頭
    u32                         m_ResourceID;                           //!< リソースID
    u32                         m_EmitterSetCount;                      //!< 総エミッタセット数
    u32                         m_EmitterCount;                         //!< 総エミッタ数
    EmitterSetResource*         m_EmitterSetResArray;                   //!< エミッタセットリソース
    EmitterSetResource**        m_BindEmitterSetResArray;               //!< エミッタセットリソース
    u32                         m_PrimitiveCount;                       //!< プリミティブ数
    Primitive*                  m_PrimitiveArray;                       //!< プリミティブリソース
    u32                         m_TextureCount;                         //!< テクスチャ数
    TextureResource*            m_TextureArray;                         //!< テクスチャリソース
    void*                       m_TextureArrayPtr;                      //!< テクスチャリソースPtr
    bool                        m_DelayCompile;                         //!< 遅延コンパイル
    u32                         m_ShaderCount;                          //!< シェーダ数
    ShaderManager               m_ShaderManager;                        //!< シェーダマネージャ
    Shader                      m_StreamOutShader;                      //!< ストリームアウトシェーダ
    u32                         m_TotalResrouceSize;                    //!< 総バイナリサイズ
    u32                         m_EmitterResourceSize;                  //!< エミッタバイナリの合計サイズ
    u32                         m_TextureResourceSize;                  //!< テクスチャバイナリの合計サイズ
    u32                         m_PrimitiveResourceSize;                //!< プリミティブバイナリの合計サイズ
    u32                         m_ShaderCodeResourceSize;               //!< シェーダコードサイズ
    ResourceHeap                m_ResHeap;                              //!< リソース用のヒープ
    u32                         m_UboAllocedSize;                       //!< heap UBO             消費量サイズ
    u32                         m_ResTexAllocedSize;                    //!< heap テクスチャ      消費量サイズ
    u32                         m_ResPrimAllocedSize;                   //!< heap プリミティブ    消費量サイズ
    u32                         m_ResShaderBinAllocedSize;              //!< heap シェーダバイナリ消費量サイズ
    u32                         m_ResShaderCodeAllocedSize;             //!< heap シェーダコード  消費量サイズ
    u32                         m_ResEmitterSetAllocedSize;             //!< heap エミッタセット  消費量サイズ
    u32                         m_ShaderManagerAllocedSize;             //!< heap エミッタセット  消費量サイズ

#ifdef EFT_OGL
    nw::eft2::ShaderCodeArg     m_VshArg;                               //!< 頂点シェーダコード配列
    nw::eft2::ShaderCodeArg     m_FshArg;                               //!< フラグメントシェーダコード配列
#endif

    Resource*                   m_pResidentResource;                    //!< 共通テクスチャを含むリソース

    friend class System;
};

} // namespace eft2
} // namespace nw
