﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nw   {
namespace eft2 {

//!< 4294967296 == 0x100000000
#define EFT_RANDOM_MAX_INVERSE_F32 (1.f / 4294967296.f)
//!< 4294967296 == 0x100000000
#define EFT_RANDOM_MAX_INVERSE_F64 (1.0 / 4294967296.0)

//---------------------------------------------------
//! @brief      乱数発生クラス
//---------------------------------------------------
class RandomGenerator
{
public:
    //----------------------------------------------------------------
    /// @name   コンストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief        引数なしのコンストラクタです。
    //!
    //!               内部でinit()を呼び、乱数生成器を初期化します。
    //---------------------------------------------------------------------------
    RandomGenerator()
    {
        Init();
    }

    //---------------------------------------------------------------------------
    //! @brief        4バイトの数値をシードとして与えるコンストラクタです。
    //!
    //!               内部でinit(seed)を呼び、乱数生成器を初期化します。
    //!               同じシードを与えたRandomクラスは、同じ乱数列を生成します。
    //!
    //! @param[in]    seed    乱数のシード。
    //---------------------------------------------------------------------------
    explicit RandomGenerator( u32 seed )
    {
        Init( seed );
    }

    //---------------------------------------------------------------------------
    //! @brief        4バイトの数値を4つシードとして与えるコンストラクタです。
    //!
    //!               内部でinit(seed_0, seed_1, seed_2, seed_3)を呼び、乱数生成器を初期化します。
    //!               同じシードを与えたRandomクラスは、同じ乱数列を生成します。
    //!               全て0の値を与えることはできませんのでご注意ください。アサートで止まります。
    //!               製品版でアサート抜けしたときは、代替の値としてinit(0)で初期化します。
    //!
    //! @param[in]    seed_0    乱数のシード。
    //! @param[in]    seed_1    乱数のシード。
    //! @param[in]    seed_2    乱数のシード。
    //! @param[in]    seed_3    乱数のシード。
    //---------------------------------------------------------------------------
    RandomGenerator( u32 seed_0, u32 seed_1, u32 seed_2, u32 seed_3 )
    {
        Init( seed_0, seed_1, seed_2, seed_3 );
    }

    //@}

    //----------------------------------------------------------------
    /// @name   初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief        初期化。
    //!
    //!               現状はシード 0 で、乱数生成器を初期化します。
    //---------------------------------------------------------------------------
    void Init()
    {
        Init( 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief        引数として与えたシードで乱数生成器を初期化します。
    //!
    //!               XorShiftは初期化するために16バイトの数値を必要としますので、この形式では、
    //!               与えられた4バイトのシードを拡張することで初期化します。
    //!               シードの値は[0-0xffffffff]の任意の値で問題ありませんが、ワンパターンすぎる
    //!               シード（例えば、0,1,2,...を順番に与えるなど）は避けてください。
    //!
    //! @param[in]    seed    乱数のシード。
    //---------------------------------------------------------------------------
    void Init( u32 seed )
    {
        /**
         *  XorShiftはシードとして4つのu32を必要とする。
         *  この数値は全て0以外であれば何でも問題ないが、シードの決め方
         *  によっては、特に最初に生成する値が一様に分布しないことがある。
         *
         *  ここでは、シードとして0,1,2,...のような線形なパターンの値を
         *  与えたときでも、最初に生成する値がある程度一様に分布するような
         *  拡張の仕方を選んだ。
         *  もちろん、もっとランダムにシードを選ぶのが望ましい。
         */
        static const u32 a = 1812433253;
        mX = (seed ^ (seed >> 30)) * a + 1;
        mY = (mX ^ (mX >> 30)) * a + 2;
        mZ = (mY ^ (mY >> 30)) * a + 3;
        mW = (mZ ^ (mZ >> 30)) * a + 4;
    }

    //---------------------------------------------------------------------------
    //! @brief        引数として与えたシードで乱数生成器を初期化します。
    //!
    //!               この形式では、初期化するために必要な16バイトの数値を直接与えます。
    //!               全て0の値を与えることはできませんのでご注意ください。アサートで止まります。
    //!               製品版でアサート抜けしたときは、代替の値としてinit(0)で初期化します。
    //!
    //! @param[in]    seed_0    乱数のシード。
    //! @param[in]    seed_1    乱数のシード。
    //! @param[in]    seed_2    乱数のシード。
    //! @param[in]    seed_3    乱数のシード。
    //---------------------------------------------------------------------------
    void Init( u32 seed_0, u32 seed_1, u32 seed_2, u32 seed_3 )
    {
        if ( seed_0 == 0 && seed_1 == 0 && seed_2 == 0 && seed_3 == 0 )
        {
            //OutputError( "Seeds must not be all zero." );
            // 全て0は使用できない。アサート抜けしたときは別の決まった値で初期化する。
            Init(0);
        } else {
            mX = seed_0;
            mY = seed_1;
            mZ = seed_2;
            mW = seed_3;
        }
    }
    //@}

    //----------------------------------------------------------------
    /// @name   乱数の生成
    //----------------------------------------------------------------
    //@{

    //---------------------------------------------------------------------------
    //! @brief        乱数をu32の範囲で生成し、返します。
    //!
    //!               記号で表すと [0, 0xffffffff] の整数を返します。
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    u32 GetU32()
    {
        u32 t = (mX^(mX<<11));
        mX = mY;
        mY = mZ;
        mZ = mW;
        mW = (mW^(mW>>19))^(t^(t>>8));
        return mW;
    }

    //---------------------------------------------------------------------------
    //! @brief        0以上でceilより小さい整数の乱数を生成し、返します。
    //!
    //!               記号で表すと [0, ceil) の整数を返します。
    //!               ceilが0か1のときは、必ず0を返します。
    //!
    //! @param[in]    ceil  取得する乱数の天井値。
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    u32 GetU32( u32 ceil )
    {
        return static_cast<u32>(((u64)GetU32() * ceil) >> 32);
    }

    //---------------------------------------------------------------------------
    //! @brief        a以上でbより小さい整数の乱数を生成し、返します。
    //!
    //!               記号で表すと [a, b) の整数を返します。
    //!               bはaより大きい値である必要があります。
    //!               b==aのときは、例外的にaを返します。
    //!
    //! @param[in]    a  取得する乱数の最小値
    //! @param[in]    b  取得する乱数の天井値
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    s32 GetS32Range(s32 a, s32 b)
    {
//        EFT_ASSERT(b >= a);
        return (GetU32(b - a) + a);
    }

    //---------------------------------------------------------------------------
    //! @brief        0以上で1より小さい実数を生成し、float型で返します。
    //!
    //!               記号で表すと [0, 1) の実数を返します。
    //!               ceilが0か1のときは、必ず0を返します。
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f32 GetF32()
    {
        // 4294967296 == 0x100000000 である。コンパイルした時点で(1.f / 4294967296.f)が計算されるように、リテラルとして書いている。
        //return GetU32() * EFT_RANDOM_MAX_INVERSE_F32;
        // [1, 2)の単精度浮動小数点を作成し、1を引いて返す
        u32 float_binary = (GetU32() >> 9) | 0x3f800000;
        return (*(f32*)&float_binary) - 1.f;
    }

    //---------------------------------------------------------------------------
    //! @brief        0以上で1より小さい実数を生成し、float型で返します。
    //!
    //!               記号で表すと [0, ceil) の実数を返します。
    //!               ceilが0のときは、必ず0を返します。
    //!
    //! @param[in]    ceil  取得する乱数の天井値
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f32 GetF32( f32 ceil )
    {
        return GetF32() * ceil;
    }

    //---------------------------------------------------------------------------
    //! @brief        a以上でbより小さい実数の乱数を生成し、float型で返します。
    //!
    //!               記号で表すと [a, b) の実数を返します。
    //!               bはaより大きい値である必要があります。
    //!               b==aのときは、例外的にaを返します。
    //!
    //! @param[in]    a  取得する乱数の最小値
    //! @param[in]    b  取得する乱数の天井値
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f32 GetF32Range(f32 a, f32 b)
    {
        return ( GetF32(b - a) + a );
    }

    //---------------------------------------------------------------------------
    //! @brief        0以上で1より小さい実数を生成し、double型で返します。
    //!
    //!               記号で表すと [0, 1) の実数を返します。
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f64 GetF64()
    {
        // 4294967296 == 0x100000000 である。コンパイルした時点で(1.0 / 4294967296.0)が計算されるように、リテラルとして書いている。
        //return GetU32() * EFT_RANDOM_MAX_INVERSE_F64;
        // [1, 2)の倍精度浮動小数点を作成し、1を引いて返す
        u64 double_binary = (static_cast<u64>(GetU32()) << 20) | 0x3ff0000000000000LL;
        return (*(f64*)&double_binary) - 1.0;
    }

    //---------------------------------------------------------------------------
    //! @brief        0以上でceilより小さい実数を生成し、double型で返します。
    //!
    //!               記号で表すと [0, ceil) の実数を返します。
    //!               ceilが0のときは、必ず0を返します。
    //!
    //! @param[in]    ceil  取得する乱数の天井値
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f64 GetF64(f64 ceil)
    {
        return GetF64() * ceil;
    }

    //---------------------------------------------------------------------------
    //! @brief        a以上でbより小さい実数の乱数を生成し、double型で返します。
    //!
    //!               記号で表すと [a, b) の実数を返します。
    //!               bはaより大きい値である必要があります。
    //!               b==aのときは、例外的にaを返します。
    //!
    //! @param[in]    a  取得する乱数の最小値
    //! @param[in]    b  取得する乱数の天井値
    //!
    //! @return       乱数。
    //---------------------------------------------------------------------------
    f64 GetF64Range(f64 a, f64 b)
    {
//      EFT_ASSERT(b >= a);
        return (GetF64(b - a) + a);
    }

    //---------------------------------------------------------------------------
    //! @brief         -1 または 1 を等確率で返します。
    //!
    //! @return        -1 または 1
    //---------------------------------------------------------------------------
    int GetSign()
    {
        return ( GetU32() & 0x2 ) - 1;
    }

    //---------------------------------------------------------------------------
    //! @brief         true または false を等確率で返します。
    //!
    //! @return        true または false
    //---------------------------------------------------------------------------
    bool GetBool()
    {
        // 線形合同法と異なり、XorShiftは最下位ビットの短周期の規則性はない。
        // 0または1を直接得ることができるため、最下位ビットを取る実装としている。
        return static_cast<bool>(GetU32() & 0x1);
    }
    //@}

    //----------------------------------------------------------------
    /// @name   コンテキストの取得
    //----------------------------------------------------------------
    //@{

    //---------------------------------------------------------------------------
    //! @brief        乱数のコンテキストを4つの4バイト数値として取得します。
    //!
    //!               これで取得した値をコンストラクタやinitに与えることで、
    //!               全く同じコンテキストを作成することができます。
    //!
    //! @param[in]    num_0  コンテキストの数値を受け取るポインタ
    //! @param[in]    num_1  コンテキストの数値を受け取るポインタ
    //! @param[in]    num_2  コンテキストの数値を受け取るポインタ
    //! @param[in]    num_3  コンテキストの数値を受け取るポインタ
    //---------------------------------------------------------------------------
    void GetContext(u32* num_0, u32* num_1, u32* num_2, u32* num_3) const
    {
        *num_0 = mX;
        *num_1 = mY;
        *num_2 = mZ;
        *num_3 = mW;
    }

    //@}

private:
    u32 mX;     //!< 乱数のコンテキストX
    u32 mY;     //!< 乱数のコンテキストY
    u32 mZ;     //!< 乱数のコンテキストZ
    u32 mW;     //!< 乱数のコンテキストW
};


//---------------------------------------------------
//! @briefprivate エフェクトシステムでグローバルに参照されるランダムクラス
//---------------------------------------------------
class Random
{
public:

    //----------------------------------------------------------------
    /// @name   コンストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief        引数なしのコンストラクタです。
    //---------------------------------------------------------------------------
    Random();

    //@}

    //----------------------------------------------------------------
    /// @name   乱数テーブル
    //@{

    //---------------------------------------------------------------------------
    //! @brief        乱数テーブル作成。
    //---------------------------------------------------------------------------
    static void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        乱数テーブル破棄。
    //---------------------------------------------------------------------------
    static void Finalize();

    //@}

    //---------------------------------------------------------------------------
    //! @brief          シード設定。
    //! @param[in] val  乱数シード値
    //---------------------------------------------------------------------------
    void SetSeed( u32 val )
    {
        if ( !g_UseGlobalSeed )
        {
            m_Vec3RndIx           = static_cast<u16>( val >> 0  );
            m_NormalizedVec3RndIx = static_cast<u16>( val >> 16 );
            m_Rnd                 = val;
        }
        else
        {
            m_Vec3RndIx           = static_cast<u16>( g_GlobalSeed >> 0  );
            m_NormalizedVec3RndIx = static_cast<u16>( g_GlobalSeed >> 16 );
            m_Rnd                 = g_GlobalSeed;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief          テーブルから Vec3 乱数取得。
    //! @return         乱数を格納したVec3
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetVec3          (){ return g_Vec3Tbl          [ ( m_Vec3RndIx++ )           & cNumVec3TblMask ]; }

    //---------------------------------------------------------------------------
    //! @brief          テーブルから Normalized Vec3 乱数取得。
    //! @return         乱数を格納したVec3（正規化済）
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetNormalizedVec3(){ return g_NormalizedVec3Tbl[ ( m_NormalizedVec3RndIx++ ) & cNumVec3TblMask ]; }

    //---------------------------------------------------------------------------
    //! @brief          線形合同法のランダム。
    //! @return         乱数値
    //---------------------------------------------------------------------------
    u32 GetU32Direct(){ u32 ret = m_Rnd; m_Rnd = m_Rnd * 1103515245 + 12345; return ret; }

    //---------------------------------------------------------------------------
    //! @brief          0.0～1.0までのランダム。
    //! @return         乱数値
    //---------------------------------------------------------------------------
    f32 GetF32 (){ return GetU32Direct() * EFT_RANDOM_MAX_INVERSE_F32; }

    //---------------------------------------------------------------------------
    //! @brief          0～255までのランダム。
    //! @return         乱数値
    //---------------------------------------------------------------------------
    s32 GetS32 (){ return GetU32Direct() >> 24; }

    //---------------------------------------------------------------------------
    //! @brief          0～valまでのランダム。
    //! @param[in]      val 乱数の上限値
    //! @return         乱数値
    //---------------------------------------------------------------------------
    s32 GetS32( int val ){ return static_cast<s32>(((u64)GetU32Direct() * val) >> 32); }

    //---------------------------------------------------------------------------
    //! @brief          グローバルランダム取得関数。
    //! @return         乱数値
    //---------------------------------------------------------------------------
    static RandomGenerator* GetGlobalRandom();

    //---------------------------------------------------------------------------
    //! @brief              グローバルシードを利用するかどうか。
    //! @param[in] enable   グローバルシードの使用可否
    //! @param[in] seed     乱数シード
    //---------------------------------------------------------------------------
    static void SetGlobalSeed( bool enable, u32 seed )
    {
        g_UseGlobalSeed = enable;
        g_GlobalSeed    = seed;
    }

private:
    //---------------------------------------------------------------------------
    //! @brief ランダムテーブルに関する定数
    //---------------------------------------------------------------------------
    enum
    {
        cNumVec3Tbl         = 512,                      //!< ランダムテーブルのサイズ
        cNumVec3TblMask     = cNumVec3Tbl - 1           //!< ランダムテーブルのマスク値
    };

    static RandomGenerator  g_Random;                   //!< グローバルランダム
    static bool             g_UseGlobalSeed;            //!< グローバルなシードを利用する
    static u32              g_GlobalSeed;               //!< グローバルなシード
    static nw::math::VEC3*  g_Vec3Tbl;                  //!< Vec3 ランダムテーブル
    static nw::math::VEC3*  g_NormalizedVec3Tbl;        //!< Normlized Vec3 ランダムテーブル
    u16                     m_Vec3RndIx;                //!< Vec3 乱数テーブルの位置
    u16                     m_NormalizedVec3RndIx;      //!< Normlized Vec3 乱数テーブルの位置
    u32                     m_Rnd;                      //!< 線形合同法の為の変数
};

} // namespace eft2
} // namespace nw
