﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nw   {
namespace eft2 {

struct Emitter;

//------------------------------------------------------------------------------
//! @brief        パーティクル 位置/速度 アトリビュート
//!
//!               GPUに送られる1パーティクルの情報。
//------------------------------------------------------------------------------
struct ParticlePosAttribute
{
    nw::math::VEC4     localPos;                    //!< ローカル座標 ( xyz / life )
    nw::math::VEC4     localVec;                    //!< ローカル速度 ( xyz / birth time )
    nw::math::VEC4     localDiff;                   //!< ローカル差分

    //------------------------------------------------------------------------------
    //! @brief                  パーティクル時間を取得
    //! @param[in] emitterTime  エミッタ時間
    //! @return                 パーティクル時間
    //------------------------------------------------------------------------------
    f32 GetTime( const f32 emitterTime ) const { return ( emitterTime - localVec.w ); }

    //------------------------------------------------------------------------------
    //! @brief  パーティクルの寿命を取得
    //! @return パーティクル寿命
    //------------------------------------------------------------------------------
    f32 GetLife() const { return localPos.w; }

    //------------------------------------------------------------------------------
    //! @brief                  パーティクルが生存しているかどうかチェック
    //! @param[in] emitterTime  エミッタ時間
    //! @return                 生存している場合true
    //------------------------------------------------------------------------------
    bool IsAlive( const f32 emitterTime ) const { return ( GetTime( emitterTime ) < GetLife() ); }
};

//------------------------------------------------------------------------------
//! @brief        パーティクル スケール/カラー/回転 アトリビュート
//!
//!               GPUに送られる1パーティクルの情報。
//------------------------------------------------------------------------------
struct ParticleAttribute
{
    nw::math::VEC4      scale;                      //!< スケール（ x / y / birth time / 運動量ランダム ）
    nw::math::VEC4      random;                     //!< ランダム
    nw::math::VEC4      initRotate;                 //!< 初期回転値
    nw::math::VEC4      initColor0;                 //!< 初期カラー0
    nw::math::VEC4      initColor1;                 //!< 初期カラー1
    nw::math::VEC4      emitterMat0;                //!< エミッタマトリクス0
    nw::math::VEC4      emitterMat1;                //!< エミッタマトリクス1
    nw::math::VEC4      emitterMat2;                //!< エミッタマトリクス2
    nw::math::VEC4      emitterRTMat0;              //!< エミッタマトリクス0
    nw::math::VEC4      emitterRTMat1;              //!< エミッタマトリクス1
    nw::math::VEC4      emitterRTMat2;              //!< エミッタマトリクス2
};


//------------------------------------------------------------------------------
//! @brief        パーティクルデータ
//!
//!               アトリビュート以外のパーティクルの情報。
//------------------------------------------------------------------------------
struct ParticleData
{
#ifdef EFT_DEGRADATION_SPEC
    nw::math::VEC4  initScale;                      //!< パーティクル初期スケール
#endif
    u32     createID;                               //!< 生成ID
    u32     collisionCount;                         //!< [フィールド]コリジョン回数
    f32     createTime;                             //!< 生成時刻（CPUアクセス用）
    f32     life;                                   //!< 寿命（CPUアクセス用）
    void*   userData;                               //!< ユーザーデータ
    void*   emitterPluginData;                      //!< エミッタプラグインが利用するユーザーデータ
    bool    isBothInitialized;                      //!< 二本のバッファがすべて初期化されたかどうか

    //------------------------------------------------------------------------------
    //! @brief      パーティクル時間を取得
    //! @param[in]  emitterTime エミッタ時間
    //! @return     パーティクル時間
    //------------------------------------------------------------------------------
    f32 GetTime( const f32 emitterTime ) const
    {
        return ( emitterTime - createTime );
    }
    //------------------------------------------------------------------------------
    //! @brief      パーティクル寿命を取得
    //! @return     パーティクル寿命
    //------------------------------------------------------------------------------
    f32 GetLife() const
    {
        return life;
    }
};

//------------------------------------------------------------------------------
//! @brief        親パーティクルデータ
//!
//!               親パーティクル固有のパーティクルの情報。
//------------------------------------------------------------------------------
struct ParentParticleData
{
    Emitter*    childEmitter[ EFT_EMITTER_INSET_NUM ];    //!< チャイルドエミッタ
};

} // namespace eft2
} // namespace nw
