﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Heap.h>
#include <nw/eft/eft2_Enum.h>

namespace nw    {
namespace eft2  {

//---------------------------------------------------------------------------
//! @briefprivate   静的ヒープからメモリを設定します。
//! @param[in]      staticHeap ヒープへのポインタ
//---------------------------------------------------------------------------
void SetStaticHeap( Heap* staticHeap );

//---------------------------------------------------------------------------
//! @briefprivate       静的ヒープからメモリを確保します。
//! @param[in] size     確保するメモリサイズ
//! @param[in] align    アライメントサイズ
//! @return             メモリの先頭アドレス
//---------------------------------------------------------------------------
void* AllocFromStaticHeap( u32 size, u32 align = 128 );

//---------------------------------------------------------------------------
//! @briefprivate   静的ヒープからメモリを解放します。
//! @param[in] ptr  メモリの先頭アドレス
//---------------------------------------------------------------------------
void FreeFromStaticHeap( void* ptr );

//---------------------------------------------------------------------------
//! @brief  静的ヒープから確保されたメモリサイズを取得します。
//! @return メモリサイズ
//---------------------------------------------------------------------------
u32 GetAllocedSizeFromStaticHeap();

//---------------------------------------------------------------------------
//! @briefprivate           動的ヒープからメモリを設定します。
//! @param[in] dynamicHeap  ヒープへのポインタ
//---------------------------------------------------------------------------
void SetDynamicHeap( Heap* dynamicHeap );

//---------------------------------------------------------------------------
//! @briefprivate       動的ヒープからメモリを確保します。
//! @param[in] size     確保するメモリサイズ
//! @param[in] align    アライメントサイズ
//! @return             メモリの先頭アドレス
//---------------------------------------------------------------------------
void* AllocFromDynamicHeap( u32 size, u32 align = 128 );

//---------------------------------------------------------------------------
//! @briefprivate           動的ヒープからメモリを解放します。
//! @param[in] ptr          メモリの先頭アドレス
//! @param[in] immediate    即時解放フラグ
//---------------------------------------------------------------------------
void FreeFromDynamicHeap( void* ptr, bool immediate = true );

//---------------------------------------------------------------------------
//! @briefprivate           遅延解放用のリストを初期化します。
//!
//!                         動的ヒープから破棄されるメモリは遅延解放されます。
//! @param[in] freeListNum  遅延リストの数
//! @param[in] delayTime    遅延時間
//---------------------------------------------------------------------------
void InitializeDelayFreeList( u32 freeListNum, u32 delayTime );

//---------------------------------------------------------------------------
//! @briefprivate 遅延解放用のリストを破棄します。
//!
//!               動的ヒープから破棄されるメモリは遅延解放されます。
//---------------------------------------------------------------------------
void FinalizeDelayFreeList();

//---------------------------------------------------------------------------
//! @briefprivate 遅延解放用のリスト内アドレスを解放します。
//---------------------------------------------------------------------------
void FlushDelayFreeList();

//---------------------------------------------------------------------------
//! @briefprivate   遅延解放の為のフリーリストへアドレスを追加する
//! @param[in] ptr  遅延解放させるメモリのアドレス
//---------------------------------------------------------------------------
void AddFreeListForDynamicHeap( void* ptr );


//---------------------------------------------------------------------------
//! @brief          ランタイムログ出力を抑制行います。
//! @param[in] flag 制御フラグ
//---------------------------------------------------------------------------
void SetSuppressOutputLog( bool flag );

//---------------------------------------------------------------------------
//! @brief              ランタイムログ出力を行います。
//! @param[in] format   フォーマット文字列
//---------------------------------------------------------------------------
void OutputLog( const char* format, ... );

//---------------------------------------------------------------------------
//! @brief              ランタイム警告出力を行います。
//! @param[in] format   フォーマット文字列
//---------------------------------------------------------------------------
void OutputWarning( const char* format, ... );

//---------------------------------------------------
//! @brief                  ランタイム警告出力を行います。
//! @param[in] context      データ参照元へのポインタ
//! @param[in] warningId    警告ID
//---------------------------------------------------
void Warning( void* context, RuntimeWarning warningId );

//---------------------------------------------------------------------------
//! @brief  ランタイム警告のビットフィールドを取得します。
//! @return ランタイム警告のビットフィールド
//---------------------------------------------------------------------------
u32 GetWarnings();

//---------------------------------------------------------------------------
//! @brief              ランタイムエラー出力を行います。
//! @param[in] format   フォーマット文字列
//---------------------------------------------------------------------------
void OutputError( const char* format, ... );

//---------------------------------------------------
//! @brief              ログ・警告・エラー出力時コールバックの定義です。
//! @param[in] format   フォーマット文字列
//! @param[in] vargs    省略可能な引数
//---------------------------------------------------
typedef void (*OutputMessageCallback)( const char* format, va_list vargs );

//---------------------------------------------------
//! @brief  ランタイムログ出力時コールバックを設定します。
//! @param[in] cb コールバック
//---------------------------------------------------
void SetOutputLogCallBack( OutputMessageCallback cb );

//---------------------------------------------------
//! @brief  ランタイム警告出力時コールバックを設定します。
//! @param[in] cb コールバック
//---------------------------------------------------
void SetOutputWarningCallBack( OutputMessageCallback cb );

//---------------------------------------------------
//! @brief  ランタイムエラー出力時コールバックを設定します。
//! @param[in] cb コールバック
//---------------------------------------------------
void SetOutputErrorCallBack( OutputMessageCallback cb );

//---------------------------------------------------------------------------
//! @brief  ランタイム警告のビットフィールドを取得します。
//! @return ランタイム警告のビットフィールド
//---------------------------------------------------------------------------
u32 GetWarnings();

//---------------------------------------------------------------------------
//! @brief  文字列出力を行います。
//---------------------------------------------------------------------------
void VPrintf(const char *fmt, va_list vlist);

//---------------------------------
// fract GLSL 4.00準拠
//---------------------------------
inline float fract( const f32 x )
{
    return ( x - nw::math::FFloor(x) );
}

//---------------------------------
// step GLSL 4.00準拠
//---------------------------------
inline float step( const f32 edge, const f32 x )
{
    return ( x < edge ) ? 0.0f : 1.0f;
}

//---------------------------------------------------------------------------
//! @brief       x を base の倍数に切り上げます。
//!
//! @tparam      ValueT  型指定
//! @param[in]   x       切り上げる数。
//! @param[in]   base    切り上げの基準となる値。
//!                      2の累乗でなければなりません。
//!
//! @return      x を base の倍数に切り上げた値を返します。
//---------------------------------------------------------------------------
template <typename ValueT>
NW_INLINE ValueT
RoundUp(ValueT x, u32 base)
{
    u32 mask = base - 1;
    return static_cast<ValueT>(static_cast<ValueT>( x + mask ) & ~mask);
}

//------------------------------------------------------------------------------
//! @brief      エルミート補間
//! @param[out] retPos      補完された位置
//! @param[in]  startPos    始点
//! @param[in]  startVel    始点の方向
//! @param[in]  endPos      終点
//! @param[in]  endVel      終点の方向
//! @param[in]  time        相対位置
//------------------------------------------------------------------------------
inline void HermiteInterpolationOnCubic(
    nw::math::VEC3*       retPos,
    const nw::math::VEC3& startPos,
    const nw::math::VEC3& startVel,
    const nw::math::VEC3& endPos,
    const nw::math::VEC3& endVel,
    float                 time )
{
    // 転置
    const static nw::math::MTX44 hermite(
         2.f, -3.f, 0.f, 1.f,
        -2.f,  3.f, 0.f, 0.f,
         1.f, -2.f, 1.f, 0.f,
         1.f, -1.f, 0.f, 0.f );

    // 転置
    nw::math::MTX44 m(
        startPos.x, endPos.x,   startVel.x, endVel.x,
        startPos.y, endPos.y,   startVel.y, endVel.y,
        startPos.z, endPos.z,   startVel.z, endVel.z,
        0,          0,          0,          0 );

    MTX44Mult( &m, &m, &hermite );

    const f32 sqTime = time * time;
    nw::math::VEC3 timeVector( time * sqTime, sqTime, time );
    retPos->x = m.m[0][0] * timeVector.x + m.m[0][1] * timeVector.y + m.m[0][2] * timeVector.z + m.m[0][3];
    retPos->y = m.m[1][0] * timeVector.x + m.m[1][1] * timeVector.y + m.m[1][2] * timeVector.z + m.m[1][3];
    retPos->z = m.m[2][0] * timeVector.x + m.m[2][1] * timeVector.y + m.m[2][2] * timeVector.z + m.m[2][3];

    return;
}

} // namespace eft
} // namespace nw
