﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Data.h>
#include <nw/math.h>

namespace nw  {
namespace eft2 {

struct Emitter;
struct ParticleAttribute;

namespace detail {

static const s32 NUM_OF_WAVES = 4;                      //!< 波形の数
static const f32 _2PI = nw::math::F_PI * 2;             //!< 円周率x2
static const nw::math::VEC4 AMP_BASE( 4, 3, 2, 1.5f );  //!< 振幅

//! 基本周期に対する比率
static const nw::math::VEC4 CYCLE_RATE(
    ( 1.0f / 0.60f ),
    ( 1.0f / 0.42f ),
    ( 1.0f / 0.23f ),
    ( 1.0f / 0.15f )
);

}

//------------------------------------------------------------------------------
//! @brief                  GPUノイズの値を計算
//! @param[in] emitter      エミッタへのポインタ
//! @param[in,out] pos      パーティクル位置
//! @param[in,out] vec      パーティクル速度
//! @param[in] attr         パーティクルアトリビュートへのポインタ
//! @param[in] randomVel    ランダム量
//------------------------------------------------------------------------------
void CalcFieldRandom( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr, const nw::math::VEC3& randomVel );

#if 0
//------------------------------------------------------------------------------
// 疑似乱数生成器（積算可能）（標準機能）
// x時点での乱数の積算値を取得
//------------------------------------------------------------------------------
inline f32 RandFuncDefault( const f32 x, const f32 interval )
{
    // 波形合成
    const f32 t = x * interval;

    f32 aggregate = 0;
    aggregate += nw::math::SinRad( t * details::CYCLE_RATE[ 0 ] ) * details::AMP_BASE[ 0 ];
    aggregate += nw::math::SinRad( t * details::CYCLE_RATE[ 1 ] ) * details::AMP_BASE[ 1 ];
    aggregate += nw::math::SinRad( t * details::CYCLE_RATE[ 2 ] ) * details::AMP_BASE[ 2 ];
    aggregate += nw::math::SinRad( t * details::CYCLE_RATE[ 3 ] ) * details::AMP_BASE[ 3 ];

    return aggregate;
}
#endif

//------------------------------------------------------------------------------
//! @briefprivate       4つの波形を合成するユーティリティ
//! @param[in] t1       時刻t1
//! @param[in] t2       時刻t2
//! @param[in] t3       時刻t3
//! @param[in] t4       時刻t4
//! @param[in] cycle    周期に乗算する値
//! @param[in] amp      波の大きさ
//! @return             合成結果
//------------------------------------------------------------------------------
inline f32 GetAggregate4( const f32 t1, const f32 t2, const f32 t3, const f32 t4, const f32 cycle, const f32 amp )
{
    f32 add = -nw::math::SinRad( t1 * cycle );
    add += nw::math::SinRad( t2 * cycle );
    add += nw::math::SinRad( t3 * cycle );
    add -= nw::math::SinRad( t4 * cycle );
    return add * amp;
}

//------------------------------------------------------------------------------
//! @briefprivate       2つの波形を合成するユーティリティ
//! @param[in] t1       時刻t1
//! @param[in] t2       時刻t2
//! @param[in] cycle    周期に乗算する値
//! @param[in] amp      波の大きさ
//! @return             合成結果
//------------------------------------------------------------------------------
inline f32 GetAggregate2(  const f32 t1, const f32 t2, const f32 cycle, const f32 amp )
{
    f32 add = -nw::math::SinRad( t1 * cycle );
    add += nw::math::SinRad( t2 * cycle );
    return add * amp;
}

//------------------------------------------------------------------------------
//! @briefprivate   疑似乱数生成器（積算可能）（標準機能）
//!                 x時点での乱数の積算値差分を取得
//! @param[in] t1   時刻t1
//! @param[in] t2   時刻t2
//! @return         疑似乱数値
//------------------------------------------------------------------------------
inline f32 RandFuncSubDefault( const f32 t1, const f32 t2 )
{
    // 波形合成
    f32 aggregate = GetAggregate2( t1, t2, detail::CYCLE_RATE.x, detail::AMP_BASE.x );
    aggregate    += GetAggregate2( t1, t2, detail::CYCLE_RATE.y, detail::AMP_BASE.y );
    aggregate    += GetAggregate2( t1, t2, detail::CYCLE_RATE.z, detail::AMP_BASE.z );
    aggregate    += GetAggregate2( t1, t2, detail::CYCLE_RATE.w, detail::AMP_BASE.w );
    return aggregate;
}

//------------------------------------------------------------------------------
//! @briefprivate   疑似乱数生成器（積算可能）（標準機能）
//!                 x時点での乱数の積算値差分を取得
//!                 ( F2 - F1 ) - ( F4 - F3 )
//! @param[in] t1   時刻t1
//! @param[in] t2   時刻t2
//! @param[in] t3   時刻t3
//! @param[in] t4   時刻t4
//! @return         疑似乱数値
//------------------------------------------------------------------------------
inline f32 RandFuncSubSubDefault( const f32 t1, const f32 t2, const f32 t3, const f32 t4 )
{
    // 波形合成
    f32 aggregate = GetAggregate4( t1, t2, t3, t4, detail::CYCLE_RATE.x, detail::AMP_BASE.x );
    aggregate    += GetAggregate4( t1, t2, t3, t4, detail::CYCLE_RATE.y, detail::AMP_BASE.y );
    aggregate    += GetAggregate4( t1, t2, t3, t4, detail::CYCLE_RATE.z, detail::AMP_BASE.z );
    aggregate    += GetAggregate4( t1, t2, t3, t4, detail::CYCLE_RATE.w, detail::AMP_BASE.w );
    return aggregate;
}

//------------------------------------------------------------------------------
//! @briefprivate  疑似乱数生成器（積算可能）（詳細オプション使用）
//!         x時点での乱数の積算値を取得
//! @param[in] x                現在時刻x
//! @param[in] interval         フレーム数からRadianへ変換する定数
//! @param[in] resFieldRandom   GPUノイズのパラメータへのポインタ
//! @return 疑似乱数値
//------------------------------------------------------------------------------
inline f32 RandFunc( const f32 x, const f32 interval, const ResFieldRandom* resFieldRandom )
{
    // 振幅
    const nw::math::VEC4 ampBase(
        resFieldRandom->waveParam0,
        resFieldRandom->waveParam1,
        resFieldRandom->waveParam2,
        resFieldRandom->waveParam3 );

    // 基本周期に対する比率
    const f32 t = x * interval;
    nw::math::VEC4 cycleRate(
        ( t / resFieldRandom->waveParamHzRate0 ),
        ( t / resFieldRandom->waveParamHzRate1 ),
        ( t / resFieldRandom->waveParamHzRate2 ),
        ( t / resFieldRandom->waveParamHzRate3 ) );

    // 波形合成
    f32 aggregate = nw::math::SinRad( cycleRate.x ) * ampBase.x;
    aggregate    += nw::math::SinRad( cycleRate.y ) * ampBase.y;
    aggregate    += nw::math::SinRad( cycleRate.z ) * ampBase.z;
    aggregate    += nw::math::SinRad( cycleRate.w ) * ampBase.w;

    return aggregate;
}

#if 0
//------------------------------------------------------------------------------
// [x1, x2]の区間の乱数の積算を取得
//------------------------------------------------------------------------------
inline f32 GetRandSum(
    const f32 x1,
    const f32 x2,
    const ResFieldRandom* resFieldRandom )
{
    return RandFunc( x2, resFieldRandom ) - RandFunc( x1, resFieldRandom );
}
#endif

//------------------------------------------------------------------------------
//! @briefprivate               現在の疑似乱数を取得（詳細オプション）（位相変化あり）
//! @param[in] seed             乱数シード
//! @param[in] time             時刻
//! @param[in] deltaS           乱数シード差分
//! @param[in] deltaV           時刻差分
//! @param[in] interval         フレーム数からRadianへ変換する定数
//! @param[in] resFieldRandom   GPUノイズのパラメータへのポインタ
//! @return                     疑似乱数値
//------------------------------------------------------------------------------
inline f32 GetRand(
    const f32 seed,
    const f32 time,
    const f32 deltaS,
    const f32 deltaV,
    const f32 interval,
    const ResFieldRandom* resFieldRandom )
{
    const f32 x = seed + time;
    return RandFunc( x + deltaV + deltaS, interval, resFieldRandom ) - RandFunc( x, interval, resFieldRandom )
        - ( RandFunc( seed + deltaS, interval, resFieldRandom ) - RandFunc( seed, interval, resFieldRandom ) );
}

//------------------------------------------------------------------------------
//! @briefprivate               現在の疑似乱数を取得（詳細オプション）（位相変化なし）
//! @param[in] seed             乱数シード
//! @param[in] time             時刻
//! @param[in] deltaV           時刻差分
//! @param[in] interval         フレーム数からRadianへ変換する定数
//! @param[in] resFieldRandom   GPUノイズのパラメータへのポインタ
//! @return                     疑似乱数値
//------------------------------------------------------------------------------
inline f32 GetRandWithNoPhaseShift(
    const f32 seed,
    const f32 time,
    const f32 deltaV,
    const f32 interval,
    const ResFieldRandom* resFieldRandom )
{
    const f32 x = seed + time;
    return RandFunc( x + deltaV, interval, resFieldRandom ) - RandFunc( x, interval, resFieldRandom );
}

//------------------------------------------------------------------------------
//! @briefprivate       現在の疑似乱数を取得（標準）（位相変化あり）
//! @param[in] seed     乱数シード
//! @param[in] time     時刻
//! @param[in] deltaS   乱数シード差分
//! @param[in] deltaV   時刻差分
//! @param[in] interval フレーム数からRadianへ変換する定数
//! @return             疑似乱数値
//------------------------------------------------------------------------------
inline f32 GetRandDefault(
    const f32 seed,
    const f32 time,
    const f32 deltaS,
    const f32 deltaV,
    const f32 interval )
{
    const f32 x = seed + time;
    const f32 sTemp = deltaS * interval;
    const f32 t1 = x * interval;
    const f32 t2 = t1 + ( deltaV * interval ) + sTemp;
    const f32 t3 = seed * interval;
    const f32 t4 = t3 + sTemp;
    return RandFuncSubSubDefault( t1, t2, t3, t4 );
}

//------------------------------------------------------------------------------
//! @briefprivate 現在の疑似乱数を取得（標準）（位相変化なし）
//! @param[in] seed     乱数シード
//! @param[in] time     パーティクル時間
//! @param[in] deltaV   時間差分
//! @param[in] interval フレーム数からRadianへ変換する定数
//! @return             疑似乱数値
//------------------------------------------------------------------------------
inline f32 GetRandWithNoPhaseShiftDefault(
    const f32 seed,
    const f32 time,
    const f32 deltaV,
    const f32 interval )
{
    const f32 x = seed + time;
    const f32 t1 = x * interval;
    const f32 t2 = t1 + deltaV * interval;
    return RandFuncSubDefault( t1, t2 );
}

}
}
