﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/math.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_EmitterSet.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//! @brief          VEC3変換
//! @param[out] out 出力用のVEC3
//! @param[in]  mtx 変換に使うマトリクス
//! @param[in]  src 入力となるVEC3
//---------------------------------------------------------------------------
inline void _VEC3Transform( nw::math::VEC3* out, const nw::math::MTX34* mtx, const nw::math::VEC3* src )
{
    out->x = mtx->f._00 * src->x + mtx->f._01 * src->y + mtx->f._02 * src->z + mtx->f._03;
    out->y = mtx->f._10 * src->x + mtx->f._11 * src->y + mtx->f._12 * src->z + mtx->f._13;
    out->z = mtx->f._20 * src->x + mtx->f._21 * src->y + mtx->f._22 * src->z + mtx->f._23;
}

//---------------------------------------------------------------------------
//! @brief          VEC4変換
//! @param[out] out 出力用のVEC4
//! @param[in]  mtx 変換に使うマトリクス
//! @param[in]  src 入力となるVEC4
//---------------------------------------------------------------------------
inline void _VEC4Transform( nw::math::VEC4* out, const nw::math::MTX34* mtx, const nw::math::VEC4* src )
{
    // src->w は無視してオフセットを足し込む。
    out->x = mtx->f._00 * src->x + mtx->f._01 * src->y + mtx->f._02 * src->z + mtx->f._03;
    out->y = mtx->f._10 * src->x + mtx->f._11 * src->y + mtx->f._12 * src->z + mtx->f._13;
    out->z = mtx->f._20 * src->x + mtx->f._21 * src->y + mtx->f._22 * src->z + mtx->f._23;
}

}
}

//---------------------------------------------------------------------------
//! @briefprivate コールバック引数で古いバージョンのAPI提供する（いずれ削除予定）
//---------------------------------------------------------------------------
#define EFT_CALLBACK_ARG_OLD_API

namespace nw   {
namespace eft2 {

struct SuperStripeHistory;

//---------------------------------------------------------------------------
//! @brief  パーティクル用コールバックに渡される引数構造体です。
//---------------------------------------------------------------------------
struct ParticleCalcArgImpl
{
public:

    union
    {
        void*       userData;               //!< ユーザーデータ(パーティクル生成コールバックで返したアドレス)
        void*       pUserData;              //!< ユーザーデータ(パーティクル生成コールバックで返したアドレス)
    };
    union
    {
        void*       emPluginData;           //!< エミッタプラグインデータ(パーティクル生成コールバックで返したアドレス)
        void*       pEmitterPluginData;     //!< エミッタプラグインデータ(パーティクル生成コールバックで返したアドレス)
    };
    union
    {
        Emitter*    emitter;                //!< エミッタインスタンス
        Emitter*    pEmitter;               //!< エミッタインスタンス
    };

#ifdef EFT_CALLBACK_ARG_OLD_API

    //---------------------------------------------------------------------------
    // const参照で値を返すインターフェースの情報取得関数群。
    // 互換性のために残すが、いずれ削除予定。
    //---------------------------------------------------------------------------

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ローカル位置を取得
    //! @return ローカル位置
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetLocalPos();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ワールド位置を取得
    //! @return ワールド位置
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetWorldPos();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ローカル速度を取得
    //! @return ローカル速度
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetLocalVec();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ワールド速度を取得
    //! @return ワールド速度
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetWorldVec();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]移動量差分を取得
    //! @return 移動量差分
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetLocalDiff();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]回転量を計算（Radian）
    //! @return 回転量
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcCurrentRotateRadian();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]回転量を計算（Degree）
    //! @return 回転量
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcCurrentRotateDegree();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]初期スケール値を計算
    //! @return 初期スケール値
    //---------------------------------------------------------------------------
    const nw::math::VEC3& CalcInitScale();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ワールドスケール値を計算
    //! @return ワールドスケール値
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcWorldScale();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]ランダムを取得
    //! @return ランダム（x/y/z/w）
    //---------------------------------------------------------------------------
    const nw::math::VEC4& GetRandom();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]カラー0を計算
    //! @return カラー0
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcLocalColor0();

    //---------------------------------------------------------------------------
    //! @brief  [ 互換性維持 ]カラー1を計算
    //! @return カラー1
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcLocalColor1();

    //---------------------------------------------------------------------------
    //! @brief  現在の回転量を計算
    //! @return 現在の回転量
    //---------------------------------------------------------------------------
    const nw::math::VEC4& CalcLocalRotate()
    {
        return CalcCurrentRotateRadian();
    }
#endif

    //---------------------------------------------------------------------------
    //! @brief  パーティクル時間を取得
    //! @return パーティクル時間
    //---------------------------------------------------------------------------
    f32 GetTime() const { return ptclTime; }

    //---------------------------------------------------------------------------
    //! @brief  パーティクル寿命を取得
    //! @return パーティクル寿命
    //---------------------------------------------------------------------------
    f32 GetLife() const { return ptclLife; }

    //---------------------------------------------------------------------------
    // const参照を返さないインターフェースの情報取得関数群。
    // 最終的にこちらに一本化する。
    //---------------------------------------------------------------------------

    //---------------------------------------------------------------------------
    //! @brief              ローカル位置を取得
    //! @param[out] outPos  ローカル位置を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetLocalPos( nw::math::VEC3* outPos ) const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド位置を取得
    //! @param[out] outPos  ワールド位置を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetWorldPos( nw::math::VEC3* outPos ) const;

    //---------------------------------------------------------------------------
    //! @brief              ローカル速度を取得
    //! @param[out] outVec  ローカル速度を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetLocalVec( nw::math::VEC3* outVec ) const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド速度を取得
    //! @param[out] outVec  ワールド速度を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetWorldVec( nw::math::VEC3* outVec ) const;

    //---------------------------------------------------------------------------
    //! @brief              ローカル移動量差分を取得
    //! @param[out] outDiff ローカル移動量差分を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetLocalDiff( nw::math::VEC3* outDiff ) const;

    //---------------------------------------------------------------------------
    //! @brief                  回転初期値を取得（Radian）
    //! @param[out] outRotate   回転初期値を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetInitRotateRadian( nw::math::VEC3* outRotate ) const;
    void GetInitialRotateRadian( nw::math::VEC3* outRotate ) const
    {
        GetInitRotateRadian( outRotate );
    }

    //---------------------------------------------------------------------------
    //! @brief                  回転初期値を取得（Degree）
    //! @param[out] outRotate   回転初期値を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetInitRotateDegree( nw::math::VEC3* outRotate ) const;
    void GetInitialRotateDegree( nw::math::VEC3* outRotate ) const
    {
        GetInitRotateDegree( outRotate );
    }

    //---------------------------------------------------------------------------
    //! @brief                  Ptclごとのランダム値を取得
    //! @param[out] outRandom   Ptclごとのランダム値を格納するVEC4へのポインタ
    //---------------------------------------------------------------------------
    void GetRandom( nw::math::VEC4* outRandom ) const;

    //---------------------------------------------------------------------------
    //! @brief                  現在の回転量を計算（Radian）
    //! @param[out] outRotate   現在の回転量を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void CalcCurrentRotateRadian( nw::math::VEC3* outRotate ) const;
    void CalculateCurrentRotateRadian( nw::math::VEC3* outRotate ) const
    {
        CalcCurrentRotateRadian( outRotate );
    }

    //---------------------------------------------------------------------------
    //! @brief                  現在の回転量を計算（Degree）
    //! @param[out] outRotate   現在の回転量を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void CalcCurrentRotateDegree( nw::math::VEC3* outRotate ) const;
    void CalculateCurrentRotateDegree( nw::math::VEC3* outRotate ) const
    {
        CalcCurrentRotateDegree( outRotate );
    }

    //---------------------------------------------------------------------------
    //! @brief                  ワールドスケール値を計算
    //! @param[out] outScale    ワールドスケール値を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void GetInitScale( nw::math::VEC3* outScale ) const;
    void GetInitialScale( nw::math::VEC3* outScale ) const
    {
        GetInitScale( outScale );
    }

    //---------------------------------------------------------------------------
    //! @brief                  ワールドスケール値を計算
    //! @param[out] outScale    ワールドスケール値を格納するVEC3へのポインタ
    //---------------------------------------------------------------------------
    void CalcWorldScale( nw::math::VEC3* outScale ) const;
    void CalculateWorldScale( nw::math::VEC3* outScale ) const
    {
        CalcWorldScale( outScale );
    }

    //---------------------------------------------------------------------------
    //! @brief                  カラー0／アルファ0を計算
    //! @param[out] outColor    カラー0／アルファ0を格納するVEC4へのポインタ
    //---------------------------------------------------------------------------
    void CalcLocalColor0( nw::math::VEC4* outColor ) const;
    void CalculateLocalColor0( nw::math::VEC4* outColor ) const
    {
        CalcLocalColor0( outColor );
    }

    //---------------------------------------------------------------------------
    //! @brief                  カラー1／アルファ1を計算
    //! @param[out] outColor    カラー1／アルファ1を格納するVEC4へのポインタ
    //---------------------------------------------------------------------------
    void CalcLocalColor1( nw::math::VEC4* outColor ) const;
    void CalculateLocalColor1( nw::math::VEC4* outColor ) const
    {
        CalcLocalColor1( outColor );
    }

    //---------------------------------------------------------------------------
    //! @brief              ローカル位置を取得。VEC4版。
    //! @param[out] outPos  ローカル位置を格納するVEC4へのポインタ
    //---------------------------------------------------------------------------
    void GetLocalPos( nw::math::VEC4* outPos ) const;

    //---------------------------------------------------------------------------
    //! @brief  ローカル位置へのポインタを取得
    //! @return ローカル位置へのポインタ
    //---------------------------------------------------------------------------
    const nw::math::VEC4* RefLocalPos() const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド位置を取得
    //! @param[out] outPos  ワールド位置
    //---------------------------------------------------------------------------
    void GetWorldPos( nw::math::VEC4* outPos ) const;

    //------------------------------------------------------------------------------
    //! @brief                  パーティクル位置をワールド座標系に変換するための、追従設定を加味した変換行列を取得します。
    //! @param[out] pOutMatrix  変換用の行列へのポインタ
    //------------------------------------------------------------------------------
    void GetEmitterTransformMatrix( nw::math::MTX34* pOutMatrix ) const;

    //---------------------------------------------------------------------------
    //! @brief          ローカル位置を設定
    //! @param[in] pos  ローカル位置
    //---------------------------------------------------------------------------
    void SetLocalPos( const nw::math::VEC3& pos );

    //---------------------------------------------------------------------------
    //! @brief          ローカル位置を設定
    //! @param[in] x    x座標
    //! @param[in] y    y座標
    //! @param[in] z    z座標
    //---------------------------------------------------------------------------
    void SetLocalPos( const f32 x, const f32 y, const f32 z );

    //---------------------------------------------------------------------------
    //! @brief          ワールド位置を設定
    //! @param[in] pos  ワールド位置
    //---------------------------------------------------------------------------
    void SetWorldPos( const nw::math::VEC3& pos );

    //---------------------------------------------------------------------------
    //! @brief          ワールド位置を設定
    //! @param[in] x    x座標
    //! @param[in] y    y座標
    //! @param[in] z    z座標
    //---------------------------------------------------------------------------
    void SetWorldPos( const f32 x, const f32 y, const f32 z );

    //---------------------------------------------------------------------------
    //! @brief          ローカル速度を設定
    //! @param[in] vec  ローカル速度
    //---------------------------------------------------------------------------
    void SetLocalVec( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief          ローカル速度を設定
    //! @param[in] x    速度のx成分
    //! @param[in] y    速度のy成分
    //! @param[in] z    速度のz成分
    //---------------------------------------------------------------------------
    void SetLocalVec( const f32 x, const f32 y, const f32 z );

    //---------------------------------------------------------------------------
    //! @brief          ワールド速度を設定
    //! @param[in] vec  ワールド速度
    //---------------------------------------------------------------------------
    void SetWorldVec( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief          ワールド速度を設定
    //! @param[in] x    速度のx成分
    //! @param[in] y    速度のy成分
    //! @param[in] z    速度のz成分
    //---------------------------------------------------------------------------
    void SetWorldVec( const f32 x, const f32 y, const f32 z );

    //---------------------------------------------------------------------------
    //! @brief              初期スケールを設定
    //! @param[in] scale    初期スケール
    //---------------------------------------------------------------------------
    void SetInitScale( const nw::math::VEC3& scale );
    void SetInitialScale( const nw::math::VEC3& scale )
    {
        SetInitScale( scale );
    }

    //---------------------------------------------------------------------------
    //! @brief              初期スケールXを設定
    //! @param[in] scaleX   初期スケールX
    //---------------------------------------------------------------------------
    void SetInitScaleX( const f32 scaleX );
    void SetInitialScaleX( const f32 scaleX )
    {
        SetInitScaleX( scaleX );
    }

    //---------------------------------------------------------------------------
    //! @brief              初期スケールYを設定
    //! @param[in] scaleY   初期スケールY
    //---------------------------------------------------------------------------
    void SetInitScaleY( const f32 scaleY );
    void SetInitialScaleY( const f32 scaleY )
    {
        SetInitScaleY( scaleY );
    }

    //---------------------------------------------------------------------------
    //! @brief              回転量を設定（Radian）
    //! @param[in] rotate   回転量
    //---------------------------------------------------------------------------
    void SetInitRotateRadian( const nw::math::VEC3& rotate );
    void SetInitialRotateRadian( const nw::math::VEC3& rotate )
    {
        SetInitRotateRadian( rotate );
    }

    //---------------------------------------------------------------------------
    //! @brief              回転量を設定（Degree）
    //! @param[in] rotate   回転量
    //---------------------------------------------------------------------------
    void SetInitRotateDegree( const nw::math::VEC3& rotate );
    void SetInitialRotateDegree( const nw::math::VEC3& rotate )
    {
        SetInitRotateDegree( rotate );
    }

    //---------------------------------------------------------------------------
    //! @brief      パーティクルを削除します
    //! @param[in]  keepChildEmitter    true の場合、チャイルドエミッタがいたときに、それらを消さずに残します。
    //---------------------------------------------------------------------------
    void Kill( bool keepChildEmitter = false );

    //---------------------------------------------------------------------------
    //! @brief              ローカル速度に変換
    //! @param[out] outDst  出力を格納するVEC3へのポインタ
    //! @param[in]  src     ワールド速度
    //---------------------------------------------------------------------------
    void TransformToLocalVec( nw::math::VEC3* outDst, const nw::math::VEC3& src ) const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド速度に変換
    //! @param[out] outDst  出力を格納するVEC3へのポインタ
    //! @param[in]  src     ローカル速度
    //---------------------------------------------------------------------------
    void TransformToWorldVec( nw::math::VEC3* outDst, const nw::math::VEC3& src ) const;

    //---------------------------------------------------------------------------
    //! @brief              ローカル位置に変換
    //! @param[out] outDst  出力を格納するVEC3へのポインタ
    //! @param[in] src      ワールド位置
    //---------------------------------------------------------------------------
    void TransformToLocalPos( nw::math::VEC3* outDst, const nw::math::VEC3& src ) const;

    //---------------------------------------------------------------------------
    //! @brief              ローカル位置に変換
    //! @param[out] outDst  出力を格納するVEC4へのポインタ
    //! @param[in] src      ワールド位置
    //---------------------------------------------------------------------------
    void TransformToLocalPos( nw::math::VEC4* outDst, const nw::math::VEC4& src ) const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド位置に変換
    //! @param[out] outDst  出力を格納するVEC3へのポインタ
    //! @param[in] src      ローカル位置
    //---------------------------------------------------------------------------
    void TransformToWorldPos( nw::math::VEC3* outDst, const nw::math::VEC3& src ) const;

    //---------------------------------------------------------------------------
    //! @brief              ワールド位置に変換
    //! @param[out] outDst  出力を格納するVEC4へのポインタ
    //! @param[in] src      ローカル位置
    //---------------------------------------------------------------------------
    void TransformToWorldPos( nw::math::VEC4* outDst, const nw::math::VEC4& src ) const;

    //---------------------------------------------------------------------------
    //! @brief                  コンストラクタ
    //! @param[in] emitter      エミッタへのポインタ
    //! @param[in] time         パーティクル時間
    //! @param[in] life         パーティクル寿命
    //! @param[in] ptclIndex    自身が格納されているパーティクル配列のインデックス
    //---------------------------------------------------------------------------
    ParticleCalcArgImpl( Emitter* emitter, const f32 time, const f32 life, const u32 ptclIndex );

    //---------------------------------------------------------------------------
    //! @internal
    //! @briefprivate   履歴式ストライプ2 での履歴点の配列への参照を取得します。
    //!                 この API は予告なく仕様が変更される可能性があります。
    //! @param[out]     pOutHistoryHead 履歴点（ nw::eft2::SuperStripeHistory ）の配列の先頭へのポインタ
    //! @param[out]     pOutHistoryNum  履歴点の総数
    //---------------------------------------------------------------------------
    void GetSuperStripeHistoryBuffer( SuperStripeHistory** pOutHistoryHead, s32* pOutHistoryNum );

private:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    ParticleCalcArgImpl();

    f32                     ptclTime;       //!< パーティクル時間
    f32                     ptclLife;       //!< パーティクル寿命
    u32                     ptclIndex;      //!< パーティクルのインデックス

#ifdef EFT_CALLBACK_ARG_OLD_API
    //---------------------------------------------------------------------------
    //! @briefprivate   const参照を返すためのデータ部分。
    //---------------------------------------------------------------------------
    struct _ParticleData
    {
        nw::math::VEC3      localDiff;      //!< ローカル系の移動差分
        nw::math::VEC3      localPos;       //!< ローカル位置
        nw::math::VEC3      worldPos;       //!< ワールド位置
        nw::math::VEC3      localVec;       //!< ローカル速度
        nw::math::VEC3      worldVec;       //!< ワールド速度
        nw::math::VEC4      localRotate;    //!< 初期回転量
        nw::math::VEC4      localScale;     //!< 初期スケール値
        nw::math::VEC4      worldScale;     //!< ワールドスケール値
        nw::math::VEC4      localColor0;    //!< カラー0（R/G/B/A）
        nw::math::VEC4      localColor1;    //!< カラー1（R/G/B/A）
        nw::math::VEC4      random;         //!< ランダム値（X/Y/Z/Wの4つ）
    };
    _ParticleData           ptcl;           //!< パーティクルデータ（const参照用）
#endif

};

//---------------------------------------------------------------------------
// 実装は現在同じ。型名だけ念のため分けておく。
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//! @brief  パーティクル計算コールバックに渡される構造体の定義
//---------------------------------------------------------------------------
typedef ParticleCalcArgImpl ParticleCalcArg;
//---------------------------------------------------------------------------
//! @brief  パーティクル放出コールバックに渡される構造体の定義
//---------------------------------------------------------------------------
typedef ParticleCalcArgImpl ParticleEmitArg;
//---------------------------------------------------------------------------
//! @brief  パーティクル削除コールバックに渡される構造体の定義
//---------------------------------------------------------------------------
typedef ParticleCalcArgImpl ParticleRemoveArg;

//---------------------------------------------------------------------------
//! @brief  カスタムフィールドコールバック
//---------------------------------------------------------------------------
typedef bool( *CustomFieldCallback )( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr, const ResFieldCustom* data, s32 particleIndex );

//---------------------------------------------------------------------------
//! @name パーティクル計算処理 コールバック
//@{

//---------------------------------------------------------------------------
//! @brief  パーティクル計算処理コールバックの定義です。
//!         パーティクル挙動の計算処理終了時にコールされます。
//---------------------------------------------------------------------------
typedef void( *ParticleCalcCallback )( ParticleCalcArg& arg );

//@}

//---------------------------------------------------------------------------
//! @name パーティクル生成/削除 コールバック
//@{

//---------------------------------------------------------------------------
//! @brief  パーティクル生成後にコールされるコールバックです。
//!         パーティクル生成をキャンセルしたい場合は、falseを返すようにしてください。
//---------------------------------------------------------------------------
typedef bool( *ParticleEmitCallback )( ParticleEmitArg& arg );

//---------------------------------------------------------------------------
//! @brief  パーティクル削除時にコールされるコールバックです。
//---------------------------------------------------------------------------
typedef bool( *ParticleRemoveCallback )( ParticleRemoveArg& arg );

//@}


//---------------------------------------------------------------------------
//! @brief  移動量差分を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetLocalDiff( nw::math::VEC3* outDiff ) const
{
    EFT_F32_VEC3_COPY( outDiff, &emitter->particlePosAttrBack[ ptclIndex ].localDiff );
}


//---------------------------------------------------------------------------
//! @brief  ローカル位置を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetLocalPos( nw::math::VEC3* outPos ) const
{
    EFT_F32_VEC3_COPY( outPos, &emitter->particlePosAttr[ ptclIndex ].localPos );
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetLocalPos( nw::math::VEC4* outPos ) const
{
    EFT_F32_VEC3_COPY( outPos, &emitter->particlePosAttr[ ptclIndex ].localPos );
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC4* ParticleCalcArg::RefLocalPos() const
{
    return &emitter->particlePosAttr[ ptclIndex ].localPos;
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetWorldPos( nw::math::VEC3* outPos ) const
{
    nw::math::VEC3 localPos;
    EFT_F32_VEC3_COPY( &localPos, &emitter->particlePosAttr[ ptclIndex ].localPos );

    // ワールド座標に変換して返す。
    TransformToWorldPos( outPos, localPos );
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetWorldPos( nw::math::VEC4* outPos ) const
{
    // ワールド座標に変換して返す。
    TransformToWorldPos( outPos, emitter->particlePosAttr[ ptclIndex ].localPos );
}

//---------------------------------------------------------------------------
//! @brief  ローカル速度を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetLocalVec( nw::math::VEC3* outVec ) const
{
    EFT_F32_VEC3_COPY( outVec, &emitter->particlePosAttr[ ptclIndex ].localVec );
}

//---------------------------------------------------------------------------
//! @brief  ワールド速度を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetWorldVec( nw::math::VEC3* outVec ) const
{
    nw::math::VEC3 localVec;
    EFT_F32_VEC3_COPY( &localVec, &emitter->particlePosAttr[ ptclIndex ].localVec );

    // ワールド速度に変換して返す。
    TransformToWorldVec( outVec, localVec );
}

//---------------------------------------------------------------------------
//! @brief  回転量を計算
//---------------------------------------------------------------------------
inline void ParticleCalcArg::CalcCurrentRotateRadian( nw::math::VEC3* outRotate ) const
{
    nw::math::VEC4 rotate;
    CalcRotationMatrix( &rotate,
                        emitter->emitterRes,
                        &emitter->particleAttrBack[ ptclIndex ].initRotate,
                        &emitter->particleAttrBack[ ptclIndex ].random,
                        ptclTime );
    EFT_F32_VEC3_COPY( outRotate, &rotate );
}

//---------------------------------------------------------------------------
//! @brief  回転量を計算
//---------------------------------------------------------------------------
inline void ParticleCalcArg::CalcCurrentRotateDegree( nw::math::VEC3* outRotate ) const
{
    CalcCurrentRotateRadian( outRotate );
    outRotate->x = nw::math::RadToDeg( outRotate->x );
    outRotate->y = nw::math::RadToDeg( outRotate->y );
    outRotate->z = nw::math::RadToDeg( outRotate->z );
}

//---------------------------------------------------------------------------
//! @brief  初期スケールを取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetInitScale( nw::math::VEC3* outScale ) const
{
    // TODO: パーティクルスケールZが入ってきたら対応。
    //       現状scale.zには別の目的の値が格納されているのでスルー。
    EFT_F32_VEC3_COPY( outScale, &emitter->particleAttr[ ptclIndex ].scale );
}

//---------------------------------------------------------------------------
//! @brief  ワールドスケール値を計算
//---------------------------------------------------------------------------
inline void ParticleCalcArg::CalcWorldScale( nw::math::VEC3* outScale ) const
{
    const nw::math::VEC4* random = &emitter->particleAttrBack[ ptclIndex ].random;
    const nw::math::VEC4* scale  = &emitter->particleAttrBack[ ptclIndex ].scale;
    const nw::math::VEC3& s = emitter->emitterSet->GetParticleScaleForCalc();

    nw::math::VEC4 worldScale;
    CalcPtclScaleVecFromTime( &worldScale,  emitter->emitterRes, scale, random, ptclLife, ptclTime );

    worldScale.x *= s.x;
    worldScale.y *= s.y;
    worldScale.z *= s.z;

    EFT_F32_VEC3_COPY( outScale, &worldScale );
}

//---------------------------------------------------------------------------
//! @brief  カラー0を計算
//---------------------------------------------------------------------------
inline void ParticleCalcArg::CalcLocalColor0( nw::math::VEC4* outColor ) const
{
    const nw::math::VEC4* random    = &emitter->particleAttrBack[ ptclIndex ].random;

    CalcPtclColor0Vec( outColor,
                       emitter->emitterRes,
                       random,
                       &emitter->color0,
                       static_cast<const nw::math::VEC3 *>( &emitter->m_EmitterAnim.vcolor0 ),
                       emitter->m_EmitterAnim.valpha0.x,
                       ptclLife, ptclTime );
}

//---------------------------------------------------------------------------
//! @brief  カラー1を計算
//---------------------------------------------------------------------------
inline void ParticleCalcArg::CalcLocalColor1( nw::math::VEC4* outColor ) const
{
    const nw::math::VEC4* random    = &emitter->particleAttrBack[ ptclIndex ].random;

    CalcPtclColor1Vec(  outColor,
                        emitter->emitterRes,
                        random,
                        &emitter->color0,
                        static_cast<const nw::math::VEC3 *>( &emitter->m_EmitterAnim.vcolor1 ),
                        emitter->m_EmitterAnim.valpha1.x,
                        ptclLife, ptclTime );
}

//---------------------------------------------------------------------------
//! @brief  ランダムを取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetRandom( nw::math::VEC4* outRandom ) const
{
    const nw::math::VEC4 rnd = emitter->particleAttrBack[ ptclIndex ].random;
    EFT_F32_VEC4_COPY( outRandom, &rnd );
}

//---------------------------------------------------------------------------
//! @brief パーティクルを削除する
//---------------------------------------------------------------------------
inline void ParticleCalcArg::Kill( bool keepChildEmitter )
{
    // 寿命を縮める
    emitter->particleData[ ptclIndex ].life = 0;
    emitter->particlePosAttr[ ptclIndex ].localPos.w = 0.0f;

    // チャイルドを残す場合は、以降何もしない
    if( keepChildEmitter )
    {
        return;
    }

    // 削除されたパーティクルが親パーティクルであれば、
    // 付随するチャイルドエミッタも削除する
    ParentParticleData* pParentParticleData = &pEmitter->parentParticleData[ ptclIndex ];
    if ( !pParentParticleData )
    {
        return;
    }

    for ( u32 j = 0; j < pEmitter->emitterRes->childEmitterResCount; j++ )
    {
        Emitter* pChildEmitter = pParentParticleData->childEmitter[ j ];
        if ( !pChildEmitter )
        {
            continue;
        }

        pChildEmitter->Kill();
    }
}

//---------------------------------------------------------------------------
//! @brief  ローカル速度に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToLocalVec( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const
{
    const u32 i = ptclIndex;
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        nw::math::MTX34 mtxSRT = emitter->matrixSRT;
        mtxSRT.m[0][3] = 0;
        mtxSRT.m[1][3] = 0;
        mtxSRT.m[2][3] = 0;
        nw::math::MTX34 mtxInvSRT;
        nw::math::MTX34Inverse( &mtxInvSRT, mtxSRT );

        nw::math::VEC3Transform( pDst, mtxInvSRT, src );
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[i].emitterMat0.x, emitter->particleAttrBack[i].emitterMat0.y, emitter->particleAttrBack[i].emitterMat0.z, 0,
            emitter->particleAttrBack[i].emitterMat1.x, emitter->particleAttrBack[i].emitterMat1.y, emitter->particleAttrBack[i].emitterMat1.z, 0,
            emitter->particleAttrBack[i].emitterMat2.x, emitter->particleAttrBack[i].emitterMat2.y, emitter->particleAttrBack[i].emitterMat2.z, 0
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        nw::math::VEC3Transform( pDst, mtxInv, src );
        return;
    }
    else
    {
        // 追従なし
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[i].emitterMat0.x, emitter->particleAttrBack[i].emitterMat0.y, emitter->particleAttrBack[i].emitterMat0.z, 0,
            emitter->particleAttrBack[i].emitterMat1.x, emitter->particleAttrBack[i].emitterMat1.y, emitter->particleAttrBack[i].emitterMat1.z, 0,
            emitter->particleAttrBack[i].emitterMat2.x, emitter->particleAttrBack[i].emitterMat2.y, emitter->particleAttrBack[i].emitterMat2.z, 0
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        nw::math::VEC3Transform( pDst, mtxInv, src );
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ワールド速度に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToWorldVec( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const
{
    const nw::math::VEC3& localVec = src;
    const u32 i = ptclIndex;
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        pDst->x = emitter->matrixSRT.f._00 * localVec.x + emitter->matrixSRT.f._01 * localVec.y + emitter->matrixSRT.f._02 * localVec.z;
        pDst->y = emitter->matrixSRT.f._10 * localVec.x + emitter->matrixSRT.f._11 * localVec.y + emitter->matrixSRT.f._12 * localVec.z;
        pDst->z = emitter->matrixSRT.f._20 * localVec.x + emitter->matrixSRT.f._21 * localVec.y + emitter->matrixSRT.f._22 * localVec.z;
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        pDst->x = emitter->particleAttrBack[i].emitterMat0.x * localVec.x + emitter->particleAttrBack[i].emitterMat0.y * localVec.y + emitter->particleAttrBack[i].emitterMat0.z * localVec.z;
        pDst->y = emitter->particleAttrBack[i].emitterMat1.x * localVec.x + emitter->particleAttrBack[i].emitterMat1.y * localVec.y + emitter->particleAttrBack[i].emitterMat1.z * localVec.z;
        pDst->z = emitter->particleAttrBack[i].emitterMat2.x * localVec.x + emitter->particleAttrBack[i].emitterMat2.y * localVec.y + emitter->particleAttrBack[i].emitterMat2.z * localVec.z;
        return;
    }
    else
    {
        // 追従なし
        pDst->x = emitter->particleAttrBack[i].emitterMat0.x * localVec.x + emitter->particleAttrBack[i].emitterMat0.y * localVec.y + emitter->particleAttrBack[i].emitterMat0.z * localVec.z;
        pDst->y = emitter->particleAttrBack[i].emitterMat1.x * localVec.x + emitter->particleAttrBack[i].emitterMat1.y * localVec.y + emitter->particleAttrBack[i].emitterMat1.z * localVec.z;
        pDst->z = emitter->particleAttrBack[i].emitterMat2.x * localVec.x + emitter->particleAttrBack[i].emitterMat2.y * localVec.y + emitter->particleAttrBack[i].emitterMat2.z * localVec.z;
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToLocalPos( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const
{
    // ローカル位置に戻して保存
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        nw::math::MTX34 mtxSRT = emitter->matrixSRT;
        nw::math::MTX34 mtxInvSRT;
        nw::math::MTX34Inverse( &mtxInvSRT, mtxSRT );

        _VEC3Transform( pDst, &mtxInvSRT, &src );
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[ptclIndex].emitterMat0.x, emitter->particleAttrBack[ptclIndex].emitterMat0.y, emitter->particleAttrBack[ptclIndex].emitterMat0.z, emitter->matrixSRT.f._03,
            emitter->particleAttrBack[ptclIndex].emitterMat1.x, emitter->particleAttrBack[ptclIndex].emitterMat1.y, emitter->particleAttrBack[ptclIndex].emitterMat1.z, emitter->matrixSRT.f._13,
            emitter->particleAttrBack[ptclIndex].emitterMat2.x, emitter->particleAttrBack[ptclIndex].emitterMat2.y, emitter->particleAttrBack[ptclIndex].emitterMat2.z, emitter->matrixSRT.f._23
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        _VEC3Transform( pDst, &mtxInv, &src );
        return;
    }
    else
    {
        // 追従なし
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[ptclIndex].emitterMat0.x, emitter->particleAttrBack[ptclIndex].emitterMat0.y, emitter->particleAttrBack[ptclIndex].emitterMat0.z, emitter->particleAttrBack[ptclIndex].emitterMat0.w,
            emitter->particleAttrBack[ptclIndex].emitterMat1.x, emitter->particleAttrBack[ptclIndex].emitterMat1.y, emitter->particleAttrBack[ptclIndex].emitterMat1.z, emitter->particleAttrBack[ptclIndex].emitterMat1.w,
            emitter->particleAttrBack[ptclIndex].emitterMat2.x, emitter->particleAttrBack[ptclIndex].emitterMat2.y, emitter->particleAttrBack[ptclIndex].emitterMat2.z, emitter->particleAttrBack[ptclIndex].emitterMat2.w
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        _VEC3Transform( pDst, &mtxInv, &src );
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToLocalPos( nw::math::VEC4* pDst, const nw::math::VEC4& src ) const
{
    // ローカル位置に戻して保存
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        nw::math::MTX34 mtxSRT = emitter->matrixSRT;
        nw::math::MTX34 mtxInvSRT;
        nw::math::MTX34Inverse( &mtxInvSRT, mtxSRT );

        _VEC4Transform( pDst, &mtxInvSRT, &src );
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[ptclIndex].emitterMat0.x, emitter->particleAttrBack[ptclIndex].emitterMat0.y, emitter->particleAttrBack[ptclIndex].emitterMat0.z, emitter->matrixSRT.f._03,
            emitter->particleAttrBack[ptclIndex].emitterMat1.x, emitter->particleAttrBack[ptclIndex].emitterMat1.y, emitter->particleAttrBack[ptclIndex].emitterMat1.z, emitter->matrixSRT.f._13,
            emitter->particleAttrBack[ptclIndex].emitterMat2.x, emitter->particleAttrBack[ptclIndex].emitterMat2.y, emitter->particleAttrBack[ptclIndex].emitterMat2.z, emitter->matrixSRT.f._23
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        _VEC4Transform( pDst, &mtxInv, &src );
        return;
    }
    else
    {
        // 追従なし
        nw::math::MTX34 mtx(
            emitter->particleAttrBack[ptclIndex].emitterMat0.x, emitter->particleAttrBack[ptclIndex].emitterMat0.y, emitter->particleAttrBack[ptclIndex].emitterMat0.z, emitter->particleAttrBack[ptclIndex].emitterMat0.w,
            emitter->particleAttrBack[ptclIndex].emitterMat1.x, emitter->particleAttrBack[ptclIndex].emitterMat1.y, emitter->particleAttrBack[ptclIndex].emitterMat1.z, emitter->particleAttrBack[ptclIndex].emitterMat1.w,
            emitter->particleAttrBack[ptclIndex].emitterMat2.x, emitter->particleAttrBack[ptclIndex].emitterMat2.y, emitter->particleAttrBack[ptclIndex].emitterMat2.z, emitter->particleAttrBack[ptclIndex].emitterMat2.w
            );

        nw::math::MTX34 mtxInv;
        nw::math::MTX34Inverse( &mtxInv, mtx );

        _VEC4Transform( pDst, &mtxInv, &src );
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToWorldPos( nw::math::VEC3* pDst, const nw::math::VEC3& src ) const
{
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        pDst->x = emitter->matrixSRT.f._00 * src.x + emitter->matrixSRT.f._01 * src.y + emitter->matrixSRT.f._02 * src.z + emitter->matrixSRT.f._03;
        pDst->y = emitter->matrixSRT.f._10 * src.x + emitter->matrixSRT.f._11 * src.y + emitter->matrixSRT.f._12 * src.z + emitter->matrixSRT.f._13;
        pDst->z = emitter->matrixSRT.f._20 * src.x + emitter->matrixSRT.f._21 * src.y + emitter->matrixSRT.f._22 * src.z + emitter->matrixSRT.f._23;
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        pDst->x = emitter->particleAttrBack[ptclIndex].emitterMat0.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat0.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat0.z * src.z + emitter->matrixSRT.f._03;
        pDst->y = emitter->particleAttrBack[ptclIndex].emitterMat1.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat1.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat1.z * src.z + emitter->matrixSRT.f._13;
        pDst->z = emitter->particleAttrBack[ptclIndex].emitterMat2.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat2.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat2.z * src.z + emitter->matrixSRT.f._23;
        return;
    }
    else
    {
        // 追従なし
        pDst->x = emitter->particleAttrBack[ptclIndex].emitterMat0.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat0.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat0.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat0.w;
        pDst->y = emitter->particleAttrBack[ptclIndex].emitterMat1.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat1.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat1.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat1.w;
        pDst->z = emitter->particleAttrBack[ptclIndex].emitterMat2.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat2.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat2.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat2.w;
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置に変換
//---------------------------------------------------------------------------
inline void ParticleCalcArg::TransformToWorldPos( nw::math::VEC4* pDst, const nw::math::VEC4& src ) const
{
    if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
    {
        // 完全追従
        pDst->x = emitter->matrixSRT.f._00 * src.x + emitter->matrixSRT.f._01 * src.y + emitter->matrixSRT.f._02 * src.z + emitter->matrixSRT.f._03;
        pDst->y = emitter->matrixSRT.f._10 * src.x + emitter->matrixSRT.f._11 * src.y + emitter->matrixSRT.f._12 * src.z + emitter->matrixSRT.f._13;
        pDst->z = emitter->matrixSRT.f._20 * src.x + emitter->matrixSRT.f._21 * src.y + emitter->matrixSRT.f._22 * src.z + emitter->matrixSRT.f._23;
        return;
    }
    else if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_POS )
    {
        // 位置のみ追従
        pDst->x = emitter->particleAttrBack[ptclIndex].emitterMat0.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat0.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat0.z * src.z + emitter->matrixSRT.f._03;
        pDst->y = emitter->particleAttrBack[ptclIndex].emitterMat1.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat1.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat1.z * src.z + emitter->matrixSRT.f._13;
        pDst->z = emitter->particleAttrBack[ptclIndex].emitterMat2.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat2.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat2.z * src.z + emitter->matrixSRT.f._23;
        return;
    }
    else
    {
        // 追従なし
        pDst->x = emitter->particleAttrBack[ptclIndex].emitterMat0.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat0.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat0.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat0.w;
        pDst->y = emitter->particleAttrBack[ptclIndex].emitterMat1.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat1.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat1.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat1.w;
        pDst->z = emitter->particleAttrBack[ptclIndex].emitterMat2.x * src.x + emitter->particleAttrBack[ptclIndex].emitterMat2.y * src.y + emitter->particleAttrBack[ptclIndex].emitterMat2.z * src.z + emitter->particleAttrBack[ptclIndex].emitterMat2.w;
        return;
    }
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetLocalPos( const f32 x, const f32 y, const f32 z )
{
    emitter->particlePosAttr[ ptclIndex ].localPos.x = x;
    emitter->particlePosAttr[ ptclIndex ].localPos.y = y;
    emitter->particlePosAttr[ ptclIndex ].localPos.z = z;
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetLocalPos( const nw::math::VEC3& pos )
{
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localPos, &pos );
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetWorldPos( const f32 worldX, const f32 worldY, const f32 worldZ )
{
    // ローカル位置に戻して保存
    nw::math::VEC3 lPos;
    TransformToLocalPos( &lPos, nw::math::VEC3( worldX, worldY, worldZ ) );
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localPos, &lPos );
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetWorldPos( const nw::math::VEC3& wPos )
{
    // ローカル位置に戻して保存
    nw::math::VEC3 lPos;
    TransformToLocalPos( &lPos, wPos );
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localPos, &lPos );
}

//---------------------------------------------------------------------------
//! @brief  ローカル速度を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetLocalVec( const f32 x, const f32 y, const f32 z )
{
    emitter->particlePosAttr[ ptclIndex ].localVec.x = x;
    emitter->particlePosAttr[ ptclIndex ].localVec.y = y;
    emitter->particlePosAttr[ ptclIndex ].localVec.z = z;
}

//---------------------------------------------------------------------------
//! @brief  ローカル速度を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetLocalVec( const nw::math::VEC3& vec )
{
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localVec, &vec );
}

//---------------------------------------------------------------------------
//! @brief  ワールド速度を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetWorldVec( const f32 x, const f32 y, const f32 z )
{
    nw::math::VEC3 lVec;
    TransformToLocalVec( &lVec, nw::math::VEC3( x, y, z ) );
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localVec, &lVec );
}

//---------------------------------------------------------------------------
//! @brief  ワールド速度を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetWorldVec( const nw::math::VEC3& worldVec )
{
    // ローカル速度に戻して保存。
    nw::math::VEC3 lVec;
    TransformToLocalVec( &lVec, worldVec );
    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[ ptclIndex ].localVec, &lVec );
}

//---------------------------------------------------------------------------
//! @brief  初期スケールを設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetInitScale( const nw::math::VEC3& scale )
{
    EFT_F32_VEC3_COPY( &emitter->particleAttr[ ptclIndex ].scale, &scale );
    MemUtil::FlushCache( &emitter->particleAttr[ ptclIndex ].scale, sizeof( nw::math::VEC3 ) );
}

//---------------------------------------------------------------------------
//! @brief  初期スケールXを設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetInitScaleX( const f32 scaleX )
{
    emitter->particleAttr[ ptclIndex ].scale.x = scaleX;

    MemUtil::FlushCache( &emitter->particleAttr[ ptclIndex ].scale, sizeof( nw::math::VEC2 ) );
}

//---------------------------------------------------------------------------
//! @brief  初期スケールYを設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetInitScaleY( const f32 scaleY )
{
    emitter->particleAttr[ ptclIndex ].scale.y = scaleY;

    MemUtil::FlushCache( &emitter->particleAttr[ ptclIndex ].scale, sizeof( nw::math::VEC2 ) );
}

//---------------------------------------------------------------------------
//! @brief  初期回転量を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetInitRotateRadian( const nw::math::VEC3& rotate )
{
    EFT_F32_VEC3_COPY( &emitter->particleAttr[ ptclIndex ].initRotate, &rotate );
    MemUtil::FlushCache( &emitter->particleAttr[ ptclIndex ].initRotate, sizeof( nw::math::VEC3 ) );
}

//---------------------------------------------------------------------------
//! @brief  初期回転量を設定
//---------------------------------------------------------------------------
inline void ParticleCalcArg::SetInitRotateDegree( const nw::math::VEC3& rotate )
{
    emitter->particleAttr[ ptclIndex ].initRotate.x = nw::math::DegToRad( rotate.x );
    emitter->particleAttr[ ptclIndex ].initRotate.y = nw::math::DegToRad( rotate.y );
    emitter->particleAttr[ ptclIndex ].initRotate.z = nw::math::DegToRad( rotate.z );
    MemUtil::FlushCache( &emitter->particleAttr[ ptclIndex ].initRotate, sizeof( nw::math::VEC3 ) );
}

//---------------------------------------------------------------------------
//! @brief  初期回転量を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetInitRotateRadian( nw::math::VEC3* outRotate ) const
{
    const nw::math::VEC4 rotate = emitter->particleAttrBack[ ptclIndex ].initRotate;
    EFT_F32_VEC3_COPY( outRotate, &rotate );
}

//---------------------------------------------------------------------------
//! @brief  初期回転量を取得
//---------------------------------------------------------------------------
inline void ParticleCalcArg::GetInitRotateDegree( nw::math::VEC3* outRotate ) const
{
    GetInitRotateRadian( outRotate );
    outRotate->x = nw::math::RadToDeg( outRotate->x );
    outRotate->y = nw::math::RadToDeg( outRotate->y );
    outRotate->z = nw::math::RadToDeg( outRotate->z );
}

#ifdef EFT_CALLBACK_ARG_OLD_API
//---------------------------------------------------------------------------
//! @brief  移動量差分を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC3& ParticleCalcArg::GetLocalDiff()
{
    GetLocalDiff( &ptcl.localDiff );
    return ptcl.localDiff;
}

//---------------------------------------------------------------------------
//! @brief  ローカル位置を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC3& ParticleCalcArg::GetLocalPos()
{
    GetLocalPos( &ptcl.localPos );
    return ptcl.localPos;
}

//---------------------------------------------------------------------------
//! @brief  ワールド位置を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC3& ParticleCalcArg::GetWorldPos()
{
    GetWorldPos( &ptcl.worldPos );
    return ptcl.worldPos;
}

//---------------------------------------------------------------------------
//! @brief  ローカル速度を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC3& ParticleCalcArg::GetLocalVec()
{
    GetLocalVec( &ptcl.localVec );
    return ptcl.localVec;
}

//---------------------------------------------------------------------------
//! @brief  ワールド速度を取得
//---------------------------------------------------------------------------
inline const nw::math::VEC3& ParticleCalcArg::GetWorldVec()
{
    GetWorldVec( &ptcl.worldVec );
    return ptcl.worldVec;
}

//---------------------------------------------------------------------------
//! @brief  回転量を計算
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::CalcCurrentRotateRadian()
{
    nw::math::VEC3 localRotate;
    CalcCurrentRotateRadian( &localRotate );
    EFT_F32_VEC3_COPY( &ptcl.localRotate, &localRotate );
    ptcl.localRotate.w = 0;
    return ptcl.localRotate;
}

//---------------------------------------------------------------------------
//! @brief  回転量を計算
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::CalcCurrentRotateDegree()
{
    ptcl.localRotate = CalcCurrentRotateRadian();
    ptcl.localRotate.x = nw::math::RadToDeg( ptcl.localRotate.x );
    ptcl.localRotate.y = nw::math::RadToDeg( ptcl.localRotate.y );
    ptcl.localRotate.z = nw::math::RadToDeg( ptcl.localRotate.z );
    return ptcl.localRotate;
}

//---------------------------------------------------------------------------
//! @brief  ワールドスケール値を計算
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::CalcWorldScale()
{
    nw::math::VEC3 worldScale;
    CalcWorldScale( &worldScale );
    EFT_F32_VEC3_COPY( &ptcl.worldScale, &worldScale );
    return ptcl.worldScale;
}

//---------------------------------------------------------------------------
//! @brief  カラー0を計算
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::CalcLocalColor0()
{
    CalcLocalColor0( &ptcl.localColor0 );
    return ptcl.localColor0;
}

//---------------------------------------------------------------------------
//! @brief  カラー1を計算
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::CalcLocalColor1()
{
    CalcLocalColor1( &ptcl.localColor1 );
    return ptcl.localColor1;
}

//---------------------------------------------------------------------------
//! @brief  ランダムを取得
//---------------------------------------------------------------------------
inline const nw::math::VEC4& ParticleCalcArg::GetRandom()
{
    GetRandom( &ptcl.random );
    return ptcl.random;
}
#endif

//---------------------------------------------------------------------------
//! @brief  コンストラクタ
//---------------------------------------------------------------------------
inline ParticleCalcArgImpl::ParticleCalcArgImpl( Emitter* const emitter, const f32 ptclTime,  const f32 ptclLife, const u32 ptclIndex )
    : userData( NULL )
    , emPluginData( NULL )
    , emitter( emitter )
    , ptclTime( ptclTime )
    , ptclLife( ptclLife )
    , ptclIndex( ptclIndex )
{
    EFT_NULL_ASSERT( emitter );
}

//---------------------------------------------------------------------------
//! @brief  パーティクル位置をワールド座標系に変換するための、追従設定を加味した変換行列を取得します。
//---------------------------------------------------------------------------
inline void ParticleCalcArgImpl::GetEmitterTransformMatrix( nw::math::MTX34* pOutMatrix ) const
{
    NW_NULL_ASSERT( pOutMatrix );
    emitter->GetEmitterTransformMatrix( pOutMatrix, ptclIndex );
}

} // namespace eft2
} // namespace nw
