﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_CallbackPtclCalcArg.h>
#include <nw/eft/eft2_Enum.h>

//------------------------------------------------------------------------------
namespace nw   {
namespace eft2 {

class System;
struct Shader;
struct EmitterSet;
struct Emitter;

//---------------------------------------------------
//! @brief パーティクル情報構造体です。
//---------------------------------------------------
struct Particle
{
    f32            time;           //!< パーティクル時間
    nw::math::VEC3 localDiff;      //!< ローカル系の移動差分
    nw::math::VEC3 localPos;       //!< ローカル位置
    nw::math::VEC3 worldPos;       //!< ワールド位置
    nw::math::VEC3 localVec;       //!< ローカル速度
    nw::math::VEC3 worldVec;       //!< ワールド速度
    nw::math::VEC4 localRotate;    //!< 初期回転量
    nw::math::VEC4 localScale;     //!< 初期スケール値
    nw::math::VEC4 worldScale;     //!< ワールドスケール値
    nw::math::VEC4 localColor0;    //!< カラー0（R/G/B/A）
    nw::math::VEC4 localColor1;    //!< カラー1（R/G/B/A）
    nw::math::VEC4 random;         //!< ランダム値（X/Y/Z/Wの4つ）
};

//----------------------------------------
//! @name エミッタ コールバック
//@{

//---------------------------------------------------
//! @brief エミッタ生成後コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterInitializeArg
{
    union
    {
        Emitter*            emitter;        //!< エミッタインスタンス
        Emitter*            pEmitter;       //!< エミッタインスタンス
    };
};

//---------------------------------------------------
//! @brief  エミッタ生成後コールバックの定義です。
//!         エミッタ生成後にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef bool (*EmitterInitializeCallback)( EmitterInitializeArg& arg );


//---------------------------------------------------
//! @brief エミッタ計算前コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterPreCalcArg
{
    union
    {
        Emitter*            emitter;        //!< エミッタインスタンス
        Emitter*            pEmitter;       //!< エミッタインスタンス
    };
};

//---------------------------------------------------
//! @brief  エミッタ計算前コールバックの定義です。
//!         エミッタ計算処理前にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void (*EmitterPreCalcCallback)( EmitterPreCalcArg& arg );


//---------------------------------------------------
//! @brief エミッタマトリクスセットコールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterMatrixSetArg
{
    union
    {
        Emitter*            emitter;        //!< エミッタインスタンス
        Emitter*            pEmitter;       //!< エミッタインスタンス
    };

    //---------------------------------------------------
    //! @brief 引数ビューマトリクスからエミッタビルボード用マトリクスを生成し設定します。
    //! @param[in] viewMatrix ビュー行列
    //---------------------------------------------------
    void SetEmitterBillboardMatrix( nw::math::Matrix34& viewMatrix );
};

//---------------------------------------------------
//! @brief  エミッタマトリクスセットコールバックの定義です。
//!         エミッタマトリクスセット後にコールされます。
//!         エミッタビルボードするためのマトリクス操作などを行えます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void( *EmitterMatrixSetCallback )( EmitterMatrixSetArg& arg );


//---------------------------------------------------
//! @brief エミッタ計算後コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterPostCalcArg
{
    union
    {
        Emitter*            emitter;        //!< エミッタインスタンス
        Emitter*            pEmitter;       //!< エミッタインスタンス
    };
};

//---------------------------------------------------
//! @brief  エミッタ計算後コールバックの定義です。
//!         エミッタ計算処理後にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void( *EmitterPostCalcCallback )( EmitterPostCalcArg& arg );


//---------------------------------------------------
//! @brief エミッタ描画処理コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterDrawArg
{
    union
    {
        Emitter*            emitter;        //!< エミッタインスタンス
        Emitter*            pEmitter;       //!< エミッタインスタンス
    };
    ShaderType              shaderType;     //!< シェーダタイプ
    union
    {
        void*               userParam;      //!< ユーザーデータ
        void*               pUserParam;     //!< ユーザーデータ
    };

    System* GetSystem();
};

//---------------------------------------------------
//! @brief  エミッタ描画処理コールバックの定義です。
//!         エミッタ描画処理時にコールされます。
//!         返り値に true を返すことで、通常のパーティクル描画も行われます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef bool( *EmitterDrawCallback )( EmitterDrawArg& arg );


//---------------------------------------------------
//! @brief エミッタ破棄後コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterFinalizeArg
{
    union
    {
        Emitter*    emitter;        //!< エミッタ
        Emitter*    pEmitter;       //!< エミッタ
    };
};

//---------------------------------------------------
//! @brief  エミッタ破棄後コールバックの定義です。
//!         エミッタ破棄後にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void( *EmitterFinalizeCallback )( EmitterFinalizeArg& arg );

//@}

//----------------------------------------
//! @name エミッタセット コールバック
//@{

//---------------------------------------------------
//! @brief  エミッタセット初期化時コールバックに引数として渡される構造体です。
//!         ここから操作対象のエミッタセット・エフェクトシステムにアクセスできます。
//---------------------------------------------------
struct EmitterSetInitializeArg
{
    union
    {
        System* system;             //!< システムへのポインタ
        System* pSystem;            //!< システムへのポインタ
    };
    union
    {
        EmitterSet* emitterSet;     //!< エミッタセットへのポインタ
        EmitterSet* pEmitterSet;    //!< エミッタセットへのポインタ
    };
};
//---------------------------------------------------
//! @brief      エミッタセット初期化時コールバックの定義です。
//! @param[in]  arg コールバック引数
//---------------------------------------------------
typedef void( *EmitterSetInitializeCallback )( EmitterSetInitializeArg& arg );

//---------------------------------------------------
//! @brief エミッタセット破棄時に渡される引数構造体です。
//! @param[in] arg コールバック引数
//---------------------------------------------------
struct EmitterSetFinalizeArg
{
    union
    {
        System* system;             //!< システムへのポインタ
        System* pSystem;            //!< システムへのポインタ
    };
    union
    {
        EmitterSet* emitterSet;     //!< エミッタセットへのポインタ
        EmitterSet* pEmitterSet;    //!< エミッタセットへのポインタ
    };
};
typedef void( *EmitterSetFinalizeCallback )( EmitterSetFinalizeArg& arg );

//@}


//----------------------------------------
//! @name 描画設定後 コールバック
//@{

//---------------------------------------------------
//! @brief 描画設定後コールバックに渡される引数構造体です。
//---------------------------------------------------
struct RenderStateSetArg
{
    //------------------------------------------------------------------------------
    //! @brief      シェーダを取得します。
    //! @return     シェーダクラスへのポインタ
    //------------------------------------------------------------------------------
    Shader* GetShader();

    //------------------------------------------------------------------------------
    //! @brief      Systemクラスへのポインタを取得します。
    //! @return     Systemクラスへのポインタ
    //------------------------------------------------------------------------------
    System* GetSystem();

    union
    {
        Emitter*            emitter;                //!< エミッタインスタンス
        Emitter*            pEmitter;               //!< エミッタインスタンス
    };
    ShaderType              shaderType;             //!< シェーダタイプ
    union
    {
        void*               userParam;              //!< ユーザーパラメータ
        void*               pUserParam;             //!< ユーザーパラメータ
    };
    union
    {
        bool                isStreamOut;            //!< ストリームアウトシェーダかどうか
        bool                isComputeShaderMode;    //!< コンピュートシェーダを計算させるかどうか。
    };
};

//---------------------------------------------------
//! @brief  描画設定後コールバックの定義です。
//!
//!         描画設定後にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef bool( *RenderStateSetCallback )( RenderStateSetArg& arg );

//@}



//----------------------------------------
//! @name エミッタ描画プロファイラコールバック
//@{

//---------------------------------------------------
//! @brief エミッタ描画プロファイラコールバックに渡される構造体です。
//---------------------------------------------------
struct DrawEmitterProfilerArg
{
    union
    {
        System*             system;                 //!< システム
        System*             pSystem;                //!< システム
    };
    union
    {
        EmitterSet*         emitterSet;             //!< エミッタセットインスタンス
        EmitterSet*         pEmitterSet;            //!< エミッタセットインスタンス
    };
    union
    {
        Emitter*            emitter;                //!< エミッタインスタンス
        Emitter*            pEmitter;               //!< エミッタインスタンス
    };
    bool                    beforeRenderEmitter;    //!< エミッタセットコール時のエミッタ描画処理前かどうか
    ShaderType              shaderType;             //!< シェーダタイプ
    union
    {
        void*               userParam;              //!< カスタムシェーダコールバックへ渡されるアドレス
        void*               pUserParam;             //!< カスタムシェーダコールバックへ渡されるアドレス
    };
    u32                     resourceID;             //!< リソースID
    bool                    calcStreamOut;          //!< ストリームアウトを計算させるかどうか。
    //bool                  cacheFlush;             //!< キャッシュフラッシュを行うかどうか
};

//---------------------------------------------------
//! @brief  エミッタ描画処理プロファイルコールバック。
//!
//!         エミッタ単位でのGPU処理負荷計測などに利用するコールバックです。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void( *DrawEmitterProfilerCallback )( DrawEmitterProfilerArg& arg );

//@}

//----------------------------------------
//! @name エンディアンフリップ コールバック
//@{

//---------------------------------------------------
//! @brief エンディアンフリップ　コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EndianFlipArg
{
    union
    {
        void*               customShaderParam;      //!< カスタムシェーダパラメータ
        void*               pCustomShaderParam;     //!< カスタムシェーダパラメータ
    };
    u32                     customShaderParamSize;  //!< カスタムシェーダパラメータサイズ
};

//---------------------------------------------------
//! @brief  エンディアンフリップコールバックの定義です。
//!
//!         リソース初期化時にコールされます。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef bool( *EndianFlipCallback )( EndianFlipArg& arg );

//@}


//----------------------------------------
//! @name 描画パス コールバック
//@{

//---------------------------------------------------
//! @brief  描画パス 描画ステート設定後コールバックの定義です。
//!
//!         描画ステート設定後にコールされます。
//!         各描画パスで特定の描画ステート設定などを行う場合に利用します。
//!         詳しくは、カスタムシェーダドキュメントをご覧ください。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef void( *DrawPathRenderStateSetCallback )( RenderStateSetArg& arg );

//@}

//----------------------------------------
//! @name カリング コールバック
//@{

//---------------------------------------------------
//! @brief エミッタ LOD コールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterCalcLodArg
{
    union
    {
        Emitter*        emitter;            //!< エミッタインスタンス
        Emitter*        pEmitter;           //!< エミッタインスタンス
    };
};

//---------------------------------------------------
//! @brief  エミッタ LOD コールバックの定義です。
//!
//!         コールバック内では、Emitter::calcSkipFlag/drawViewFlag の操作を行います。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef EmitterCalculationResult ( *EmitterCalcLodCallback )( EmitterCalcLodArg& arg );

//---------------------------------------------------
//! @brief エミッタカリングコールバックに渡される引数構造体です。
//---------------------------------------------------
struct EmitterDrawCullArg
{
    union
    {
        Emitter*        emitter;            //!< エミッタインスタンス
        Emitter*        pEmitter;           //!< エミッタインスタンス
    };
    union
    {
        void*           userParam;          //!< ユーザーパラメータ
        void*           pUserParam;         //!< ユーザーパラメータ
    };
};

//---------------------------------------------------
//! @brief  エミッタカリングコールバックの定義です。
//!
//!         コールバック内では、Emitter::calcSkipFlag/drawViewFlag の操作を行います。
//! @param[in] arg コールバック引数
//---------------------------------------------------
typedef bool( *EmitterDrawCullingCallback )( EmitterDrawCullArg& arg );

//@}

//---------------------------------------------------
//! @brief コールバックセット構造体です。
//---------------------------------------------------
struct CallbackSet
{
    EndianFlipCallback              endianFlip;
    EmitterInitializeCallback       emitterInitialize;
    union
    {
        EmitterPreCalcCallback      emitterPreCalc;
        EmitterPreCalcCallback      emitterPreCalculate;
    };
    union
    {
        EmitterMatrixSetCallback    emitterMtxSet;
        EmitterMatrixSetCallback    emitterMatrixSet;
    };
    union
    {
        EmitterPostCalcCallback     emitterPostCalc;
        EmitterPostCalcCallback     emitterPostCalculate;
    };
    EmitterDrawCallback             emitterDraw;
    EmitterFinalizeCallback         emitterFinalize;

    ParticleEmitCallback            particleEmit;
    ParticleRemoveCallback          particleRemove;
    union
    {
        ParticleCalcCallback        particleCalc;
        ParticleCalcCallback        particleCalculate;
    };
    RenderStateSetCallback          renderStateSet;

    //------------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //------------------------------------------------------------------------------
    CallbackSet()
        : endianFlip( NULL )
        , particleEmit( NULL )
        , particleRemove( NULL )
        , emitterInitialize( NULL )
        , particleCalc( NULL )
        , emitterPreCalc( NULL )
        , emitterMtxSet( NULL )
        , emitterPostCalc( NULL )
        , emitterDraw( NULL )
        , renderStateSet( NULL )
        , emitterFinalize( NULL )
    {}

    //------------------------------------------------------------------------------
    //! @brief  有効なコールバックが入っているかを取得
    //! @return コールバックが入っていればtrue
    //------------------------------------------------------------------------------
    bool IsValid()
    {
        bool ret = false;

        if ( endianFlip         ) ret = true;
        if ( particleEmit       ) ret = true;
        if ( particleRemove     ) ret = true;
        if ( emitterInitialize  ) ret = true;
        if ( particleCalc       ) ret = true;
        if ( emitterPreCalc     ) ret = true;
        if ( emitterPostCalc    ) ret = true;
        if ( emitterDraw        ) ret = true;
        if ( renderStateSet     ) ret = true;
        if ( emitterFinalize    ) ret = true;

        return ret;
    }
};

//------------------------------------------------------------------------------
//! @brief UBO反転 コールバック
//! @param[in] arg  コールバック引数
//! @return         正常に終了した場合trueを返します。
//------------------------------------------------------------------------------
bool _EndianFlipCallback( nw::eft2::EndianFlipArg& arg );

//------------------------------------------------------------------------------
//! @brief          描画設定後コールバック
//! @param[in] arg  コールバック引数
//! @return         正常に終了した場合trueを返します。
//------------------------------------------------------------------------------
bool _BindReservedCustomShaderUniformBlock( nw::eft2::RenderStateSetArg& arg );
bool BindReservedCustomShaderConstantBuffer( nw::eft2::RenderStateSetArg& arg );

//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//! @brief コールバックの呼び出し処理の集約（RenderStateSet）
//! @param[in] arg  コールバック引数
//! @return         正常に終了した場合trueを返します。
//------------------------------------------------------------------------------
bool InvokeRenderStateSetCallback( nw::eft2::RenderStateSetArg& arg );

//------------------------------------------------------------------------------
//! @brief          コールバックの呼び出し処理の集約（エミッタ初期化時）
//! @param[in] arg  コールバック引数
//! @return         正常に終了した場合trueを返します。
//------------------------------------------------------------------------------
bool InvokeEmitterInitializeCallback( nw::eft2::EmitterInitializeArg& arg );

//------------------------------------------------------------------------------
//! @brief          コールバックの呼び出し処理の集約（エミッタ解放時）
//! @param[in] arg  コールバック引数
//------------------------------------------------------------------------------
void InvokeEmitterFinalizeCallback( nw::eft2::EmitterFinalizeArg& arg );

//------------------------------------------------------------------------------
//! @brief              エミッタマトリクス設定後のコールバック呼び出し
//! @param[in] emitter  エミッタへのポインタ
//------------------------------------------------------------------------------
void InvokeEmitterMatrixSetCallback( Emitter* emitter );

//------------------------------------------------------------------------------
//! @brief          コールバックの呼び出し処理の集約（パーティクル計算前）
//! @param[in] arg  コールバック引数
//------------------------------------------------------------------------------
void InvokeEmitterPreCalcCallback( nw::eft2::EmitterPreCalcArg& arg );

//------------------------------------------------------------------------------
//! @brief              コールバックの呼び出し処理の集約（パーティクル計算後）
//! @param[in] emitter  エミッタへのポインタ
//------------------------------------------------------------------------------
void InvokeEmitterPostCalcCallback( Emitter* emitter );

//------------------------------------------------------------------------------
//! @brief          コールバックの呼び出し処理の集約（エミッタ描画時）
//! @param[in] arg  コールバック引数
//! @return         正常に終了した場合trueを返します。
//------------------------------------------------------------------------------
bool InvokeEmitterDrawCallback( nw::eft2::EmitterDrawArg& arg );

//------------------------------------------------------------------------------
//! @brief              コールバックの呼び出し処理の集約（全パーティクル削除時）
//! @param[in] emitter  エミッタへのポインタ
//------------------------------------------------------------------------------
void InvokeParticleRemoveCallbackAll( Emitter* emitter );

//------------------------------------------------------------------------------
} // namespace eft2
} // namespace nw

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//! @brief パーティクル放出時のコールバック呼び出し
//! @param[in] emitter      エミッタへのポインタ
//! @param[in] writeIndex   自身が格納されるパーティクル配列のインデックス
//! @param[in] ptclEmtCbEP  エミッタプラグインコールバックへのポインタ
//! @param[in] ptclEmtCbCA  カスタムアクションコールバックへのポインタ
//! @param[in] ptclEmtCbCS  カスタムシェーダコールバックへのポインタ
//---------------------------------------------------------------------------
inline void InvokeParticleEmitCallback( Emitter*                emitter,
                                        u32                     writeIndex,
                                        ParticleEmitCallback    ptclEmtCbEP,
                                        ParticleEmitCallback    ptclEmtCbCA,
                                        ParticleEmitCallback    ptclEmtCbCS )
{
    //放出された瞬間なのでタイムは0
    const f32 time   = 0;
    const f32 life   = emitter->particlePosAttr[writeIndex].localPos.w;
    ParticleEmitArg arg( emitter, time, life, writeIndex );

    if ( ptclEmtCbEP ) { ptclEmtCbEP( arg ); }
    if ( ptclEmtCbCA ) { ptclEmtCbCA( arg ); }
    else if ( ptclEmtCbCS ) { ptclEmtCbCS( arg ); }

    // 返り値のユーザーデータを保持する
    emitter->particleData[writeIndex].emitterPluginData = arg.emPluginData;
    emitter->particleData[writeIndex].userData          = arg.userData;
}

//---------------------------------------------------------------------------
//! @brief パーティクル計算時のコールバック呼び出し
//! @param[in] emitter      エミッタへのポインタ
//! @param[in] index        自身が格納されているパーティクル配列のインデックス
//! @param[in] time         パーティクル時間
//! @param[in] life         パーティクル寿命
//! @param[in] ptclClcCbEP  エミッタプラグインコールバックへのポインタ
//! @param[in] ptclClcCbCA  カスタムアクションコールバックへのポインタ
//! @param[in] ptclClcCbCS  カスタムシェーダコールバックへのポインタ
//---------------------------------------------------------------------------
inline void InvokeParticleCalcCallback( Emitter* emitter,
                                        const u32 index,
                                        const f32 time,
                                        const f32 life,
                                        ParticleCalcCallback ptclClcCbEP,
                                        ParticleCalcCallback ptclClcCbCA,
                                        ParticleCalcCallback ptclClcCbCS )
{
    ParticleCalcArg arg( emitter, time, life, index );
    arg.userData        = emitter->particleData[ index ].userData;
    arg.emPluginData    = emitter->particleData[ index ].emitterPluginData;

    if ( ptclClcCbEP ) { ptclClcCbEP( arg ); }
    if ( ptclClcCbCA ) { ptclClcCbCA( arg ); }
    if ( ptclClcCbCS ) { ptclClcCbCS( arg ); }

    emitter->particleData[ index ].userData = arg.userData;
}

//---------------------------------------------------------------------------
//! @brief パーティクル削除時のコールバック呼び出し
//! @param[in] emitter      エミッタへのポインタ
//! @param[in] index        自身が格納されているパーティクル配列のインデックス
//! @param[in] time         パーティクル時間
//! @param[in] life         パーティクル寿命
//! @param[in] ptclRmvCbEP  エミッタプラグインコールバックへのポインタ
//! @param[in] ptclRmvCbCA  カスタムアクションコールバックへのポインタ
//! @param[in] ptclRmvCbCS  カスタムシェーダコールバックへのポインタ
//---------------------------------------------------------------------------
inline void InvokeParticleRemoveCallback( Emitter* emitter,
                                   const u32 index,
                                   const f32 time,
                                   const f32 life,
                                   ParticleRemoveCallback ptclRmvCbEP,
                                   ParticleRemoveCallback ptclRmvCbCA,
                                   ParticleRemoveCallback ptclRmvCbCS )
{
    ParticleRemoveArg arg( emitter, time, life, index );
    arg.userData        = emitter->particleData[ index ].userData;
    arg.emPluginData    = emitter->particleData[ index ].emitterPluginData;

    if ( ptclRmvCbEP ) { ptclRmvCbEP( arg ); }
    if ( ptclRmvCbCA ) { ptclRmvCbCA( arg ); }
    if ( ptclRmvCbCS ) { ptclRmvCbCS( arg ); }

    // ユーザーデータの参照を消去。
    arg.emitter->particleData[ index ].emitterPluginData = NULL;
    arg.emitter->particleData[ index ].userData = NULL;
}

//------------------------------------------------------------------------------
// エミッタ描画時のコールバック呼び出し
//------------------------------------------------------------------------------
inline bool InvokeEmitterDrawCallback( nw::eft2::EmitterDrawArg& arg )
{
    if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_EP ] && arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_EP ]->emitterDraw )
    {
        if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_EP ]->emitterDraw( arg ) )
        {
            return true;
        }
    }
    if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CA ] && arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CA ]->emitterDraw )
    {
        if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CA ]->emitterDraw( arg ) )
        {
            return true;
        }
    }
    if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CS ] && arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CS ]->emitterDraw )
    {
        if( arg.emitter->callbackSet[ EFT_CALLBACK_SET_TYPE_CS ]->emitterDraw( arg ) )
        {
            return true;
        }
    }
    return false;
}

//------------------------------------------------------------------------------
// エミッタ計算前のコールバック呼び出し
//------------------------------------------------------------------------------
inline void InvokeEmitterPreCalcCallback( nw::eft2::EmitterPreCalcArg& arg )
{
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterPreCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterPreCalc( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterPreCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterPreCalc( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterPreCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterPreCalc( arg );
    }
}

//------------------------------------------------------------------------------
// エミッタマトリクス設定後のコールバック呼び出し
//------------------------------------------------------------------------------
inline void InvokeEmitterMatrixSetCallback( Emitter* emitter )
{
    nw::eft2::EmitterMatrixSetArg arg;
    arg.emitter = emitter;

    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterMtxSet )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterMtxSet( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterMtxSet )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterMtxSet( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterMtxSet )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterMtxSet( arg );
    }
}

//------------------------------------------------------------------------------
// エミッタ計算後のコールバック呼び出し
//------------------------------------------------------------------------------
inline void InvokeEmitterPostCalcCallback( Emitter* emitter )
{
    nw::eft2::EmitterPostCalcArg arg;
    arg.emitter = emitter;

    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterPostCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->emitterPostCalc( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterPostCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->emitterPostCalc( arg );
    }
    if( arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] && arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterPostCalc )
    {
        arg.emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->emitterPostCalc( arg );
    }
}

} // namespace eft2
} // namespace nw
