﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DW_WINDOW_H_
#define NW_DW_WINDOW_H_

#include <nw/types.h>
#include <nw/ut.h>
#include <nw/dev/dev_Pad.h>
#include <nw/dw/system/dw_Inputs.h>
#include <nw/dw/system/dw_NwUIRenderer.h>
#include <nw/dw/window/dw_WindowContext.h>

namespace nw {
namespace internal {
namespace dw {

//! @details :private
class Window
{
public:
    //! @details :private
    static const int TITLE_LEN_MAX = 64;

private:
    static const f32 FRAME_WIDTH;         // フレームの幅
    static const f32 TITLE_HEIGHT;        // タイトルの高さ
    static const f32 CLIENT_OUT_WIDTH;    // クライアント領域外の幅
    static const f32 CLIENT_OUT_HEIGHT;   // クライアント領域外の高さ

    static const f32 WIDTH_MAX;
    static const f32 WIDTH_MIN;
    static const f32 HEIGHT_MAX;
    static const f32 HEIGHT_MIN;
    static const f32 TITLE_PADDING_LEFT;

public:
    Window();
    Window( const char* title, f32 width, f32 height );
    virtual ~Window();

public:
    //! @details :private
    bool Create( f32 x, f32 y );
    //! @details :private
    bool Create( const nw::math::Vector2& pos );
    //! @details :private
    void Destroy();
    //! @details :private
    bool IsCreated() const { return m_IsCreated; }

    // 更新
    //! @details :private
    void UpdateInputs(const nw::internal::dw::Inputs& inputs);
    //! @details :private
    void Update(UIRenderer& renderer);

    // 描画
    //! @details :private
    void Draw( WindowContext& context, UIRenderer& renderer );
    //! @details :private
    void Draw( WindowContext& context, UIRenderer& renderer, float alpha, bool isShade );

    // ウィンドウ位置
    //! @details :private
    const nw::math::Vector2& GetPosition() const { return m_Pos; }
    //! @details :private
    void SetPosition( const nw::math::Vector2& pos ) { m_Pos = pos; }

    // ウィンドウサイズ
    //! @details :private
    const nw::math::Vector2& GetSize() const { return m_Size; }

    //---------------------------------------------------------------------------
    //! @brief    ウィンドウの横幅を取得します。
    //!
    //! @return   横幅です。
    //! @date 2012/02/17 初版
    //---------------------------------------------------------------------------
    f32 GetWidth() const { return m_Size.x; }

    //---------------------------------------------------------------------------
    //! @brief    ウィンドウの縦幅を取得します。
    //!
    //! @return   縦幅です。
    //! @date 2012/02/17 初版
    //---------------------------------------------------------------------------
    f32 GetHeight() const { return m_Size.y; }

    //! @details :private
    void SetSize( f32 width, f32 height );
    //! @details :private
    void SetSize( const nw::math::Vector2& size );
    //! @details :private
    void SetSizeFromClientSize( f32 width, f32 height );
    //! @details :private
    void SetSizeFromClientSize( const nw::math::Vector2& clientSize );

    //! @details :private
    const ut::Rect GetClientRect() const;
    //! @details :private
    const nw::math::Vector2 GetClientSize() const;

    //! @details :private
    const nw::math::Vector2& GetMinimumSize() const { return m_MinimumSize; }
    //! @details :private
    void SetMinimumSize( const nw::math::Vector2& size ) { m_MinimumSize = size; }
    //! @details :private
    const nw::math::Vector2& GetMaximumSize() const { return m_MaximumSize; }
    //! @details :private
    void SetMaximumSize( const nw::math::Vector2& size ) { m_MaximumSize = size; }

    //! @details :private
    const nw::math::Vector2 MeasureTitle(UIRenderer& renderer) const;

    // ウィンドウタイトル
    //! @details :private
    const char* GetTitle() const { return m_Title; }
    //! @details :private
    void SetTitle( const char* title );
    //! @details :private
    f32 GetTitleBarHeight() const { return TITLE_HEIGHT + FRAME_WIDTH * 2; }

    // カラー
    //! @details :private
    NW_DEPRECATED_FUNCTION(const ut::Color4f& GetBackgroundColor() const) { return m_BackgroundColor0; }
    //! @details :private
    Window& SetBackgroundColor(const ut::Color4f& color)
    {
        m_BackgroundColor0 = color;
        m_BackgroundColor1 = color;
        return *this;
    }
    //! @details :private
    Window& SetBackgroundColor(const ut::Color4f& color0, const ut::Color4f& color1)
    {
        m_BackgroundColor0 = color0;
        m_BackgroundColor1 = color1;
        return *this;
    }
    //! @details :private
    const ut::Color4f& GetBackgroundColor0() const { return m_BackgroundColor0; }
    //! @details :private
    Window& SetBackgroundColor0(const ut::Color4f& color)
    {
        m_BackgroundColor0 = color;
        return *this;
    }
    //! @details :private
    const ut::Color4f& GetBackgroundColor1() const { return m_BackgroundColor1; }
    //! @details :private
    Window& SetBackgroundColor1(const ut::Color4f& color)
    {
        m_BackgroundColor1 = color;
        return *this;
    }

    // ウィンドウ状態
    //! @details :private
    bool GetIsActive() const { return m_IsActive; }
    //! @details :private
    void SetIsActive( bool value ) { m_IsActive = value; }
    //! @details :private
    bool GetIsVisible() const { return m_IsVisible; }
    //! @details :private
    void SetIsVisible( bool value ) { m_IsVisible = value; }
    //! @details :private
    bool GetIsShade() const { return m_IsShade; }
    //! @details :private
    void SetIsShade( bool value ) { m_IsShade = value; }
    //! @details :private
    bool GetHasCloseButton() const { return m_HasCloseButton; }
    //! @details :private
    void SetHasCloseButton( bool value ) { m_HasCloseButton = value; }
    //! @details :private
    bool GetIsClosed() const { return m_IsClosed; }
    //! @details :private
    void SetIsClosed( bool value ) { m_IsClosed = value; }

    // ウィンドウオーダー
    //! @details :private
    void detail_SetWindowListOrder( int order ) { m_WindowOrder = order; }
    //! @details :private
    int detail_GetWindowListOrder() const { return m_WindowOrder; }
    //! @details :private
    void detail_SetWindowZOrder( int order ) { m_ZOrder = order; }
    //! @details :private
    int detail_GetWindowZOrder() const { return m_ZOrder; }

protected:
    //! @details :private
    virtual void OnUpdateInputs(const nw::internal::dw::Inputs& inputs) { (void)inputs; }
    //! @details :private
    virtual void OnUpdate(UIRenderer& renderer) { (void)renderer; }
    //! @details :private
    virtual void OnDraw( IUIRenderContext& context, UIRenderer& renderer) { (void)context; (void)renderer; }

    //! @details :private
    virtual bool OnCreate() { return true; }
    //! @details :private
    virtual void OnDestroy() { }

    //! @details :private
    virtual void OnSize() { }

protected:
    //! @details :private
    virtual void DrawImpl( WindowContext& context, UIRenderer& renderer, float alpha, bool isShade );
    //! @details :private
    void DrawFrame( WindowContext& context, UIRenderer& renderer, float alpha, bool isShade );
    //! @details :private
    void DrawTitle( WindowContext& context, UIRenderer& renderer, float alpha, bool isShade );

private:
    void DrawCloseButton( UIRenderer& renderer, f32 alpha);

private:
    bool m_IsCreated;
    bool m_IsActive;
    bool m_IsShade;
    bool m_IsVisible;
    bool m_HasCloseButton;
    bool m_IsClosed;
    char m_Title[ TITLE_LEN_MAX ];
    ut::Color4f m_BackgroundColor0;
    ut::Color4f m_BackgroundColor1;
    nw::math::Vector2 m_Pos;
    nw::math::Vector2 m_Size;
    nw::math::Vector2 m_MinimumSize;
    nw::math::Vector2 m_MaximumSize;

    int m_WindowOrder;
    int m_ZOrder;

public: // for WindowManager
    //! @details :private
    ut::LinkListNode m_WindowListLink;
    //! @details :private
    ut::LinkListNode m_ZOrderLink;
};

//! @details :private
typedef ut::LinkList< Window, offsetof(Window,m_WindowListLink)> WindowList;
//! @details :private
typedef ut::LinkList< Window, offsetof(Window,m_ZOrderLink)> WindowZOrderList;

} // namespace dw
} // namespace internal
} // namespace nw

#endif // NW_DW_WINDOW_H_
