﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/dw/system/dw_DrawRectangleArgs.h>

namespace nw {
namespace internal {
namespace dw {

class NwTypeUtility
{
public:
    static nw::math::Vector3 ToVector3(const nw::math::Vector2& value);
    static nw::dev::PrimitiveRenderer::QuadArg ToQuadArg(const DrawRectangleArgs& value);

    //! @brief カラー値を SRGB 空間から Linear 空間へ変換します。
    static f32 SRGBToLinear(f32 src);

    //! @brief カラー値を Linear 空間から SRGB 空間へ変換します。
    static f32 LinearToSRGB(f32 src);

    //! @brief カラー値を SRGB 空間から Linear 空間へ変換します。
    static u8 SRGBToLinear(u8 src)
    {
        f32 c = SRGBToLinear(static_cast<f32>(src) * (1.f / 255.f));
        return static_cast<u8>(c * 255.f);
    }

    //! @brief カラー値を Linear 空間から SRGB 空間へ変換します。
    static u8 LinearToSRGB(u8 src)
    {
        f32 c = LinearToSRGB(static_cast<f32>(src) * (1.f / 255.f));
        return static_cast<u8>(c * 255.f);
    }

    //! @brief カラー値を SRGB 空間から Linear 空間へ変換します。
    static NW_INLINE nw::ut::Color4u8 SRGBToLinear(const nw::ut::Color4u8& src)
    {
        return nw::ut::Color4u8(SRGBToLinear(src.r), SRGBToLinear(src.g), SRGBToLinear(src.b), src.a);
    }

    //! @brief カラー値を Linear 空間から SRGB 空間へ変換します。
    static NW_INLINE nw::ut::Color4u8 LinearToSRGB(const nw::ut::Color4u8& src)
    {
        return nw::ut::Color4u8(LinearToSRGB(src.r), LinearToSRGB(src.g), LinearToSRGB(src.b), src.a);
    }

    //! @brief カラー値を SRGB 空間から Linear 空間へ変換します。
    static NW_INLINE nw::ut::Color4f SRGBToLinear(const nw::ut::Color4f& src)
    {
        return nw::ut::Color4f(SRGBToLinear(src.r), SRGBToLinear(src.g), SRGBToLinear(src.b), src.a);
    }

    //! @brief カラー値を Linear 空間から SRGB 空間へ変換します。
    static NW_INLINE nw::ut::Color4f LinearToSRGB(const nw::ut::Color4f& src)
    {
        return nw::ut::Color4f(LinearToSRGB(src.r), LinearToSRGB(src.g), LinearToSRGB(src.b), src.a);
    }

    //! @brief カラー値を SRGB 空間から Linear 空間へ変換します。
    static NW_INLINE nw::ut::Color4u8 SRGBToLinear(u32 src)
    {
        return SRGBToLinear(nw::ut::Color4u8(src));
    }

    //! @brief カラー値を Linear 空間から SRGB 空間へ変換します。
    static NW_INLINE nw::ut::Color4u8 LinearToSRGB(u32 src)
    {
        return LinearToSRGB(nw::ut::Color4u8(src));
    }
};

} // dw
} // internal
} // nw
