﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DW_TAB_CONTROL_H_
#define NW_DW_TAB_CONTROL_H_

#include <nw/dw/control/dw_UIElement.h>
#include <nw/dw/control/dw_StackPanel.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief     タブコントロールの基底クラスです。
//---------------------------------------------------------------------------
class TabControl : public UIElement
{
public:
    //---------------------------------------------------------------------------
    //! @brief     コンストラクタです。
    //---------------------------------------------------------------------------
    TabControl();


    //---------------------------------------------------------------------------
    //! @brief     タブアイテムを追加します。
    //!
    //! @param[in] pItem 追加するタブアイテムです。
    //! @param[in] tabName 追加するタブアイテムの名前です。
    //---------------------------------------------------------------------------
    virtual void AddTabItem( UIElement* pItem, const char* tabName ) = 0;

    //---------------------------------------------------------------------------
    //! @brief     タブアイテムの数を取得します。
    //!
    //! @return    タブアイテムの数です。
    //---------------------------------------------------------------------------
    virtual const s32 GetTabCount() const = 0;

    //---------------------------------------------------------------------------
    //! @brief     タブが境界をもつかどうかを設定します。
    //!
    //! @param[in] value 境界を持つかどうかです。
    //---------------------------------------------------------------------------
    virtual void SetTabIsBordered( bool value ) { (void)value; }

    //---------------------------------------------------------------------------
    //! @brief     選択中タブの背景色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    virtual void SetTabSelectedBackgroundColor( bool value ) { (void)value; }

    //---------------------------------------------------------------------------
    //! @brief     タブの背景色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    virtual void SetTabBackgroundColor( bool value ) { (void)value; }

    //---------------------------------------------------------------------------
    //! @brief     選択中タブの文字色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    virtual void SetTabSelectedTextColor( bool value ) { (void)value; }

    //---------------------------------------------------------------------------
    //! @brief     タブの文字色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    virtual void SetTabTextColor( bool value ) { (void)value; }


    //---------------------------------------------------------------------------
    //! @brief     タブパネルのサイズ決定方法を設定します。
    //!
    //! @param[in] value 設定するサイズの決定方法です。
    //---------------------------------------------------------------------------
    void SetTabPanelMeasurement( Measurement value )
    {
        m_TabPanel.SetMeasurement( value );
    }

    //---------------------------------------------------------------------------
    //! @brief     タブパネルの幅を設定します。
    //!
    //! @param[in] value 設定する幅です。
    //---------------------------------------------------------------------------
    void SetTabPanelWidth( f32 value )
    {
        m_TabPanel.SetWidth( value );
    }

    //---------------------------------------------------------------------------
    //! @brief     タブパネルの高さを設定します。
    //!
    //! @param[in] value 設定する高さです。
    //---------------------------------------------------------------------------
    void SetTabPanelHeight( f32 value )
    {
        m_TabPanel.SetHeight( value );
    }

    //---------------------------------------------------------------------------
    //! @brief     タブパネルが境界をもつかどうかを設定します。
    //!
    //! @param[in] value 境界を持つかどうかです。
    //---------------------------------------------------------------------------
    void SetTabPanelIsBordered( bool value )
    {
        m_TabPanel.SetIsBordered( value );
    }

    //---------------------------------------------------------------------------
    //! @brief     タブパネルの境界色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    void SetTabPanelBorderColor( nw::ut::Color4f value )
    {
        m_TabPanel.SetBorderColor( value );
    }

    //---------------------------------------------------------------------------
    //! @brief     タブパネルの背景色を設定します。
    //!
    //! @param[in] value 設定する色です。
    //---------------------------------------------------------------------------
    void SetTabPanelBackgroundColor( nw::ut::Color4f value )
    {
        m_TabPanel.SetBackgroundColor( value );
    }


protected:
    //---------------------------------------------------------------------------
    //! @brief       パッド入力を更新します。
    //!
    //! @param[in]   inputs  nw::dev::Inputs を指定します。
    //!
    //! @return      入力を処理した場合は true、処理しなかった場合は false を返します。
    //!              false を返すと、親UI要素に処理を委譲します。
    //---------------------------------------------------------------------------
    /*override*/ bool OnUpdateFocusedInput( const nw::internal::dw::Inputs& inputs );

    //---------------------------------------------------------------------------
    //! @brief       タブの選択を指定オフセットだけ移動します。
    //!
    //! @details     タブの選択を指定オフセットだけ移動します。
    //!
    //! @param[in]   offset  オフセット値を指定します。
    //---------------------------------------------------------------------------
    void MoveSelection( s32 offset );

    //---------------------------------------------------------------------------
    //! @brief       指定タブを選択します。
    //!
    //! @details     指定タブを選択します。
    //!
    //! @param[in]   index  選択対象のインデックスを指定します。
    //---------------------------------------------------------------------------
    virtual void SelectTab( s32 index ) = 0;


    static const Measurement DEFAULT_MEASUREMENT;
    static const f32         DEFAULT_TAB_PANEL_WIDTH;
    static const f32         DEFAULT_TAB_PANEL_HEIGHT;
    static const bool        DEFAULT_TAB_PANEL_IS_BORDERED;

    StackPanel      m_TabPanel;
    bool            m_TabsBordered;
    nw::ut::Color4f m_TabsSelectedBackgroundColor;
    nw::ut::Color4f m_TabsBackgroundColor;
    nw::ut::Color4f m_TabsSelectedTextColor;
    nw::ut::Color4f m_TabsTextColor;
    s32             m_CurrentTabIndex;


#if defined(NW_DEBUG) || defined(NW_DEVELOP)
public:
    virtual const char* ToString() const
    {
        return "TabControl";
    }
#endif
};

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */


#endif //  NW_DW_TAB_CONTROL_H_
