﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef DW_NUMERIC_PARAMETER_LIST_ITEM_H_
#define DW_NUMERIC_PARAMETER_LIST_ITEM_H_

#include <nw/dw/control/dw_ParameterListItem.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief     数値を扱うパラメータリストアイテムクラスです。
//---------------------------------------------------------------------------
template<class TModel, typename TValue>
class NumericParameterListItem : public ModelParameterListItem<TModel>
{
private:
    typedef ModelParameterListItem<TModel> Base;

public:
    typedef TValue (*GetValueHandler)(TModel* pModel);
    typedef void (*SetValueHandler)(TModel* pModel, TValue value);
    typedef void (*PrintValueHandler)(TModel* pModel, char* pText, u32 bufferLength, TValue value);


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    NumericParameterListItem();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~NumericParameterListItem() {}


    //---------------------------------------------------------------------------
    //! @brief        初期化処理を行います。
    //!
    //! @param[in]    pGetValueHandler   値取得ハンドラです。
    //! @param[in]    pSetValueHandler   値設定ハンドラです。
    //! @param[in]    pPrintValueHandler 描画ハンドラです。
    //---------------------------------------------------------------------------
    void Initialize(
        GetValueHandler pGetValueHandler,
        SetValueHandler pSetValueHandler,
        PrintValueHandler pPrintValueHandler
    );

    //---------------------------------------------------------------------------
    //! @brief        値を取得します。
    //!
    //! @return       値を返します。
    //---------------------------------------------------------------------------
    TValue GetValue() const;

    //---------------------------------------------------------------------------
    //! @brief        値を設定します。
    //!
    //! @param[in]    value     設定する値です。
    //---------------------------------------------------------------------------
    void SetValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        デフォルトの値を取得します。
    //!
    //! @return       TValue 型の値を返します。
    //---------------------------------------------------------------------------
    TValue GetDefaultValue() const;

    //---------------------------------------------------------------------------
    //! @brief        デフォルトの値を設定します。
    //!
    //! @param[in]    value     TValue 型の値です。
    //---------------------------------------------------------------------------
    void SetDefaultValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        値の変化量を取得します。
    //!
    //! @return       値の変化量を返します。
    //---------------------------------------------------------------------------
    TValue GetValueChange() const;

    //---------------------------------------------------------------------------
    //! @brief        値の変化量を設定します。
    //!
    //! @param[in]    value     設定する値の変化量です。
    //---------------------------------------------------------------------------
    void SetValueChange(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        大きく変化するときの値の変化量を取得します。
    //!
    //! @return       大きく変化するときの値の変化量を返します。
    //---------------------------------------------------------------------------
    TValue GetValueLargeChange() const;

    //---------------------------------------------------------------------------
    //! @brief        大きく変化するときの値の変化量を設定します。
    //!
    //! @param[in]    value     設定する大きく変化するときの値の変化量です。
    //---------------------------------------------------------------------------
    void SetValueLargeChange(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        値の範囲の最小値を取得します。
    //!
    //! @return       値の範囲の最小値を返します。
    //---------------------------------------------------------------------------
    TValue GetMinimumValue() const;

    //---------------------------------------------------------------------------
    //! @brief        値の範囲の最小値を設定します。
    //!
    //! @param[in]    value     設定する値の範囲の最小値です。
    //---------------------------------------------------------------------------
    void SetMinimumValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        値の範囲の最大値を取得します。
    //!
    //! @return       値の範囲の最大値を返します。
    //---------------------------------------------------------------------------
    TValue GetMaximumValue() const;

    //---------------------------------------------------------------------------
    //! @brief        値の範囲の最大値を設定します。
    //!
    //! @param[in]    value     設定する値の範囲の最大値です。
    //---------------------------------------------------------------------------
    void SetMaximumValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        値をリセットします。
    //---------------------------------------------------------------------------
    /*override*/ void ResetValue();


protected:
    //! @brief 次の値にします。
    /*override*/ void NextValue(bool isLargeChange);
    //! @brief 前の値にします。
    /*override*/ void PreviousValue(bool isLargeChange);
    //! @brief 値を表示します。
    /*override*/ void PrintValue(char* pText, u32 bufferLength);

    TValue m_DefaultValue;
    TValue m_ValueChange;
    TValue m_ValueLargeChange;
    TValue m_MinimumValue;
    TValue m_MaximumValue;

    GetValueHandler   m_pGetValueHandler;
    SetValueHandler   m_pSetValueHandler;
    PrintValueHandler m_pPrintValueHandler;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
public:
    virtual const char* ToString() const
    {
        return "NumericParameterListItem";
    }

#endif
};

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */

#include "dw_NumericParameterListItem.hpp"

#endif // DW_NUMERIC_PARAMETER_LIST_ITEM_H_
