﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut/ut_Delegate.h>
#include <nw/dw/control/dw_UIElement.h>
#include <nw/dw/control/dw_FixedUIElementList.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief  任意の UI 要素をアイテムに格納する ListBox クラスです。
//---------------------------------------------------------------------------
class ListBox : public UIElement
{
private:
    typedef UIElement Base;

    static const s32 INVALID_INDEX = -1;

public:
    typedef ut::DelegateEvent0 SelectionChangedEvent;
    typedef SelectionChangedEvent::Slot SelectionChangedEventHandler;

    //---------------------------------------------------------------------------
    //! @brief  ListBoxItem の状態等を管理するプレースホルダクラスです。
    //---------------------------------------------------------------------------
    class ListBoxItemPlaceHolder : public UIElement
    {
    private:
        typedef UIElement Base;

        friend class ListBox;

    public:
        ListBoxItemPlaceHolder();

    public:
        bool GetIsSelected() const;
        void SetIsSelected(bool value);

        bool GetIsSelectable() const;
        void SetIsSelectable(bool value);

        UIElement* GetContent() const;
        void SetContent(UIElement* pContent);

    protected:
        /* override */ bool OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs);

        //---------------------------------------------------------------------------
        //! @brief       UI要素描画の準備を行います。
        //!
        //! @details     UI要素描画の準備を行います。
        //!              特別な更新処理を行う場合は、この関数をオーバーライドしてください。
        //!
        //! @param[in]   context  UIElementTreeContext を指定します。
        //!
        //! @return      描画に使用するパラメータへの参照を返します。
        //---------------------------------------------------------------------------
        /*override*/ UIElementRenderArgs& OnPrepareRender(const UIElementTreeContext& context);

    private:
        void SetOwner(ListBox* pOwner);
        void CallbackSelectionChanged();

    private:
        bool m_isSelected;
        bool m_isSelectable;

        FixedUIElementList<1> m_Contents;
        ListBox* m_pOwner;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    public:
        /*override*/ const char* ToString() const
        {
            return "ListBoxItemPlaceHolder";
        }
#endif
    };

public:
    ListBox();

public:
    s32 GetItemCount() const;

    virtual ListBoxItemPlaceHolder& GetItemPlaceHolder(s32 index) = 0;
    virtual const ListBoxItemPlaceHolder& GetItemPlaceHolder(s32 index) const = 0;

    UIElement* GetItem(s32 index) const;
    ListBox& AddItem(UIElement* pItem, bool isSelectable = true);
    ListBox& InsertItem(s32 index, UIElement* pItem, bool isSelectable = true);
    ListBox& SetItem(s32 index, UIElement* pItem);
    void ClearItems();
    s32 IndexOf(const UIElement* pItem) const;

    SelectionChangedEvent& GetSelectionChangedEvent();

    //---------------------------------------------------------------------------
    //! @brief       フォーカスを持つアイテムを取得します。
    //!
    //! @return      フォーカスを持つアイテムへのポインタを返します。
    //!              フォーカスを持つアイテムが無いときは NULL を返します。
    //---------------------------------------------------------------------------
    UIElement* GetFocusedItem() const;

    //---------------------------------------------------------------------------
    //! @brief       フォーカスを持つアイテムのインデックスを取得します。
    //!
    //! @return      フォーカスを持つアイテムのインデックスを返します。
    //!              フォーカスを持つアイテムが無いときは -1 を返します。
    //---------------------------------------------------------------------------
    s32 GetFocusedItemIndex() const;

protected:
    //---------------------------------------------------------------------------
    //! @brief       アイテムリストを取得します。
    //!
    //! @details     アイテムリストを取得します。
    //!
    //! @return      アイテムリストを返します。
    //---------------------------------------------------------------------------
    virtual UIElementList& GetItems() = 0;

    //---------------------------------------------------------------------------
    //! @brief       アイテムリストを取得します。
    //!
    //! @details     アイテムリストを取得します。
    //!
    //! @return      アイテムリストを返します。
    //---------------------------------------------------------------------------
    virtual const UIElementList& GetItems() const = 0;

    /*override*/ bool OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs);

private:
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief       指定アイテムを選択します。
    //!
    //! @details     指定アイテムを選択します。
    //!
    //! @param[in]   index  選択対象のインデックスを指定します。
    //---------------------------------------------------------------------------
    void SelectItem(s32 index);

    //---------------------------------------------------------------------------
    //! @brief       アイテムの選択を指定オフセットだけ移動します。
    //!
    //! @details     アイテムの選択を指定オフセットだけ移動します。
    //!
    //! @param[in]   offset  オフセット値を指定します。
    //---------------------------------------------------------------------------
    void MoveSelection(s32 offset);

    s32 FindSelectableItem(s32 index, bool isAscending);

    void CallbackSelectionChanged(ListBoxItemPlaceHolder& target);

private:
    s32 m_FocusedItemIndex;

    SelectionChangedEvent m_SelectionChangedCallbacks;
};

} // dw
} // internal
} // nw
