﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PRIMITIVE_RENDERER_GL_H_
#define NW_DEV_PRIMITIVE_RENDERER_GL_H_

#include <nw/dev/dev_PrimitiveRendererBase.h>
#include <nw/gfnd/gl/gfnd_TextureGL.h>
#include <gl/glew.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        GL を使用するプリミティブ描画クラスです。
//---------------------------------------------------------------------------
class PrimitiveRendererGL : public PrimitiveRendererBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator   アロケーターです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit PrimitiveRendererGL( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~PrimitiveRendererGL();


    //---------------------------------------------------------------------------
    //! @brief        描画のための準備を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //! @param[in]    binaryData  シェーダバイナリへのポインタです。
    //! @param[in]    binarySize  シェーダバイナリのサイズです。
    //---------------------------------------------------------------------------
    virtual void InitializeFromBinaryImpl( nw::ut::IAllocator* allocator, const void* binaryData, u32 binarySize );

    //---------------------------------------------------------------------------
    //! @brief        描画のための準備を行います。
    //!
    //! @param[in]    allocator   アロケーターです。
    //---------------------------------------------------------------------------
    virtual void InitializeImpl( nw::ut::IAllocator* allocator, const char* /*shader_path*/ )
    {
        InitializeFromBinaryImpl( allocator, NULL, 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief        頂点バッファをバインドします。
    //!
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    virtual void CreateVertexBufferImpl( DrawBuffer& buf )
    {
        CreateVertexBuffer(
            buf.buffer,
            buf.vertex,
            buf.vertexNum,
            buf.index,
            buf.indexNum
        );
    }


    //----------------------------------------
    //! @name 行列設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビュー行列を設定します。
    //!
    //! @param[in]    viewMtx     設定するビュー行列です。
    //---------------------------------------------------------------------------
    virtual void SetViewMtxImpl( const nw::math::MTX34* viewMtx ) { m_ViewMtx = viewMtx; }

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うプロジェクション行列を設定します。
    //!
    //! @param[in]    projectionMtx   設定するプロジェクション行列です。
    //---------------------------------------------------------------------------
    virtual void SetProjectionMtxImpl( const nw::math::MTX44* projectionMtx ) { m_ProjMtx = projectionMtx; }

    //---------------------------------------------------------------------------
    //! @brief        PrimitiveRenderer で使うビューポートを設定します。
    //!
    //! @param[in]    x           設定する x 座標です。
    //! @param[in]    y           設定する y 座標です。
    //! @param[in]    width       設定する幅です。
    //! @param[in]    height      設定する高さです。
    //---------------------------------------------------------------------------
    virtual void SetViewportImpl( const s32 x, const s32 y, const s32 width, const s32 height );

    //  @}


    //----------------------------------------
    //! @name 描画
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        描画に必要な設定を行います。
    //---------------------------------------------------------------------------
    virtual void BeginImpl();

    //---------------------------------------------------------------------------
    //! @brief        描画を終了します。
    //---------------------------------------------------------------------------
    virtual void EndImpl();

    //---------------------------------------------------------------------------
    //! @brief        平面の三角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawTriangleImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color
    );

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して三角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //---------------------------------------------------------------------------
    virtual void DrawTriangleImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color,
        const DrawBuffer& buf
    );

    //---------------------------------------------------------------------------
    //! @brief        平面の四角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawQuadImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight
    );

    //--------------------------------------------------------------------------
    //! @brief        テクスチャを張り込んだ平面の四角形を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    texture     貼り付けるテクスチャです。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //! @param[in]    uvSrc       左下の点の UV 値です。
    //! @param[in]    uvSize      UV 値の範囲です(右上の点の UV 値 - 左下の点の UV 値)。
    //--------------------------------------------------------------------------
    virtual void DrawQuadImpl(
        const nw::math::MTX34& modelMtx,
        const nw::gfnd::Texture& texture,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight,
        const nw::math::VEC2& uvSrc,
        const nw::math::VEC2& uvSize
    );

    //---------------------------------------------------------------------------
    //! @brief        四角形の枠線を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    colorLeft   左側の点の頂点色です。
    //! @param[in]    colorRight  右側の点の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawBoxImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& colorLeft,
        const nw::ut::Color4u8& colorRight
    );

    //---------------------------------------------------------------------------
    //! @brief        直方体を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawCubeImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    //--------------------------------------------------------------------------
    //! @brief        ワイヤーフレームの直方体を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      手前左上の頂点色です。
    //! @param[in]    color1      奥右下の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawWireCubeImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    //---------------------------------------------------------------------------
    //! @brief        線分を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color0      1 つ目の頂点色です。
    //! @param[in]    color1      2 つ目の頂点色です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    virtual void DrawLineImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        const f32 lineWidth = 1.f
    );

    //---------------------------------------------------------------------------
    //! @brief        バッファを指定して線分を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    color       頂点色です。
    //! @param[in]    buf         バッファの構造体です。
    //! @param[in]    lineWidth   線分の幅です。
    //---------------------------------------------------------------------------
    virtual void DrawLineImpl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& color,
        const DrawBuffer& buf,
        const f32 lineWidth = 1.f
    );

    //---------------------------------------------------------------------------
    //! @brief        4 x 8 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawSphere4x8Impl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        8 x 16 分割、直径 1 の球を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    north       北極(y が + 側)の頂点色です。
    //! @param[in]    south       南極(y が - 側)の頂点色です。
    //---------------------------------------------------------------------------
    virtual void DrawSphere8x16Impl(
        const nw::math::MTX34& modelMtx,
        const nw::ut::Color4u8& north,
        const nw::ut::Color4u8& south
    );

    //---------------------------------------------------------------------------
    //! @brief        16 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    edge        円の色です。
    //---------------------------------------------------------------------------
    virtual void DrawCircle16Impl( const nw::math::MTX34& modelMtx, const nw::ut::Color4u8& edge );

    //---------------------------------------------------------------------------
    //! @brief        32 分割、直径 1 の円を描画します。
    //!
    //! @param[in]    modelMtx    モデル行列です。
    //! @param[in]    edge        円の色です。
    //---------------------------------------------------------------------------
    virtual void DrawCircle32Impl( const nw::math::MTX34& modelMtx, const nw::ut::Color4u8& edge );

    //  @}

private:
    bool CreateShader( GLuint shader, const char* source, nw::ut::IAllocator* allocator );

    void CreateVertexBuffer(
        GLuint* vertexBuffer,
        const Vertex* vtx,
        s32 vtxNum,
        const u16* idx,
        s32 idxNum
    );

    void DrawVertexBuffer(
        const nw::math::MTX34& modelMtx,
        GLuint* vb,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    void DrawPolygon(
        const nw::math::MTX34& modelMtx,
        GLuint* vb,
        s32 idxNum,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1
    );

    void DrawLines(
        const nw::math::MTX34& modelMtx,
        GLenum lineType,
        GLuint* vb,
        s32 idxNum,
        const nw::ut::Color4u8& color0,
        const nw::ut::Color4u8& color1,
        const f32 lineWidth = 1.f
    );

    GLuint m_VertexShader;              //!< 頂点シェーダ
    GLuint m_FragmentShader;            //!< フラグメントシェーダ
    GLuint m_ShaderProgram;             //!< シェーダプログラム
    GLuint m_Sampler;                   //!< サンプラーオブジェクト
    const nw::math::MTX34* m_ViewMtx;   //!< ビュー行列
    const nw::math::MTX44* m_ProjMtx;   //!< プロジェクション行列

    //----------------------------------------
    //! @name ユニフォーム／頂点属性位置のキャッシュ
    //  @{

    GLuint m_ParamWVP;
    GLuint m_ParamUser;
    GLuint m_ParamRate;
    GLuint m_ParamTex;
    GLuint m_ParamColor0;
    GLuint m_ParamColor1;
    GLuint m_ParamUVSrc;
    GLuint m_ParamUVSize;
    GLuint m_AttrVertex;
    GLuint m_AttrTexCoord0;
    GLuint m_AttrColorRate;

    //  @}

    //----------------------------------------
    //! @name バッファ
    //  @{

    GLuint m_TriangleBuf[ 2 ];
    GLuint m_QuadBuf[ 2 ];
    GLuint m_BoxBuf[ 2 ];
    GLuint m_CubeBuf[ 2 ];
    GLuint m_WireCubeBuf[ 2 ];
    GLuint m_LineBuf[ 2 ];
    GLuint m_SphereSBuf[ 2 ];
    GLuint m_SphereLBuf[ 2 ];
    GLuint m_CircleSBuf[ 2 ];
    GLuint m_CircleLBuf[ 2 ];

    //  @}
};

} // namespace dev
} // namespace nw

#endif  // NW_DEV_PRIMITIVE_RENDERER_GL_H_

