﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PRIMITIVE_RENDERER_UTIL_H_
#define NW_DEV_PRIMITIVE_RENDERER_UTIL_H_

#include <nw/math.h>
#include <nw/ut/ut_Color.h>
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <GL/glew.h>
#endif

namespace nw
{
namespace dev
{

//! @brief 各 PrimitiveRenderer で使う共通の頂点情報です。
struct Vertex
{
    nw::math::VEC3 pos;     //!< 位置です。
    nw::math::VEC2 uv;      //!< UV 座標です。
    nw::math::VEC4 color;   //!< 頂点色(ただし、r 成分を係数として利用するのみ)です。
};

//! @brief 描画用バッファを表す構造体です。
struct DrawBuffer
{
    Vertex* vertex;         //!< 頂点列です。
    u32 vertexNum;          //!< 頂点列の数です。
    u16* index;             //!< 頂点の結び順を示すインデックス列です。
    u32 indexNum;           //!< インデックス列の数です。
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    GLuint buffer[2];
#endif
};

namespace internal
{
    //---------------------------------------------------------------------------
    //! @brief        バッファに四角形を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファ ( 4 個 ) です。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファ ( 6 個 ) です。
    //---------------------------------------------------------------------------
    void SetQuadVertex( Vertex* vtxBuf, u16* idxBuf );

    //---------------------------------------------------------------------------
    //! @brief        バッファに立方体を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファ ( 8 個 ) です。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファ ( 36 個 ) です。
    //---------------------------------------------------------------------------
    void SetCubeVertex( Vertex* vtxBuf, u16* idxBuf );

    //---------------------------------------------------------------------------
    //! @brief        バッファにワイヤーフレーム ( LINE_LOOP ) の立方体を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファ ( 8 個 ) です。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファ ( 17 個 ) です。
    //---------------------------------------------------------------------------
    void SetWireCubeVertex( Vertex* vtxBuf, u16* idxBuf );

    //---------------------------------------------------------------------------
    //! @brief        バッファに直線を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファ ( 2 個 ) です。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファ ( 2 個 ) です。
    //---------------------------------------------------------------------------
    void SetLineVertex( Vertex* vtxBuf, u16* idxBuf );

    //---------------------------------------------------------------------------
    //! @brief        バッファに球体を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファです。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファです。
    //! @param[in]    sizeX    X 方向の分割数です。
    //! @param[in]    sizeY    Y 方向の分割数です。
    //---------------------------------------------------------------------------
    void SetSphereVertex( Vertex* vtxBuf, u16* idxBuf, s32 sizeX, s32 sizeY );

    //---------------------------------------------------------------------------
    //! @brief        バッファに円盤を現す頂点・インデックス情報を詰めます。
    //!
    //! @param[in]    vtxBuf   頂点情報を詰め込むバッファです。
    //! @param[in]    idxBuf   インデックス情報を詰め込むバッファです。
    //! @param[in]    divNum   円周の分割数です。
    //---------------------------------------------------------------------------
    void SetDiskVertex( Vertex* vtxBuf, u16* idxBuf, s32 divNum );

    //---------------------------------------------------------------------------
    //! @brief        球全体の頂点数を計算します。
    //!
    //! @param[in]    x        円周の分割数です。
    //! @param[in]    y        縦方向の(極以外の)頂点数です。
    //!
    //! @return       球全体の頂点数を返します。
    //---------------------------------------------------------------------------
    inline s32 CalcSphereVertexNum( s32 x, s32 y ) { return ( x * y ) + 2; }

    //---------------------------------------------------------------------------
    //! @brief        球全体のインデックス数を計算します。
    //!
    //! @param[in]    x        円周の分割数です。
    //! @param[in]    y        縦方向の(極以外の)頂点数です。
    //!
    //! @return       球全体のインデックス数を返します。
    //---------------------------------------------------------------------------
    inline s32 CalcSphereIndexNum( s32 x, s32 y ) { return 3 * ( x + x * 2 * ( y - 1 ) + x ); }

    //---------------------------------------------------------------------------
    //! @brief        円盤全体の頂点数を計算します。
    //!
    //! @param[in]    divNum   円周の分割数です。
    //!
    //! @return       円盤全体の頂点数を返します。
    //---------------------------------------------------------------------------
    inline s32 CalcDiskVertexNum( s32 divNum ){ return divNum + 1; }

    //---------------------------------------------------------------------------
    //! @brief        円盤全体のインデックス数を計算します。
    //!
    //! @param[in]    divNum   円周の分割数です。
    //!
    //! @return       円盤全体のインデックス数を返します。
    //---------------------------------------------------------------------------
    inline s32 CalcDiskIndexNum( s32 divNum ){ return divNum * 3; }

} // namespace internal
} // namespace dev
} // namespace nw

#endif  // NW_DEV_PRIMITIVE_RENDERER_UTIL_H_

