﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_LOGGER_BASE_H_
#define NW_DEV_LOGGER_BASE_H_

#include <nw/types.h>
#include <nw/ut/os/ut_Mutex.h>
#include <nw/ut/ut_String.h>
#include <nw/font/font_TextWriter.h>
#include <nw/font/font_RectDrawer.h>

#if !defined(NW_LOGGER_ENABLE) && !defined(NW4F_RELEASE)
    #define NW_LOGGER_ENABLE 1
#endif

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        デバッグ出力を行うクラスです。
//!
//! @details :category     デバッグ
//!
//! @tparam       CharType 文字の型。
//---------------------------------------------------------------------------
template <typename CharType>
class LoggerBase
{
public:
    //! デバッグ表示の出力先を表します。
    typedef enum
    {
        OUTPUT_NONE         =   0x00,   //!< 出力を行いません。
        OUTPUT_DISPLAY      =   0x01,   //!< TV 画面へ出力します。
        OUTPUT_TERMINAL     =   0x02,   //!< ターミナルへ出力します。
        OUTPUT_ALL          =   ( OUTPUT_DISPLAY | OUTPUT_TERMINAL )    //!< TV 画面、ターミナル両方へ出力します。
    }
    OutputType;

    //! ロガーの属性を表します。
    enum
    {
        ATTR_NOT_TURNOVER_LINE      =   0x01,           //!<   溢れた行の折り返し表示を行いません。
        ATTR_STOP_ON_BUFFER_FULL    =   0x02,           //!<   バッファが一杯になった場合に更新をストップします。
        ATTR_TAB_SIZE_2             =   (0x00 << 2),    //!<   タブ幅 2 です。
        ATTR_TAB_SIZE_4             =   (0x01 << 2),    //!<   タブ幅 4 です。
        ATTR_TAB_SIZE_8             =   (0x02 << 2),    //!<   タブ幅 8 です。
        ATTR_TAB_SIZE_MASK          =   (0x03 << 2),    //!<   タブサイズのマスクです。

        ATTR_DEFAULT                =   ATTR_TAB_SIZE_4     //!<   デフォルト設定です。
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //  @{

    //! @brief コンストラクタです。
    LoggerBase()
        : m_LoggerBuffer(NULL),
          m_Next(NULL)
        {}

    //! @brief デストラクタです。
    ~LoggerBase() {}

    //  @}

    //----------------------------------------
    //! @name 作成／破棄
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ロガーを作成します。
    //!
    //! @param[in]    buffer      ロガーで使用するバッファへのポインタです。
    //! @param[in]    width       １行あたりの文字数です。
    //! @param[in]    height      バッファに記録しておく文字のライン数です。
    //! @param[in]    viewHeight  画面上に一度に表示する文字のライン数です。
    //! @param[in]    priority    表示優先度です。
    //! @param[in]    attr        ロガーの属性値です。
    //---------------------------------------------------------------------------
    void Create(
        void*   buffer,
        u16     width,
        u16     height,
        u16     viewHeight,
        u16     priority,
        u16     attr = ATTR_DEFAULT
        );

    //---------------------------------------------------------------------------
    //! @brief        ロガーを破棄します。
    //!
    //! @return       破棄されたロガーで使用していたバッファへのポインタです。
    //---------------------------------------------------------------------------
    void* Destroy();

    //  @}

    //----------------------------------------
    //! @name 描画設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ロガー作成に必要なバッファサイズを取得します。
    //!
    //! @param[in]    width   1 行あたりの文字数です。
    //! @param[in]    height  行数です。
    //!
    //! @return       ロガー作成に必要なバッファサイズです。
    //---------------------------------------------------------------------------
    static u32 GetBufferSize( u32 width, u32 height )
    {
        return sizeof(LoggerBuffer) + ( (width) + 1 ) * height * sizeof( CharType );
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファサイズから確保できる文字行数を取得します。
    //!
    //! @param[in]    width       1 行あたりの文字数です。
    //! @param[in]    bufferSize  バッファサイズです。
    //!
    //! @return       ロガーに保存可能な文字行数です。
    //---------------------------------------------------------------------------
    static u32 GetLineSize( u32 width, u32 bufferSize )
    {
        return ( bufferSize - sizeof(LoggerBuffer) ) / ( ( width + 1 ) * sizeof( CharType ) );
    }

    //---------------------------------------------------------------------------
    //! @brief        ロガーの表示優先度を変更します。
    //!
    //! @param[in]    priority      優先度です。
    //!
    //! @return       変更前の表示優先度です。
    //---------------------------------------------------------------------------
    u16 ChangePriority( u16 priority );

    //---------------------------------------------------------------------------
    //! @brief        ロガーに設定されているバッファのライン数を取得します。
    //!
    //! @return       ロガーに設定されているバッファのライン数です。
    //---------------------------------------------------------------------------
    u16 GetBufferHeight()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );

        return m_LoggerBuffer->height;
    }

    //---------------------------------------------------------------------------
    //! @brief        モニタに一度に表示する行数設定を取得します。
    //!
    //! @return       モニタに一度に表示する行数設定です。
    //---------------------------------------------------------------------------
    u16 GetViewHeight()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );

        return m_LoggerBuffer->viewLines;
    }

    //---------------------------------------------------------------------------
    //! @brief        画面に表示する基準行番号を取得します。
    //!
    //! @return       画面へ表示する基準行番号(初期化、クリア時の行番号を 0 とする累積行番号)です。
    //---------------------------------------------------------------------------
    s32 GetViewBaseLine()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );

        return m_LoggerBuffer->viewTopLine;
    }

    //---------------------------------------------------------------------------
    //! @brief        画面に表示する基準行番号を指定します。
    //!
    //! @param[in]    line   表示する行番号(初期化、クリア時の行番号を 0 とする累積行番号)です。
    //!
    //! @return       設定前の行番号です。
    //---------------------------------------------------------------------------
    s32 SetViewBaseLine( s32 line )
    {
        s32 before;

        NW_ASSERT_NOT_NULL( m_LoggerBuffer );

        before = m_LoggerBuffer->viewTopLine;
        m_LoggerBuffer->viewTopLine = line;
        return before;
    }

    //---------------------------------------------------------------------------
    //! @brief        ロガーの表示/非表示設定を変更します。
    //!
    //! @param[in]    isVisible   表示フラグです。
    //!
    //! @return       変更前の表示/非表示設定です。
    //---------------------------------------------------------------------------
    bool SetVisible( bool isVisible )
    {
        bool before;

        NW_ASSERT_NOT_NULL( m_LoggerBuffer );

        before = m_LoggerBuffer->isVisible;
        m_LoggerBuffer->isVisible = isVisible;

        return before;
    }

    //---------------------------------------------------------------------------
    //! @brief        ロガーの表示/非表示設定を取得します。
    //!
    //! @return       現在の表示/非表示設定です。
    //---------------------------------------------------------------------------
    bool IsVisible()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        return m_LoggerBuffer->isVisible;
    }

    //---------------------------------------------------------------------------
    //! @brief        モニタに表示する際の左上隅の座標を設定します。
    //!
    //! @param[in]    x   左上 X 座標です。
    //! @param[in]    y   左上 Y 座標です。
    //---------------------------------------------------------------------------
    void SetPosition( s32 x, s32 y )
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        m_LoggerBuffer->viewPosX = static_cast<s16>( x );
        m_LoggerBuffer->viewPosY = static_cast<s16>( y );
    }

    //---------------------------------------------------------------------------
    //! @brief        モニタに表示する際の左上隅の X 座標を取得します。
    //!
    //! @return       左上隅の X 座標です。
    //---------------------------------------------------------------------------
    s16 GetPositionX()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        return static_cast<s16>( m_LoggerBuffer->viewPosX );
    }

    //---------------------------------------------------------------------------
    //! @brief        モニタに表示する際の左上隅の Y 座標を取得します。
    //!
    //! @return       左上隅の Y 座標です。
    //---------------------------------------------------------------------------
    s16 GetPositionY()
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        return static_cast<s16>( m_LoggerBuffer->viewPosY );
    }

    //  @}

    //----------------------------------------
    //! @name 描画
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        描画用バッファ、ターミナル両方へ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //---------------------------------------------------------------------------
    void Printf( const CharType* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        描画用バッファへ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //---------------------------------------------------------------------------
    void PrintfD( const CharType* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        ターミナルへ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //---------------------------------------------------------------------------
    void PrintfT( const CharType* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        出力先を指定して、文字列を出力します。
    //!
    //! @param[in]    type        文字列の出力先を指定します。
    //! @param[in]    format      出力文字列フォーマットです。
    //---------------------------------------------------------------------------
    void FPrintf( OutputType type, const CharType* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        描画用バッファ、ターミナル両方へ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //! @param[in]    vlist       可変引数リストです。
    //---------------------------------------------------------------------------
    void VPrintf( const CharType* format, std::va_list vlist )
    {
        VFPrintf( OUTPUT_ALL, format, vlist );
    }

    //---------------------------------------------------------------------------
    //! @brief        描画用バッファへ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //! @param[in]    vlist       可変引数リストです。
    //---------------------------------------------------------------------------
    void VPrintfD( const CharType* format, std::va_list vlist )
    {
        VFPrintf( OUTPUT_DISPLAY, format, vlist );
    }

    //---------------------------------------------------------------------------
    //! @brief        ターミナルへ文字列を出力します。
    //!
    //! @param[in]    format      出力文字列フォーマットです。
    //! @param[in]    vlist       可変引数リストです。
    //---------------------------------------------------------------------------
    void VPrintfT( const CharType* format, std::va_list vlist )
    {
        VFPrintf( OUTPUT_TERMINAL, format, vlist );
    }

#if defined(NW_LOGGER_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        出力先を指定して、文字列を出力します。
    //!
    //! @param[in]    type        文字列の出力先を指定します。
    //! @param[in]    format      出力文字列フォーマットです。
    //! @param[in]    vlist       可変引数リストです。
    //---------------------------------------------------------------------------
    void VFPrintf( OutputType type, const CharType* format, std::va_list vlist );

    //---------------------------------------------------------------------------
    //! @brief        ロガーを TextWriter、RectDrawer を使用して描画します。
    //!
    //! @param[in]    writer   ロガーを描画する際に使用する TextWriter を指定します。
    //! @param[in]    drawer   ロガーを描画する際に使用する RectDrawer を指定します。
    //---------------------------------------------------------------------------
    void Draw( nw::font::TextWriterBase<CharType>& writer, nw::font::RectDrawer& drawer );

    //---------------------------------------------------------------------------
    //! @brief        ロガーを DirectPrint を使用して描画します。
    //---------------------------------------------------------------------------
    void DrawDirect();

    //---------------------------------------------------------------------------
    //! @brief        リストのロガーを全て TextWriter、RectDrawer を使用して描画します。
    //!
    //! @param[in]    writer   ロガーを描画する際に使用する TextWriter を指定します。
    //! @param[in]    drawer   ロガーを描画する際に使用する RectDrawer を指定します。
    //---------------------------------------------------------------------------
    static void DrawAll( nw::font::TextWriterBase<CharType>& writer, nw::font::RectDrawer& drawer );

    //---------------------------------------------------------------------------
    //! @brief        リストのロガーを全て DirectPrint を使用して描画します。
    //---------------------------------------------------------------------------
    static void DrawDirectAll();
#else
    void VFPrintf( OutputType type, const CharType* format, std::va_list vlist )
    {
        NW_UNUSED_VARIABLE( type );
        NW_UNUSED_VARIABLE( format );
        NW_UNUSED_VARIABLE( vlist );
    }

    void Draw( nw::font::TextWriterBase<CharType>& writer )
    {
        NW_UNUSED_VARIABLE( writer );
    }

    void DrawDirect() {}

    void DrawAll() {}

    void DrawDirectAll() {}
#endif

    //---------------------------------------------------------------------------
    //! @brief        保存されている文字バッファを TextWriter へ送ります。
    //! @param[in]    writer 文字バッファを送る先の TextWriter です。
    //---------------------------------------------------------------------------
    void PrintToWriter( nw::font::TextWriterBase<CharType>& writer );

    //---------------------------------------------------------------------------
    //! @brief        保存されている文字バッファをクリアします。
    //---------------------------------------------------------------------------
    void Clear();

    //---------------------------------------------------------------------------
    //! @brief        ロガーに出力された履歴のうちバッファに残っている最も古い
    //!               行の行番号を取得します。
    //!
    //! @return       バッファに残っている最も古い出力の行番号です。
    //---------------------------------------------------------------------------
    s32 GetBufferHeadLine() const
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        return m_LoggerBuffer->ringTopLineCnt;
    }

    //---------------------------------------------------------------------------
    //! @brief        ロガーに出力された累計行数を取得します。
    //!
    //! @return       初期化又はクリア時の行番号を 0 とする累積出力行数です。
    //---------------------------------------------------------------------------
    s32 GetTotalLines();

    typedef void (*LoggerVisitor)( const CharType* str, s32 lineNum, u32 userParam );

    //---------------------------------------------------------------------------
    //! @brief        ロガーに保存された文字列に対してユーザ処理を行います。
    //!
    //! @param[in]    visitor    各行に対して呼ばせる関数です。
    //! @param[in]    userParam  visitor 関数に渡すユーザ指定のパラメータです。
    //---------------------------------------------------------------------------
    void VisitString( LoggerVisitor visitor, u32 userParam );

    //---------------------------------------------------------------------------
    //! @brief        画面に最新行が表示されるように基準行番号を設定します。
    //!
    //! @return       表示先頭行の行番号を返します。
    //---------------------------------------------------------------------------
    s32 ShowLatestLine()
    {
        s32 baseLine = GetTotalLines() - GetViewHeight();

        if ( baseLine < 0 )
        {
            baseLine = 0;
        }
        (void)SetViewBaseLine( baseLine );
        return baseLine;
    }

    //  @}

private:
    //! @brief 優先度を元にロガーリストへの挿入位置を検索します。
    static LoggerBase* SearchLoggerFromListByPriority( u16 priority );

    //! @brief ロガーリストへ新規ロガーハンドルを追加します。
    void AppendLoggerToList();

    //! @brief ロガーリストからロガーを削除します。
    void RemoveLoggerFromList();

#if defined(NW_LOGGER_ENABLE)
    //! @brief 文字列を画面表示用バッファ、ターミナルへ出力します。
    void PrintString( OutputType type, const CharType* str );

    //! @brief 文字列を画面表示用バッファへ出力します。
    void PrintToBuffer( const CharType* str );
#endif

    //! @brief 行番号と水平位置から文字列へのポインタを取得します。
    CharType* GetTextPtr( u16 line, u16 xPos )
    {
        return &m_LoggerBuffer->textBuf[ (m_LoggerBuffer->width + 1) * line + xPos ];
    }

    //! @brief マルチバイト文字の１文字コードが何バイトのデータで表現されているかを取得します。
    u32 CodeWidth( const char* p )
    {
        static const u32 UTF8Bytes[] =
        {
            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
            2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 6, 6
        };

        return UTF8Bytes[static_cast<unsigned char>(*p) >> 1];
    }

    u32 CodeWidth( const char16* p )
    {
        NW_UNUSED_VARIABLE( p );
        return 1;
    }

    //! @brief ロガーに設定されたタブサイズを取得します。
    u32 GetTabSize() const
    {
        s32 tab = (m_LoggerBuffer->attr & ATTR_TAB_SIZE_MASK) >> 2;
        return static_cast<u32>(0x2 << tab);
    }

    //! @brief 文字列を行末まで読み飛ばして行末のポインタを取得します。
    CharType* SearchEndOfLine( const CharType* str )
    {
        while ( *str != '\n' && *str != '\0' )
        {
            ++str;
        }
        return const_cast<CharType*>( str );
    }

    //! @brief 現在リングバッファで使用中の行数を取得します。
    u16 GetRingUsedLines() const
    {
        NW_ASSERT_NOT_NULL( m_LoggerBuffer );
        {
            s32 lines = m_LoggerBuffer->printTop - m_LoggerBuffer->ringTop;

            if ( lines < 0 )
            {
                lines += m_LoggerBuffer->height;
            }
            return static_cast<u16>(lines);
        }
    }

    //! @brief 現在表示するべきアクティブなデータが格納されているバッファ行数を取得します。
    u16 GetActiveLines() const
    {
        u16 lines = GetRingUsedLines();
        if ( m_LoggerBuffer->printXPos > 0 )
        {
            lines++;
        }
        return lines;
    }

    //! @brief 文字列の終端記号を付加します。
    void TerminateLine()
    {
        *( GetTextPtr( m_LoggerBuffer->printTop, m_LoggerBuffer->printXPos ) ) = '\0';
    }

    //! @brief 改行します。
    //!
    //! @return 次の行の先頭へのポインタを返します。
    CharType* NextLine();

    //! @brief タブ文字を必要サイズ分のスペースに置き換えます。
    //!
    //! @return タブ出力後のバッファ位置へのポインタを返します。
    CharType* PutTab( CharType* dstPtr );

    //! @brief 文字を１文字分出力します。
    //!
    //! @return １文字のバイト数を返します。
    u32 PutChar( const CharType* str, CharType* dstPtr );

    //! @brief 一行を TextWriter へ送ります。
    void DoDrawString( u32 printLine, const CharType* str, nw::font::TextWriterBase<CharType>* writer );

    //! @brief 書き込まれた文字列を TextWriter へ送ります。
    void DoDraw( nw::font::TextWriterBase<CharType>* writer );

    //! @brief 書式指定して文字列をバッファに書き込みます。
    static int VSNPrintf(
        char*           buffer,
        std::size_t     count,
        const char*     format,
        std::va_list    arg
        )
    {
        return nw::ut::vsnprintf(buffer, count, count - 1, format, arg);
    }

    static int VSNPrintf(
        char16*         buffer,
        std::size_t     count,
        const char16*   format,
        std::va_list    arg
        )
    {
        return nw::ut::vsnw16printf(buffer, count, count - 1, format, arg);
    }

    //! @brief 文字列をコピーします。
    static size_t StrCpy( char* dst, std::size_t dstCount, const char* src )
    {
        return nw::ut::strcpy( dst, dstCount, src );
    }

    static size_t StrCpy( char16* dst, std::size_t dstCount, const char16* src )
    {
        return nw::ut::wcs16cpy( dst, dstCount, src );
    }


    //! 画面描画用バッファと設定です。
    struct LoggerBuffer
    {
        CharType*             textBuf;            //!<  テキストバッファです。
        u16                   width;              //!<  １行あたりの文字幅です。
        u16                   height;             //!<  出力可能な最大文字行数です。
        u16                   priority;           //!<  表示優先度です。
        u16                   attr;               //!<  表示属性です。
        u16                   printTop;           //!<  出力先頭行の行番号です。
        u16                   printXPos;          //!<  出力位置です。

        u16                   ringTop;            //!<  バッファリングの先頭行番号です。
        s32                   ringTopLineCnt;     //!<  バッファリング先頭行の累積行番号です。
        s32                   viewTopLine;        //!<  画面表示行の行番号です。

        s16                   viewPosX;           //!<  画面表示位置の X 座標です。
        s16                   viewPosY;           //!<  画面表示位置の Y 座標です。
        u16                   viewLines;          //!<  画面表示行数です。

        bool                  isVisible;          //!<  表示/非表示を表すフラグです。
    };

    LoggerBuffer* m_LoggerBuffer;
    LoggerBase* m_Next;

    static LoggerBase* s_LoggerList;

    static nw::ut::Mutex s_Mutex;
    static bool s_Initialized;
};

} // namespace dev
} // namespace nw

#endif  // NW_DEV_LOGGER_BASE_H_
