﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_LOAD_METER_H_
#define NW_DEV_LOAD_METER_H_

#include <nw/ut/os/ut_Time.h>
#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_MoveArray.h>
#include <nw/ut/ut_SafeString.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <gl/glew.h>
#endif

namespace nw
{
namespace dev
{

class MeterDrawer;

//---------------------------------------------------------------------------
//! @brief        負荷メーターの基底クラスです。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
class LoadMeterBase
{
private:
    friend class MeterDrawer;

public:
    //! @brief １つの区間を表します。
    struct Section
    {
        nw::ut::Tick     begin;     //!< 区間の開始時刻です。
        nw::ut::Tick     end;       //!< 区間の終了時刻です。
        nw::ut::Color4u8 color;     //!< 区間の色です。
        s32              parent;    //!< 親区間（-1ならルート）です。
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    buffer     測定結果を保存するバッファです。 sizeof( Section ) * sectionNum * 2 以上のサイズが必要です。
    //! @param[in]    sectionNum 1 フレームあたり、測定結果をいくつ保存できるかです。
    //! @param[in]    name       名前です。
    //! @param[in]    color      デフォルトの色です。
    //---------------------------------------------------------------------------
    /* ctor */ LoadMeterBase( Section* buffer, s32 sectionNum, const char* name = NULL, nw::ut::Color4u8 color = nw::ut::Color4u8(nw::ut::Color4u8::GREEN) );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */ ~LoadMeterBase() {}


    //---------------------------------------------------------------------------
    //! @brief        計測開始時間を設定します。
    //!
    //! @param[in]    tick      開始時間です。
    //---------------------------------------------------------------------------
    void SetBeginMeasure( nw::ut::Tick tick );

    //---------------------------------------------------------------------------
    //! @brief        計測開始時間を設定します。
    //!
    //! @param[in]    tick      開始時間です。
    //! @param[in]    color     メーターの色を変更します。
    //---------------------------------------------------------------------------
    void SetBeginMeasure( nw::ut::Tick tick, const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        計測終了時間を設定します。
    //!
    //! @param[in]    tick      終了時間です。
    //---------------------------------------------------------------------------
    void SetEndMeasure( nw::ut::Tick tick );


    //---------------------------------------------------------------------------
    //! @brief        前のフレームの最初の開始時刻を取得します。
    //!
    //! @return       前のフレームの最初の開始時刻を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Tick& GetLastFirstBegin() const { return m_SectionArray[ 1 - m_CurrentBuffer ].at( 0 ).begin; }

    //---------------------------------------------------------------------------
    //! @brief        前のフレームの最後の終了時刻を取得します。
    //!
    //! @return       前のフレームの最後の終了時刻を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Tick& GetLastFinalEnd() const { return m_FinalEnd[ 1 - m_CurrentBuffer ]; }

    //---------------------------------------------------------------------------
    //! @brief        前のフレームの消費時間を取得します。
    //!
    //! @return       前のフレームの消費時間を返します。
    //---------------------------------------------------------------------------
    nw::ut::Tick GetLastTotalSpan() const;


    //---------------------------------------------------------------------------
    //! @brief        表示名を設定します。
    //!
    //! @param[in]    name      表示名です。
    //---------------------------------------------------------------------------
    void SetName( const char* name ) { m_Name = nw::ut::SafeString(name); }

    //---------------------------------------------------------------------------
    //! @brief        表示名を取得します。
    //!
    //! @return       表示名です。
    //---------------------------------------------------------------------------
    const char* GetName() const { return m_Name.c_str(); }

    //---------------------------------------------------------------------------
    //! @brief        表示色を設定します。
    //!
    //! @param[in]    color     表示色です。
    //---------------------------------------------------------------------------
    void SetColor( const nw::ut::Color4u8& color ) { m_Color = color; }

    //---------------------------------------------------------------------------
    //! @brief        表示色を取得します。
    //!
    //! @return       表示色です。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8& GetColor() const { return m_Color; }


protected:
    //! @brief フレーム終了時に MeterDrawer から呼ばれます。
    virtual void OnEndFrame();
    //! @brief セクションを追加します。
    //! @param[in] tick セクションの開始時刻です。
    //! @param[in] color MeterDrawer で描画する際のバーの色です。
    void AddSection( nw::ut::Tick tick, const nw::ut::Color4u8& color );
    //! @brief 現在のセクションを終了します。
    //! @param[in] tick セクションの終了時刻です。
    void EndSection( nw::ut::Tick tick );

    //! @brief 前のフレームの計測結果の個数を取得します。
    //! @return 前のフレームの計測結果の個数を返します。
    s32 GetLastSectionNum() const { return m_SectionNum[ 1 - m_CurrentBuffer ]; }

    //! @brief 前のフレームの計測結果を取得します。
    //! @param[in] idx インデックスです。
    //! @return 前のフレームの計測結果を返します。
    const Section& GetLastResult( s32 idx ) const { return m_SectionArray[ 1 - m_CurrentBuffer ].at( idx ); }


    nw::ut::FixedSafeString<128> m_Name;    //!< メーターの名前です。
    nw::ut::Color4u8 m_Color;               //!< デフォルトの色です。

    nw::ut::MoveArray<Section> m_SectionArray[2];   //!< セクションのリスト（ダブルバッファ）です。
    s32                        m_SectionNum[2];     //!< セクションリストに含まれる個数です。
    nw::ut::Tick               m_FinalEnd[2];       //!< 最後の終了時刻です。
    s32                        m_CurrentBuffer;     //!< 現在のバッファです。
    s32                        m_TopSection;        //!< 現在計測中のセクションです。
    s32                        m_OverNum;           //!< 保持できるセクション数をオーバーしている個数です。

public:
    nw::ut::LinkListNode       m_linkNode;    //!< @details :private
};



//---------------------------------------------------------------------------
//! @brief        複数区間計測可能な負荷メータークラスです。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
template<s32 N>
class MultiLoadMeter : public LoadMeterBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    name       名前です。
    //! @param[in]    color      デフォルトの色です。
    //---------------------------------------------------------------------------
    /* ctor */ MultiLoadMeter( const char* name = NULL, nw::ut::Color4u8 color = nw::ut::Color4u8::GREEN )
        : LoadMeterBase( m_Buffer, N, name, color ) {}

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */ ~MultiLoadMeter() {}

private:
    Section m_Buffer[ N * 2 ];  //!< 計測結果を格納するバッファです。
};

//---------------------------------------------------------------------------
//! @brief        単区間計測を行う負荷メータークラスです。
//---------------------------------------------------------------------------
typedef MultiLoadMeter<1> LoadMeter;



//---------------------------------------------------------------------------
//! @brief        複数区間計測可能な CPU 負荷メータークラスです。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
template<s32 N>
class MultiCPUMeter : public MultiLoadMeter<N>
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    name       名前です。
    //! @param[in]    color      デフォルトの色です。
    //---------------------------------------------------------------------------
    /* ctor */ MultiCPUMeter( const char* name = NULL, nw::ut::Color4u8 color = nw::ut::Color4u8(nw::ut::Color4u8::GREEN) )
      : MultiLoadMeter<N>( name, color ) {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */ ~MultiCPUMeter() {}


    //---------------------------------------------------------------------------
    //! @brief        負荷計測を開始します。
    //---------------------------------------------------------------------------
    void BeginMeasure()
    {
        LoadMeterBase::SetBeginMeasure( nw::ut::Tick::GetSystemCurrent() );
    }

    //---------------------------------------------------------------------------
    //! @brief        負荷計測を開始します。
    //!
    //! @param[in]    color     メーターの色を変更します。
    //---------------------------------------------------------------------------
    void BeginMeasure( const nw::ut::Color4u8& color )
    {
        LoadMeterBase::SetBeginMeasure( nw::ut::Tick::GetSystemCurrent(), color );
    }

    //---------------------------------------------------------------------------
    //! @brief        負荷計測を終了します。
    //---------------------------------------------------------------------------
    void EndMeasure()
    {
        LoadMeterBase::SetEndMeasure( nw::ut::Tick::GetSystemCurrent() );
    }
};

//---------------------------------------------------------------------------
//! @brief        単区間計測を行う CPU 負荷メータークラスです。
//---------------------------------------------------------------------------
typedef MultiCPUMeter<1> CPUMeter;



//---------------------------------------------------------------------------
//! @brief        GPU 負荷メータークラスです。
//!
//!               OnEndFrame() が呼び出されたタイミングで、
//!               BeginMeasure() ～ EndMeasure() で積まれた
//!               コマンドの処理時間の計算を行います。
//!
//!               Cafe 版では GPU のサイクル値を取得する GX2 の関数を使用して
//!               負荷を計測します。
//!               コマンド処理によっては値や取得タイミングが変化する場合があります。
//!               詳細は GX2SampleTopGPUCycle, GX2SampleBottomGPUCycle をご参照ください。
//!
//! @details :category     デバッグ
//---------------------------------------------------------------------------
template<s32 N>
class MultiGPUMeter : public MultiLoadMeter<N>
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    name       名前です。
    //! @param[in]    color      デフォルトの色です。
    //---------------------------------------------------------------------------
    /* ctor */ MultiGPUMeter( const char* name = NULL, nw::ut::Color4u8 color = nw::ut::Color4u8(nw::ut::Color4u8::GREEN) )
      : MultiLoadMeter<N>( name, color )
#if defined(NW_PLATFORM_CAFE)
        ,m_GpuCounters( NULL )
#endif
        {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */ ~MultiGPUMeter() {}

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //! @param[in]    gpuCountersBuf 計測用カウンタの配列のポインタです。
    //---------------------------------------------------------------------------
    void Initialize( u64* gpuCountersBuf );
#else
    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();
#endif

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief        計測用カウンタのバッファに必要なサイズを取得します。
    //!
    //! @return       バッファサイズを返します。
    //---------------------------------------------------------------------------
    u32 GetBufferSize();
#endif

    //---------------------------------------------------------------------------
    //! @brief        負荷計測を開始します。
    //!
    //!               BeginMeasure が呼ばれたタイミングの計測を行い、
    //!               これ以降に積まれたコマンドが負荷計算の対象になります。
    //!               計測したいコマンドは BeginMeasure 以降に配置してください。
    //---------------------------------------------------------------------------
    void BeginMeasure()
    {
        BeginMeasure( LoadMeterBase::GetColor() );
    }

    //---------------------------------------------------------------------------
    //! @brief        負荷計測を開始します。
    //!
    //!               BeginMeasure が呼ばれたタイミングの計測を行い、
    //!               これ以降に積まれたコマンドが負荷計算の対象になります。
    //!               計測したいコマンドは BeginMeasure 以降に配置してください。
    //!
    //! @param[in]    color     メーターの色を変更します。
    //---------------------------------------------------------------------------
    void BeginMeasure( const nw::ut::Color4u8& color );

    //---------------------------------------------------------------------------
    //! @brief        負荷計測を終了します。
    //!
    //!               EndMeasure が呼ばれたタイミングの計測を行い、
    //!               これまでに積まれたコマンドが負荷計算の対象になります。
    //!               計測したいコマンドは EndMeasure より前に配置してください。
    //---------------------------------------------------------------------------
    void EndMeasure();


private:
    //! @brief フレーム終了時に MeterDrawer から呼ばれます。BeginMeasure, EndMeasure での計測値を元に負荷の計算を行います。
    virtual void OnEndFrame();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    GLuint m_GpuQueries[N * 2];     //!< 計測用オブジェクトの ID を保持します。
    GLint64 frameBeginTimeGL[N];
#else
    u64* m_GpuCounters;             //!< 計測用カウンタの配列のポインタです。
#endif

    nw::ut::IAllocator* m_Allocator;
};

//---------------------------------------------------------------------------
//! @brief        単区間計測を行う GPU 負荷メータークラスです。
//---------------------------------------------------------------------------
typedef MultiGPUMeter<1> GPUMeter;


} // namespace dev
} // namespace nw

#include <nw/dev/dev_LoadMeter.hpp>

#endif // NW_DEV_LOAD_METER_H_
