﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_DEV_TEXT_WRITER_H_
#define NW_DEV_DEV_TEXT_WRITER_H_

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_MoveArray.h>
#include <nw/font/font_ResFont.h>
#include <nw/font/font_RectDrawer.h>
#include <nw/font/font_DispStringBuffer.h>
#include <nw/font/font_CharWriter.h>
#include <nw/font/font_TextWriter.h>
#include <nw/font/font_TagProcessorBase.h>

namespace nw {
namespace dev {


//---------------------------------------------------------------------------
//! @brief        TextWriter を容易に使えるようにするクラスです。
//---------------------------------------------------------------------------
class DevTextWriter
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DevTextWriter()
      : m_CharMax( 256 ),
        m_OriginX( 0.f ),
        m_OriginY( 0.f ),
        m_TagProcessor( m_OriginX, m_OriginY ),
        m_DrawerBuffer( NULL ),
        m_ProjectionMtx( math::MTX44::Identity() ),
        m_ViewMtx( math::MTX34::Identity() ),
        m_ShadowColor( DEFAULT_SHADOW_COLOR ),
        m_Allocator( NULL )
    {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    charMax   バッファとして確保する文字数の最大数です。
    //---------------------------------------------------------------------------
    /* ctor */ explicit DevTextWriter( u32 charMax )
      : m_CharMax( charMax ),
        m_OriginX( 0.f ),
        m_OriginY( 0.f ),
        m_TagProcessor( m_OriginX, m_OriginY ),
        m_DrawerBuffer( NULL ),
        m_ProjectionMtx( math::MTX44::Identity() ),
        m_ViewMtx( math::MTX34::Identity() ),
        m_ShadowColor( DEFAULT_SHADOW_COLOR ),
        m_Allocator( NULL )
    {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~DevTextWriter() {}


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //! @param[in]    fontData       フォントデータへのポインタです。
    //! @param[in]    fontDataSize   フォントデータのサイズです。
    //! @param[in]    shaderData     シェーダーバイナリへのポインタです。
    //! @param[in]    shaderDataSize シェーダーバイナリのサイズです。
    //! @param[in]    allocator      バッファを確保するためのアロケーターです。
    //---------------------------------------------------------------------------
    void Initialize( void* fontData, u32 fontDataSize, void* shaderData, u32 shaderDataSize, ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        現在保持しているバッファをクリアします。
    //---------------------------------------------------------------------------
    void Clear()
    {
        DestroyStringBuffer();
        ResetCursorToOrigin();
    }


    //---------------------------------------------------------------------------
    //! @brief        文字列を描画します。
    //!
    //! @param[in]    str       文字列です。
    //---------------------------------------------------------------------------
    void Print( const char* str );

    //---------------------------------------------------------------------------
    //! @brief        書式指定して文字列を描画します。
    //!
    //! @param[in]    format    書式文字列です。
    //---------------------------------------------------------------------------
    void Printf( const char* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        書式指定して文字列を描画します。
    //!
    //! @param[in]    format    書式文字列です。
    //! @param[in]    args      引数リストです。
    //---------------------------------------------------------------------------
    void VPrintf( const char* format, std::va_list args );

    //---------------------------------------------------------------------------
    //! @brief        影付きで文字列を描画します。
    //!
    //! @param[in]    str       文字列です。
    //---------------------------------------------------------------------------
    void PrintWithShadow( const char* str );

    //---------------------------------------------------------------------------
    //! @brief        書式指定して影付きで文字列を描画します。
    //!
    //! @param[in]    format    書式文字列です。
    //---------------------------------------------------------------------------
    void PrintfWithShadow( const char* format, ... );

    //---------------------------------------------------------------------------
    //! @brief        書式指定して影付きで文字列を描画します。
    //!
    //! @param[in]    format    書式文字列です。
    //! @param[in]    args      引数リストです。
    //---------------------------------------------------------------------------
    void VPrintfWithShadow( const char* format, std::va_list args );

private:
    //! @brief 描画用バッファの構造体です。
    struct StringBuffer
    {
        font::DispStringBuffer* dispStringBuffer;   //!< 文字列表示用バッファです。
        void*                   graphicsBuffer;     //!< 描画用頂点バッファです。
    };

    typedef nw::ut::MoveArray<StringBuffer> StringBufferArray;

    //! @brief 描画に必要な TextWriter とバッファの構造体です。
    struct Writer
    {
        StringBufferArray stringBufferArray;        //!< 文字列バッファを格納する MoveArray です。
        StringBuffer*     currentStringBuffer;      //!< 現在の文字列バッファへのポインタです。
        font::TextWriter  textWriter;               //!< TextWriter オブジェクトです。
        math::MTX34       modelMtx;                 //!< 描画用モデル行列です。

        //! @brief コンストラクタです。
        Writer()
            : currentStringBuffer(NULL),
            modelMtx(nw::math::MTX34::Identity()) {}
    };

    //! @brief 保持しているバッファをすべて描画します。
    void Flush(Writer& writer);

public:
    //---------------------------------------------------------------------------
    //! @brief        保持しているバッファをすべて描画します。
    //---------------------------------------------------------------------------
    void Flush()
    {
        Flush( m_ShadowWriter );
        Flush( m_Writer );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列描画用の行列を設定します。
    //!
    //! @param[in]    projectionMtx プロジェクション行列です。
    //! @param[in]    viewMtx       ビュー行列です。
    //---------------------------------------------------------------------------
    void SetMatrix( const nw::math::MTX44& projectionMtx, const nw::math::MTX34& viewMtx )
    {
        m_ProjectionMtx = projectionMtx;
        m_ViewMtx = viewMtx;
    }

    //---------------------------------------------------------------------------
    //! @brief        描画位置の基点を設定します。
    //!
    //! @param[in]    x         基点の X 座標です。
    //! @param[in]    y         基点の Y 座標です。
    //---------------------------------------------------------------------------
    void SetOrigin( f32 x, f32 y )
    {
        m_OriginX = x;
        m_OriginY = y;
        m_TagProcessor.SetOrigin( x, y );
        m_Writer.textWriter.SetCursor( x, y );
        m_ShadowWriter.textWriter.SetCursor( x, y );
    }

    //---------------------------------------------------------------------------
    //! @brief        描画位置の基点の X 座標を取得します。
    //!
    //! @return       描画位置の基点の X 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetOriginX()
    {
        return m_OriginX;
    }

    //---------------------------------------------------------------------------
    //! @brief        描画位置の基点の Y 座標を取得します。
    //!
    //! @return       描画位置の基点の Y 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetOriginY()
    {
        return m_OriginY;
    }

    //---------------------------------------------------------------------------
    //! @brief        カーソル位置を設定します。
    //!
    //! @param[in]    x         カーソル位置の X 座標です。
    //! @param[in]    y         カーソル位置の Y 座標です。
    //---------------------------------------------------------------------------
    void SetCursor( f32 x, f32 y )
    {
        m_Writer.textWriter.SetCursor( x, y );
        m_ShadowWriter.textWriter.SetCursor( x, y );
    }

    //---------------------------------------------------------------------------
    //! @brief        カーソル位置の X 座標を設定します。
    //!
    //! @param[in]    x         カーソル位置の X 座標です。
    //---------------------------------------------------------------------------
    void SetCursorX( f32 x )
    {
        m_Writer.textWriter.SetCursorX( x );
        m_ShadowWriter.textWriter.SetCursorX( x );
    }

    //---------------------------------------------------------------------------
    //! @brief        カーソル位置の Y 座標を設定します。
    //!
    //! @param[in]    y         カーソル位置の Y 座標です。
    //---------------------------------------------------------------------------
    void SetCursorY( f32 y )
    {
        m_Writer.textWriter.SetCursorY( y );
        m_ShadowWriter.textWriter.SetCursorY( y );
    }

    //---------------------------------------------------------------------------
    //! @brief     描画位置の基点にカーソル位置をリセットします。
    //---------------------------------------------------------------------------
    void ResetCursorToOrigin()
    {
        SetCursor( m_OriginX, m_OriginY );
    }

    //---------------------------------------------------------------------------
    //! @brief        カーソル位置の X 座標を取得します。
    //!
    //! @return       カーソル位置の X 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetCursorX()
    {
        return m_Writer.textWriter.GetCursorX();
    }

    //---------------------------------------------------------------------------
    //! @brief        カーソル位置の Y 座標を取得します。
    //!
    //! @return       カーソル位置の Y 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetCursorY()
    {
        return m_Writer.textWriter.GetCursorY();
    }

    //---------------------------------------------------------------------------
    //! @brief        文字色を設定します。
    //!
    //! @param[in]    color     文字色です。
    //---------------------------------------------------------------------------
    void SetTextColor( ut::Color4u8 color )
    {
        m_Writer.textWriter.SetTextColor( color );
    }

    //---------------------------------------------------------------------------
    //! @brief        影の色を設定します。
    //!
    //! @param[in]    color     影の色です。
    //---------------------------------------------------------------------------
    void SetShadowColor( ut::Color4u8 color )
    {
        m_ShadowWriter.textWriter.SetTextColor( m_ShadowColor = color );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字の拡大率を設定します。
    //!
    //! @param[in]    scale     文字の拡大率です。
    //---------------------------------------------------------------------------
    void SetScale( f32 scale )
    {
        m_Writer.textWriter.SetScale( scale );
        m_ShadowWriter.textWriter.SetScale( scale );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字のサイズを設定します。
    //!
    //! @param[in]    height    拡大後のセルの高さです。
    //---------------------------------------------------------------------------
    void SetFontSize( f32 height )
    {
        m_Writer.textWriter.SetFontSize( height );
        m_ShadowWriter.textWriter.SetFontSize( height );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字のサイズを設定します。
    //!
    //! @param[in]    width     拡大後のセルの幅です。
    //! @param[in]    height    拡大後のセルの高さです。
    //---------------------------------------------------------------------------
    void SetFontSize( f32 width, f32 height )
    {
        m_Writer.textWriter.SetFontSize( width, height );
        m_ShadowWriter.textWriter.SetFontSize( width, height );
    }

    //---------------------------------------------------------------------------
    //! @brief        影の現れる位置を設定します。
    //!
    //! @param[in]    deltaX    X 方向への大きさです。
    //! @param[in]    deltaY    Y 方向への大きさです。
    //---------------------------------------------------------------------------
    void SetShadowPos( f32 deltaX, f32 deltaY )
    {
        m_ShadowWriter.modelMtx.SetTranslate( nw::math::VEC3( deltaX, deltaY, 0.f ) );
    }

    //---------------------------------------------------------------------------
    //! @brief        TextWriter を取得します。
    //!
    //! @return       TextWriter を返します。
    //---------------------------------------------------------------------------
    const font::TextWriter& GetTextWriter() const { return m_Writer.textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        TextWriter を取得します。
    //!
    //! @return       TextWriter を返します。
    //---------------------------------------------------------------------------
    font::TextWriter& GetTextWriter() { return m_Writer.textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        影用の TextWriter を取得します。
    //!
    //! @return       影用のTextWriter を返します。
    //---------------------------------------------------------------------------
    const font::TextWriter& GetShadowTextWriter() const { return m_ShadowWriter.textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        影用のTextWriter を取得します。
    //!
    //! @return       影用のTextWriter を返します。
    //---------------------------------------------------------------------------
    font::TextWriter& GetShadowTextWriter() { return m_ShadowWriter.textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        書式文字列展開用のバッファを割り当てます。
    //!
    //! @param[in]      buffer  割り当てるバッファへのポインタです。
    //! @param[in]      size    バッファサイズです。
    //!
    //! @return         以前に割り当てられたバッファへのポインタを返します。
    //---------------------------------------------------------------------------
    static void* SetFormatBuffer( char* buffer, std::size_t size )
    {
        NW_ASSERT_VALID_POINTER( buffer );
        NW_FONT_MIN_ASSERT( size, 1 );
        void* oldBuffer = s_FormatBuffer;

        s_FormatBuffer = buffer;
        s_FormatBufferSize = size;

        return oldBuffer;
    }


private:
    //! @brief DevTextWriter 用の TagProcessor です。
    class EdTagProcessor : public font::TagProcessorBase<char>
    {
    public:
        //! @name コンストラクタ / デストラクタ
        //@{

        //! コンストラクタです。
        EdTagProcessor( f32 originX, f32 originY )
          : font::TagProcessorBase<char>(),
            m_OriginX( originX ),
            m_OriginY( originY )
            {}

        //! デストラクタです。
        virtual ~EdTagProcessor() {}

        //@}

        //! @name タグ処理
        //@{

        //! @brief      タグに従って処理を行います。
        //!
        //! @param[in]      code      タグ処理の起点となった制御文字コード。(0x0000 ～ 0x001F)
        //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
        //!
        //! @return     呼び出し元 (TextWriter) が行うべき処理を返します。
        //!
        virtual Operation   Process( u16 code, font::PrintContext<char>* pContext );

        //! @brief      タグの影響範囲を計算します。
        //!
        //! @param[out]     pRect     タグの影響範囲を格納する矩形構造体へのポインタ。
        //! @param[in]      code      タグ処理の起点となった制御文字コード。(0x0000 ～ 0x001F)
        //! @param[in,out]  pContext  文字列の描画状態情報へのポインタ。
        //!
        //! @return     呼び出し元 (TextWriter) が行うべき処理を返します。
        //!             同じタグに対して Process() 関数が返す値と同じ値を返すべきです。
        //!
        virtual Operation   CalcRect( ut::Rect* pRect, u16 code, font::PrintContext<char>* pContext);

        //@}

        //! @brief        描画位置の基点を設定します。
        //!
        //! @param[in]    originX   基点の X 座標です。
        //! @param[in]    originY   基点の Y 座標です。
        void SetOrigin( f32 originX, f32 originY )
        {
            m_OriginX = originX;
            m_OriginY = originY;
        }

    private:
        void ProcessLinefeed( font::PrintContext<char>* pContext );

        f32 m_OriginX;  //!< 描画基準位置の X 座標です。
        f32 m_OriginY;  //!< 描画基準位置の Y 座標です。
    };

    //! @brief 描画用バッファを生成します。
    void CreateStringBuffer( Writer& writer );

    //! @brief 描画用バッファを解放します。
    void DestroyStringBuffer();

    //! @brief 描画の準備をします。
    void PreparePrint( Writer& writer );

    //! @brief 文字列を描画します。
    void PrintImpl( const char* str, Writer& writer );


    static const std::size_t DEFAULT_FORMAT_BUFFER_SIZE = 256;                  //!< デフォルトの書式展開用のバッファサイズです。
    static char              s_DefaultFormatBuffer[DEFAULT_FORMAT_BUFFER_SIZE]; //!< デフォルトの書式展開用のバッファです。
    static std::size_t       s_FormatBufferSize;                                //!< 書式展開用のバッファサイズです。
    static char*             s_FormatBuffer;                                    //!< 書式展開用のバッファです。

    Writer                   m_Writer;              //!< 文字描画に必要な TextWriter とバッファです。
    Writer                   m_ShadowWriter;        //!< 影描画に必要な TextWriter とバッファです。

    u32                      m_CharMax;             //!< 最大描画文字数です。
    f32                      m_OriginX;             //!< 描画基準位置の X 座標です。
    f32                      m_OriginY;             //!< 描画基準位置の Y 座標です。

    EdTagProcessor           m_TagProcessor;        //!< タグプロセッサです。
    font::ResFont            m_ResFont;             //!< フォントリソースです。
    void*                    m_DrawerBuffer;        //!< RectDrawer 用バッファです。
    font::RectDrawer         m_Drawer;              //!< RectDrawer オブジェクトです。

    math::MTX44              m_ProjectionMtx;       //!< 描画用プロジェクション行列です。
    math::MTX34              m_ViewMtx;             //!< 描画用ビュー行列です。
    nw::ut::Color4u8         m_ShadowColor;         //!< 影の色です。
    static const u32         DEFAULT_SHADOW_COLOR;  //!< デフォルトの影の色です。
    static const f32         DEFAULT_SHADOW_POS_X;  //!< デフォルトの影の位置（X方向）です。
    static const f32         DEFAULT_SHADOW_POS_Y;  //!< デフォルトの影の位置（Y方向）です。

    ut::IAllocator*          m_Allocator;           //!< バッファ生成に用いたアロケータです。
};

} // namespace dev
} // namespace nw

#endif //  NW_DEV_DEV_TEXT_WRITER_H_
