﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_KEYBOARD_MOUSE_DEVICE_CAFE_H_
#define NW_DEV_KEYBOARD_MOUSE_DEVICE_CAFE_H_

#include <nw/dev/dev_KeyboardMouseDevice.h>
#include <nw/mcs/mcs_InputDevice.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief   PC キーボード・マウスデバイス情報を取得するクラスです。
//!
//!
//! @details :category     入力デバイス
//!          mcs の InputDevice を用いることで PC からキーボード・マウス入力を取得します。
//---------------------------------------------------------------------------
class KeyboardMouseDeviceCafe : public KeyboardMouseDevice
{
public:
    //! @brief マウス状態を表す構造体です。
    struct MouseStatus
    {
        u8      button;
        u8      reserved;
        s16     moveX;
        s16     moveY;
        s16     wheel;

        //! @brief コンストラクタです。
        MouseStatus()
          : button( 0 ),
            reserved( 0 ),
            moveX( 0 ),
            moveY( 0 ),
            wheel( 0 )
        {}
    };

    //! @brief イベントを表します。
    enum EventType
    {
        IDLE        = 0,
        MOVE        = 1 << 0,
        DOWN        = 1 << 1,
        UP          = 1 << 2,
        DOUBLECLICK = 1 << 3,
        WHEEL       = 1 << 4
    };


#if defined(NW_DEV_PAD_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //!
    //! @return       シングルトンのインスタンスを返します。
    //---------------------------------------------------------------------------
    static KeyboardMouseDeviceCafe* GetInstance()
    {
        static KeyboardMouseDeviceCafe s_Instance;
        return &s_Instance;
    }


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //!               あらかじめ mcs の InputDevice が初期化されている必要があります。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize() {}

    //---------------------------------------------------------------------------
    //! @brief        デバイスを更新します。
    //---------------------------------------------------------------------------
    virtual void Update();


    //---------------------------------------------------------------------------
    //! @brief        デバイスが利用可能かどうかを取得します。
    //!
    //! @return       利用可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsAvailable() { return m_IsAvailable; }

    //---------------------------------------------------------------------------
    //! @brief        マウス入力を取得します。
    //!
    //! @return       マウス状態を返します。
    //---------------------------------------------------------------------------
    const MouseStatus& GetMouseStatus() const { return m_InputEventHandler.GetMouseStatus(); }

    //---------------------------------------------------------------------------
    //! @brief        発生したマウスイベントの種類を取得します。
    //!
    //! @return       マウスイベントの種類を返します。
    //---------------------------------------------------------------------------
    const u32 GetMouseEventType() const { return m_InputEventHandler.GetMouseEventType(); }

    //---------------------------------------------------------------------------
    //! @brief        キーリピートの設定を行います。
    //!
    //! @param[in]    delayFrame ボタンが押されてから何フレーム後に最初のリピートが発生するかを指定します。
    //! @param[in]    pulseFrame 最初にリピートになった後、何フレーム毎にリピートが発生するかを指定します。
    //---------------------------------------------------------------------------
    void SetKeyRepeat(u8 delayFrame, u8 pulseFrame);


protected:
    //! @brief mcs InputDevice イベントハンドラクラスです。
    class KeyboardMouseInputEventHandler : public nw::mcs::InputEventHandler
    {
    public:
        //! @brief コンストラクタです。
        KeyboardMouseInputEventHandler()
          : nw::mcs::InputEventHandler(),
            m_MouseEventType( IDLE )
        {
            for ( u32 i = 0; i < V_KEY_MAX; ++i )
            {
                m_KeyStatus[ i ] = 0;
            }
        }

        //! @brief マウス状態を取得します。
        const MouseStatus& GetMouseStatus() const { return m_MouseStatus; }
        //! @brief キー状態を取得します。
        const u8 GetKeyStatus( int key ) const { return m_KeyStatus[ key ]; }
        //! @brief 発生したマウスイベントの種類を取得します。
        const u32 GetMouseEventType() const { return m_MouseEventType; }
        //! @brief イベント発生フラグをリセットします。
        void ResetEvent()
        {
            m_MouseEventType = IDLE;

            m_MouseStatus.button = 0;
            m_MouseStatus.reserved = 0;
            m_MouseStatus.moveX = 0;
            m_MouseStatus.moveY = 0;
            m_MouseStatus.wheel = 0;
        }

    private:
        //! @brief キーボードのキーが押されたときに呼び出されます。
        virtual void KeyDownEvent(const nw::mcs::KeyEventArg& arg)
        {
            m_KeyStatus[ arg.GetKeyCode() ] = V_KEY_ON;
        }

        //! @brief キーボードのキーが離されたときに呼び出されます。
        virtual void KeyUpEvent(const nw::mcs::KeyEventArg& arg)
        {
            m_KeyStatus[ arg.GetKeyCode() ] = V_KEY_OFF;
        }

        //! @brief 文字が入力されたときに呼び出されます。
        virtual void CharEvent(const nw::mcs::CharEventArg& arg)
        {
        }

        //! @brief マウスが移動したときに呼び出されます。
        virtual void MouseMoveEvent(const nw::mcs::MouseEventArg& arg)
        {
            m_MouseStatus.moveX += arg.GetMoveX();
            m_MouseStatus.moveY += arg.GetMoveY();
            m_MouseEventType |= MOVE;
        }

        //! @brief マウスボタンが押されたときに呼び出されます。
        virtual void MouseDownEvent(const nw::mcs::MouseEventArg& arg)
        {
            m_MouseStatus.button = arg.GetButton();
            m_MouseEventType &= ~UP;
            m_MouseEventType |= DOWN;

        }

        //! @brief マウスボタンが離されたときに呼び出されます。
        virtual void MouseUpEvent(const nw::mcs::MouseEventArg& arg)
        {
            m_MouseStatus.button = arg.GetButton();
            m_MouseEventType &= ~DOWN;
            m_MouseEventType |= UP;
        }

        //! @brief マウスボタンがダブルクリックされたときに呼び出されます。
        virtual void MouseDoubleClickEvent(const nw::mcs::MouseEventArg& arg)
        {
            m_MouseStatus.button = arg.GetButton();
            m_MouseEventType |= DOUBLECLICK;
        }

        //! @brief マウスホイールが回転したときに呼び出されます。
        virtual void MouseWheelEvent(const nw::mcs::MouseEventArg& arg)
        {
            m_MouseStatus.wheel += arg.GetWheel();
            m_MouseEventType |= WHEEL;
        }

        MouseStatus       m_MouseStatus;
        u8                m_KeyStatus[ V_KEY_MAX ];
        u32               m_MouseEventType;
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    KeyboardMouseDeviceCafe();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~KeyboardMouseDeviceCafe() {}
#else
    static KeyboardMouseDeviceCafe* GetInstance() { return NULL; }

    void Initialize() {}

    void Finalize() {}

    virtual void Update() {}

    bool IsAvailable() { return false; }

    const MouseStatus& GetMouseStatus() const { return m_InputEventHandler.GetMouseStatus(); }

    const u32 GetMouseEventType() const { return 0; }

    void SetKeyRepeat(u8, u8) {}

protected:
    class KeyboardMouseInputEventHandler : public nw::mcs::InputEventHandler
    {
    public:
        KeyboardMouseInputEventHandler() {}

        const MouseStatus& GetMouseStatus() const { return m_Dummy; }

        const u8 GetKeyStatus( int key ) const { return 0; }

        const u32 GetMouseEventType() const { return 0; }

        void ResetEvent() {}

    private:
        MouseStatus       m_Dummy;
    };

    KeyboardMouseDeviceCafe() {}

    virtual ~KeyboardMouseDeviceCafe() {}
#endif // NW_DEV_PAD_ENABLE

    KeyboardMouseInputEventHandler m_InputEventHandler; //!< mcs InputDevice イベントハンドラです。
    bool m_IsAvailable;                                 //!< デバイスが利用可能かを表します。

    u32 m_PadHoldCounts[V_KEY_MAX];                     //!< ボタンが押され続けているフレーム数です。
    u8 m_PadRepeatDelays;                               //!< ボタンを押し続けたときに一番最初にリピートになるフレーム数です。
    u8 m_PadRepeatPulses;                               //!< 最初にリピートになった後、何フレーム毎にリピートにするかです。

    static const u8 PAD_REPEAT_DELAY_DEFAULT = 30;      //!< ボタンを押し続けたときに一番最初にリピートになるフレーム数のデフォルト値です。
    static const u8 PAD_REPEAT_PULSE_DEFAULT = 1;       //!< 最初にリピートになった後、何フレーム毎にリピートにするかのデフォルト値です。

    static KeyboardMouseDeviceCafe* s_pInstance;        //!< シングルトンオブジェクトへのポインタです。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_KEYBOARD_MOUSE_DEVICE_CAFE_H_
