﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_CTRL_DRAG_BUTTON_H_
#define NW_CTRL_DRAG_BUTTON_H_

#include <nw/ctrl/ctrl_AnimButton.h>

namespace nw
{

namespace ctrl
{

//---------------------------------------------------------------------------
//! @brief ボタンを掴んで移動させることができるボタンです。
//!
//! @details
//! SelectButtonと挙動が似ていますが、STATE_DOWN_IDLEの状態でCancelアクションが発行されると、
//! STATE_ONに戻ります。
//!
//! ON/OFFとDOWNの他にもう一つ、DOWN_IDLEからON_IDLEに戻るアニメ(Releaseアニメ)が
//! 必要です。Releaseアニメは機能アニメの3つ目に指定してください。
//!
//! Releaseアニメは、内部的にはCancelアニメとして扱っています。
//!
//! なお、このボタンはタッチスクリーンでは正しく動作しません。タッチスクリーンで使う場合は、
//! TouchDragButtonを使用してください。
//!
//---------------------------------------------------------------------------
class DragButton : public AnimButton
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(AnimButton);

    //! @brief コンストラクタです。
    //!
    DragButton();

    //! @brief ボタンの構築を行います。
    //!
    //! AnimButtonの標準の動作に加えて、パラメータアニメの3つ目に指定された
    //! キャンセルアニメを作成します。
    //!
    //! @param[in] controlSrc   コントロールの情報
    //! @param[in,out] layout   所属するレイアウト
    //!
    void Build(const lyt::ControlSrc& controlSrc, lyt::Layout* layout);

    //! @brief ドラッグ時の位置を初期化します。
    //!
    //! @param[in] pos  現在ドラッグしている位置です。
    //!
    virtual void InitDragPosition(const math::VEC2& pos);

    //! @brief ドラッグ時の位置を更新します。
    //!
    //! @param[in] pos  現在ドラッグしている位置です。ポインタが画面から外れている場合はNULLとなります。
    //!
    virtual void UpdateDragPosition(const math::VEC2* pos);

    //! @brief ドラッグ時、水平方向に移動するか否かを取得します。
    //!
    //! @return 水平方向に移動するか否か
    //!
    bool IsMoveHorizontal()
    {
        return m_IsMoveHorizontal;
    }

    //! @brief ドラッグ時、水平方向に移動するか否かを指定します。
    //!
    //! @details
    //! デフォルトでは移動する設定になっています。
    //!
    //! @param[in] b  水平方向に移動するか
    //!
    void SetMoveHorizontal(bool b)
    {
        m_IsMoveHorizontal = b;
    }

    //! @brief ドラッグ時、垂直方向に移動するか否かを取得します。
    //!
    //! @return 垂直方向に移動するか否か
    //!
    bool IsMoveVertical()
    {
        return m_IsMoveVertical;
    }

    //! @brief ドラッグ時、垂直方向に移動するか否かを指定します。
    //!
    //! @details
    //! デフォルトでは移動する設定になっています。
    //!
    //! @param[in] b  垂直方向に移動するか
    //!
    void SetMoveVertical(bool b)
    {
        m_IsMoveVertical = b;
    }

protected:
    //! @brief ボタンの挙動をカスタマイズするためにオーバーライドします。
    //!
    virtual bool ProcessOff();
    virtual bool ProcessCancel();
    virtual void FinishCancel();

    lyt::Pane* mRootPane;
    math::VEC2 mDragStartPos;
    math::VEC2 mDragStartTransrate;
    bool m_IsMoveHorizontal;
    bool m_IsMoveVertical;

};

} // namespace nw::ctrl
} // namespace nw

#endif // NW_CTRL_DRAG_BUTTON_H_
