﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief オフラインページを開くのに利用する API 宣言(内部開発者向け)
 */

#pragma once

#include <string>

#include <nn/nn_ApplicationId.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/web/web_OfflineShowArg.h>
#include <nn/web/detail/web_OfflineTypesDetail.h>

namespace nn { namespace web { namespace detail {

//--------------------------------------------------------------------------
/**
 * @brief アプリケーションの「遊び方」を開くための情報を書き込む為のクラスです。
 */
class ShowOfflineHtmlPageArgWriter
    : public ShowOfflineHtmlPageArg
{
public:
    /**
     * @internal
     *
     * @brief   オフラインページを開く為の変数を一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;
};

//--------------------------------------------------------------------------
/**
 * @brief OfflinePageを開く際のパラメータを設定する為のクラスです。
 */
class ShowOfflinePageArgDetail
{
public:
    /**
     * @brief アプリケーション ID を設定します。
     *
     * @param[in,out]   pDst    設定対象となるデータを指定してください。
     * @param[in]       id      ApplicationId を指定ください
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    static void SetApplicationId(uint8_t* pDst, const nn::ApplicationId& id) NN_NOEXCEPT;

    /**
     * @brief ドキュメントパスを設定します。
     *
     * @param[in,out]   pDst            設定対象となるデータを指定してください。
     * @param[in]       fixedSize       設定するパラメータを格納する際の固定サイズを指定してください。
     * @param[in]       pDocumentPath   ドキュメントパスを指定してください。
     *
     * @details ドキュメントパスは NULL 終端されている必要があります。
     *           DocumentPathMaxLength -1 文字以下にする必要があります。
     *           二度呼び出した場合、値は上書きされます。
     */
    static void SetDocumentPath(uint8_t* pDst, size_t fixedSize, const char* pDocumentPath) NN_NOEXCEPT;

    /**
     * @brief ドキュメントの種類を設定します。
     *
     * @param[in,out]   pDst            設定対象となるデータを指定してください。
     * @param[in]       documentKind    ドキュメントの種類を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    static void SetDocumentKind(uint8_t* pDst, const OfflineDocumentKind& documentKind) NN_NOEXCEPT;

    /**
     * @brief システム ID を設定します。
     *
     * @param[in,out]   pDst    設定対象となるデータを指定してください。
     * @param[in]       id      SystemDataId を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    static void SetSystemDataId(uint8_t* pDst, const nn::ncm::SystemDataId& id) NN_NOEXCEPT;

    /**
     * @brief 左スティックモードをポインターに切り替える事を許可するかどうかを設定します。
     *
     * @param[in,out]   pDst            設定対象となるデータを指定してください。
     * @param[in]       pointerEnabled  左スティックモードをポインターに切り替える事を許可する場合 true を指定してください。
     *
     * @details デフォルト値は true です。
     *          二度呼び出した場合、値は上書きされます。
     */
    static void SetPointerEnabled(uint8_t* pDst, const bool pointerEnabled) NN_NOEXCEPT;

    /**
     * @brief 起動時の左スティックモードを設定します。
     *
     * @param[in,out]   pDst    設定対象となるデータを指定してください。
     * @param[in]       mode    起動時の左スティックモードを指定してください。
     *
     * @details SetPointerEnabled を false にした場合はここでの設定に関わらず空間ナビが使用されます。
     *          二度呼び出した場合、値は上書きされます。
     */
    static void SetLeftStickMode(uint8_t* pDst, const OfflineLeftStickMode& mode) NN_NOEXCEPT;

};

//--------------------------------------------------------------------------
/**
 * @brief ドキュメントを開いた結果の情報を設定する為のクラスです。
 */
class OfflineHtmlPageReturnValueDetail
    : protected OfflineHtmlPageReturnValue
{
public:
    /**
     * @brief 終了理由を設定します。
     *
     * @param[in]   loginExitReason   終了理由を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    inline void SetOfflineExitReason(OfflineExitReason loginExitReason) NN_NOEXCEPT;

    /**
     * @brief 終了時開いていたページの URL を設定します。
     *
     * @param[in]   lastUrl 終了時開いていたページの URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          LastUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    inline void SetLastUrl(const char* lastUrl) NN_NOEXCEPT;
};

//------------------------------------------------------------------------
/**
* @brief (廃止予定) ドキュメントを開いた結果の情報を設定する為のクラスです。
* @deprecated 型名「OfflinePageReturnValueDetail」は廃止予定です。OfflineHtmlPageReturnValueDetail を利用してください。
*/
NN_DEPRECATED typedef OfflineHtmlPageReturnValueDetail OfflinePageReturnValueDetail;

//------------------------------------------------------------------------
void OfflineHtmlPageReturnValueDetail::SetOfflineExitReason(OfflineExitReason exitReason) NN_NOEXCEPT
{
    m_ExitReason = static_cast<uint64_t>(exitReason);
}

//------------------------------------------------------------------------
void OfflineHtmlPageReturnValueDetail::SetLastUrl(const char* lastUrl) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(lastUrl);
    size_t size = std::strlen(lastUrl) + 1;
    NN_SDK_ASSERT(size <= LastUrlMaxLength);

    std::memcpy(m_LastUrl, lastUrl, size);

    m_LastUrlSize = static_cast<uint64_t>(size);
}

}}} // namespace nn::web::detail
