﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Result.h>

namespace nn { namespace rid {

    /**
     * @brief    本体更新の進行状況です。
     */
    struct SystemUpdateProgress
    {
        enum class State
        {
            DoNothing,   //!< 本体更新処理が始まっていない
            Checking,    //!< 本体更新が必要か確認している
            Downloading, //!< 本体更新をダウンロードしている
            Applying,    //!< 本体更新を適用している
            Completed,   //!< 本体更新が完了した
            Failed,      //!< 本体更新に失敗した
            NeedNoUpdate,//!< 本体更新の必要がなかった
            Cancelled,   //!< 本体更新がキャンセルされた
        }state;

        int64_t loaded; //!< ロードされた容量です。state=Downloading のときのみ参照可能です。
        int64_t total;  //!< トータルの容量です。state=Downloading のときのみ参照可能です。
    };

    /**
     * @brief    本体更新を実行する処理を行うクラスです。
     */
    class SystemUpdater
    {
    public:
        SystemUpdater() NN_NOEXCEPT;
        ~SystemUpdater() NN_NOEXCEPT {};

        /**
         * @brief    本体更新を実行します。
         *
         * @details  本体更新に必要な処理をこの関数の中で行います。
         *           本体更新の必要の有無を確認して必要な場合本体更新のダウンロードを行い実行します。
         *           本体更新が実行された場合、SystemUpdateProgress::State が Completed となります。本体更新を適用するために本体を再起動してください。
         *           本体更新の必要がなかった場合、SystemUpdateProgress::State が NeedNoUpdate となります。
         *
         * @return   処理の結果が返ります。
         * @retval   ResultNetworkNotConnected   ネットワークに接続されていないので更新に失敗しました。
         * @retval   ResultSystemUpdateCancelled 本体更新がキャンセルされました。
         */
        Result Execute() NN_NOEXCEPT;

        /**
        * @brief        本体更新をキャンセルします。
        */
        void Cancel() NN_NOEXCEPT;

        /**
         * @brief    本体更新の進行状況を取得します。
         *
         * @return   本体更新の進行状況が返ります。
         */
        SystemUpdateProgress GetProgress() NN_NOEXCEPT;

    private:
        Result ExecuteImpl() NN_NOEXCEPT;
        void UpdateProgress(SystemUpdateProgress::State state, int64_t loaded = 0, int64_t total = 0) NN_NOEXCEPT;

        SystemUpdateProgress m_Progress;
        os::Mutex m_Mutex;
        bool m_IsCancelled;
    };
}}
