﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Result.h>

namespace nn { namespace rid {

    /**
     * @brief    試遊台メニュー更新の進行状況です。
     */
    enum class MenuUpdateState : Bit8
    {
        DoNothing,        //!< 試遊台メニュー更新処理が始まっていない、または開始処理を行っている
        Downloading,      //!< 試遊台メニュー更新をダウンロードしている
        Completed,        //!< 試遊台メニュー更新が完了した
        Failed,           //!< 試遊台メニュー更新に失敗した
        NeedNoUpdate,     //!< 試遊台メニュー更新の必要がなかった
        Cancelled,        //!< 試遊台メニュー更新がキャンセルされた
    };

    /**
     * @brief    試遊台メニュー更新を実行する処理を行うクラスです。
     */
    class MenuUpdater
    {
    public:
        MenuUpdater() NN_NOEXCEPT;
        ~MenuUpdater() NN_NOEXCEPT {};

        /**
         * @brief    試遊台メニュー更新を実行します。
         *
         * @details  試遊台メニュー更新に必要な処理をこの関数の中で行います。
         *           試遊台メニュー更新が実行された場合、本体が再起動する必要があります。
         *           試遊台メニュー更新の必要がなかった場合、更新は行われずに成功が返ります。
         *
         * @return   処理の結果が返ります。
         * @retval   ResultNetworkNotConnected   ネットワークに接続されていないので更新に失敗しました。
         * @retval   ResultMenuUpdateFailed      試遊台の更新のダウンロードに失敗しました。
         * @retval   ResultMenuUpdateCancelled   試遊台の更新がキャンセルされました。
         */
        Result Execute() NN_NOEXCEPT;

        /**
         * @brief        試遊台メニュー更新をキャンセルします。
         */
        void Cancel() NN_NOEXCEPT;

        /**
         * @brief    試遊台メニュー更新の進行状況を取得します。
         *
         * @return   試遊台メニュー更新の進行状況が返ります。
         */
        MenuUpdateState GetState() NN_NOEXCEPT;

    private:
        Result ExecuteImpl() NN_NOEXCEPT;
        void UpdateState(MenuUpdateState state) NN_NOEXCEPT;

        MenuUpdateState m_State;
        os::Mutex m_Mutex;
        bool m_IsCancelled;
    };
}}
